# -*- Perl -*-
# Tests for CNAME records
# Note that not all possible cases are caught!
# Template records mean not all published records are natively present in the DB

use Test::More;
use Data::Dumper;

use lib 't';

use DNSTest;
my $dtest = DNSTest::new;

my $code,$msg;
my $rectype = 5;
my $newname;
my $newval;

## Domain tests
subtest 'Domain tests' => sub {

subtest 'CNAME add - new name' => sub {
  $newname = 'newname.example.com';
  $newval = 'fredshosting.example.net';
  ($code, $msg) = $dnsdb->addRec('n', 'n', 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'OK', "addRec() claimed succeess" );
  if ($code eq 'OK') {
    # crosscheck in the DB
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE domain_id = 1 AND host = '$newname'");
    ok( $rcount == 1, " ... [$rcount] yep, hostname only occurs once" );
  } else {
    print "not ok: $msg";
  }
};

subtest 'CNAME add - existing/colliding non-CNAME' => sub {
  $newname = 'mx1.example.com';
  ($code, $msg) = $dnsdb->addRec('n', 'n', 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'FAIL', "addRec() claimed failure" );
  if ($code eq 'FAIL') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE domain_id = 1 AND host = '$newname' AND type <> 5");
    ok( $rcount == 2, " ... [$rcount] record(s) with $newname already exist" );
    like( $msg, qr/One or more non-CNAME records/, " ... returned matching error" );
  }
}; 

subtest 'CNAME add - existing/colliding CNAME' => sub {
  $newname = 'www.example.com';
  ($code, $msg) = $dnsdb->addRec('n', 'n', 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'FAIL', "addRec() claimed failure adding duplicate CNAME" );
  if ($code eq 'FAIL') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE domain_id = 1 AND host = '$newname' AND type = 5");
    ok( $rcount == 1, " ... [$rcount] CNAME already exists" );
    like( $msg, qr/already a CNAME present/, " ... returned matching error" );
  }
};

subtest 'CNAME update - non-CNAME to CNAME, non-colliding' => sub {
  $newname = 'smtp.example.com';
  $newval = 'example.com';
  ($code, $msg) = $dnsdb->updateRec('n', 'n', 39, 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'OK', "updateRec() claimed success" );
  if ($code eq 'OK') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE domain_id = 1 AND host = '$newname'");
    ok( $rcount == 1, " ... [$rcount] yep, hostname only occurs once" );
  } else {
    print "not ok: $msg";
  }
};

subtest 'CNAME update - non-CNAME to CNAME, colliding' => sub {
  $newname = 'mx1.example.com';
  ($code, $msg) = $dnsdb->updateRec('n', 'n', 39, 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'FAIL', "updateRec() claimed failure" );
  if ($code eq 'FAIL') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE domain_id = 1 AND host = '$newname' AND type <> 5");
    ok( $rcount == 2, " ... [$rcount] record(s) with $newname already exist" );
    like( $msg, qr/One or more non-CNAME records/, " ... returned matching error" );
  }
};

subtest 'CNAME update - name to non-colliding name' => sub {
  $newname = 'imap.example.com';
  ($code, $msg) = $dnsdb->updateRec('n', 'n', 37, 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'OK', "updateRec() claimed success" );
  if ($code eq 'OK') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE domain_id = 1 AND host = '$newname'");
    ok( $rcount == 1, " ... [$rcount] yep, hostname only occurs once" );
  } else {
    print "not ok: $msg";
  }
};

subtest 'CNAME update - name to colliding name' => sub {
  $newname = 'mx1.example.com';
  ($code, $msg) = $dnsdb->updateRec('n', 'n', 41, 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'FAIL', "updateRec() claimed failure" );
  if ($code eq 'FAIL') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE domain_id = 1 AND host = '$newname' AND type <> 5");
    ok( $rcount == 2, " ... [$rcount] record(s) with $newname already exist" );
    like( $msg, qr/One or more non-CNAME records/, " ... returned matching error" );
  }
};

}; # domain tests


## Reverse zone
subtest 'Reverse zone tests' => sub {

subtest 'CNAME add - new reverse name' => sub {
  $newval = '192.168.2.12';
  $newname = '12.8-29.2.168.192.in-addr.arpa';
  ($code, $msg) = $dnsdb->addRec('n', 'y', 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'OK', "addRec() claimed succeess adding reverse zone CNAME" );
  if ($code eq 'OK') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE rdns_id = 1 AND val = '$newval'");
    ok( $rcount == 1, " ... [$rcount] yep, IP only occurs once" );
  } else {
    print "not ok: $msg\n";
  }
};

subtest 'CNAME add - existing/colliding non-CNAME' => sub {
  $newval = '192.168.2.14';
  $newname = '14.8-29.2.168.192.in-addr.arpa';
  ($code, $msg) = $dnsdb->addRec('n', 'y', 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'FAIL', "addRec() claimed failure adding colliding reverse zone CNAME" );
  if ($code eq 'FAIL') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE rdns_id = 1 AND val = '$newval' AND type <> 5");
    ok( $rcount == 2, " ... [$rcount] record(s) with $newname already exist" );
    like( $msg, qr/One or more non-CNAME records/, " ... returned matching error" );
  }
};

subtest 'CNAME add - existing/colliding CNAME' => sub {
  $newval = '192.168.2.13';
  $newname = '13.8-29.2.168.192.in-addr.arpa';
  ($code, $msg) = $dnsdb->addRec('n', 'y', 1, \$newname, \$rectype, \$newval, 900);
  ok( $code eq 'FAIL', "addRec() claimed failure adding duplicate reverse zone CNAME" );
  if ($code eq 'FAIL') {
    ($rcount) = $dbh->selectrow_array("SELECT count(*) FROM records WHERE rdns_id = 1 AND val = '$newval' AND type = 5");
    ok( $rcount == 1, " ... [$rcount] CNAME already exists" );
    like( $msg, qr/already a CNAME present/, " ... returned matching error" );
  }
};

}; # reverse zone tests

done_testing();
