#!/usr/bin/perl -w -T
# XMLRPC interface to manipulate most DNS DB entities
##
# $Id: dns-rpc.cgi 401 2012-10-03 22:17:51Z kdeugau $
# Copyright 2012 Kris Deugau <kdeugau@deepnet.cx>
# 
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version. 
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##

use strict;
use warnings;

# don't remove!  required for GNU/FHS-ish install from tarball
use lib '.';	##uselib##

use DNSDB;	# note we're not importing subs;  this lets us (ab)use the same sub names here for convenience
use Data::Dumper;

#use Frontier::RPC2;
use Frontier::Responder;

## We need to handle a couple of things globally, rather than pasting the same bit into *every* sub.
## So, let's subclass Frontier::RPC2 + Frontier::Responder, so we can override the single sub in each
## that needs kicking
#### hmm.  put this in a separate file?
#package DNSDB::RPC;
#our @ISA = ("Frontier::RPC2", "Frontier::Responder");
#package main;

DNSDB::loadConfig(rpcflag => 1);

# need to create a DNSDB object too
my ($dbh,$msg) = DNSDB::connectDB($DNSDB::config{dbname}, $DNSDB::config{dbuser},
	$DNSDB::config{dbpass}, $DNSDB::config{dbhost});

DNSDB::initGlobals($dbh);

my $methods = {
	'dnsdb.addDomain'	=> \&addDomain,
	'dnsdb.delZone'		=> \&delZone,
	'dnsdb.addGroup'	=> \&addGroup,
	'dnsdb.delGroup'	=> \&delGroup,
	'dnsdb.addUser'		=> \&addUser,
	'dnsdb.updateUser'	=> \&updateUser,
	'dnsdb.delUser'		=> \&delUser,
	'dnsdb.getSOA'		=> \&getSOA,
	'dnsdb.getRecLine'	=> \&getRecLine,
	'dnsdb.getDomRecs'	=> \&getDomRecs,
	'dnsdb.getRecCount'	=> \&getRecCount,
	'dnsdb.addRec'		=> \&addRec,
	'dnsdb.delRec'		=> \&delRec,
	'dnsdb.domStatus'	=> \&domStatus,

	'dnsdb.getMethods'	=> \&get_method_list
};

my $res = Frontier::Responder->new(
	methods => $methods
	);

# "Can't do that" errors
if (!$dbh) {
  print "Content-type: text/xml\n\n".$res->{_decode}->encode_fault(5, $msg);
  exit;
}
##fixme:  fail on missing rpcuser/rpcsystem args

print $res->answer;

exit;

##
## Subs below here
##

# Utility subs
sub _aclcheck {
  my $subsys = shift;
  return 1 if grep /$ENV{REMOTE_ADDR}/, @{$DNSDB::config{rpcacl}{$subsys}};
  return 0;
}

#sub connectDB {
#sub finish {
#sub initGlobals {
#sub initPermissions {
#sub getPermissions {
#sub changePermissions {
#sub comparePermissions {
#sub changeGroup {
#sub _log {

sub addDomain {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  my ($code, $msg) = DNSDB::addDomain($dbh, $args{domain}, $args{group}, $args{state});
  die $msg if $code eq 'FAIL';
  return $msg;	# domain ID
}

sub delZone {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

# delZone takes zone id and forwrad/reverse flag
  my ($code,$msg);
  # Let's be nice;  delete based on domid OR domain name.  Saves an RPC call round-trip, maybe.
  if ($args{domain} =~ /^\d+$/) {
    ($code,$msg) = DNSDB::delDomain($dbh, $args{domain});
  } else {
    my $domid = DNSDB::domainID($dbh, $args{domain});
    die "Can't find domain\n" if !$domid;
    ($code,$msg) = DNSDB::delDomain($dbh, $domid);
  }
  die $msg if $code eq 'FAIL';
}

#sub domainName {
#sub domainID {

sub addGroup {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

# not sure how to usefully represent permissions from any further out from DNSDB.pm :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  my $perms = {domain_edit => 1, domain_create => 1, domain_delete => 1,
	record_edit => 1, record_create => 1, record_delete => 1
	};
## optional $inhert arg?
  my ($code,$msg) = DNSDB::addGroup($dbh, $args{groupname}, $args{parent_id}, $perms);
  die $msg if $code eq 'FAIL';
  return $msg;
}

sub delGroup {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  my ($code,$msg);
  # Let's be nice;  delete based on groupid OR group name.  Saves an RPC call round-trip, maybe.
  if ($args{group} =~ /^\d+$/) {
    ($code,$msg) = DNSDB::delGroup($dbh, $args{group});
  } else {
    my $grpid = DNSDB::groupID($dbh, $args{group});
    die "Can't find group" if !$grpid;
    ($code,$msg) = DNSDB::delGroup($dbh, $grpid);
  }
  die $msg if $code eq 'FAIL';
}

#sub getChildren {
#sub groupName {
#sub groupID {

sub addUser {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

# not sure how to usefully represent permissions from any further out from DNSDB.pm :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  my $perms = {domain_edit => 1, domain_create => 1, domain_delete => 1,
	record_edit => 1, record_create => 1, record_delete => 1
	};
  # bend and twist;  get those arguments in in the right order!
  $args{type} = 'u' if !$args{type};
  $args{permstring} = 'i' if !defined($args{permstring});
  my @userargs = ($args{username}, $args{group}, $args{pass}, $args{state}, $args{type}, $args{permstring});
  for my $argname ('fname','lname','phone') {
    last if !$args{$argname};
    push @userargs, $args{$argname};
  }
  my ($code,$msg) = DNSDB::addUser($dbh, @userargs);
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub checkUser {

sub updateUser {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  die "Missing UID\n" if !$args{uid};

# not sure how to usefully represent permissions from any further out from DNSDB.pm :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  my $perms = {domain_edit => 1, domain_create => 1, domain_delete => 1,
	record_edit => 1, record_create => 1, record_delete => 1
	};
  # bend and twist;  get those arguments in in the right order!
  my @userargs = ($args{uid}, $args{username}, $args{group}, $args{pass}, $args{state}, $args{type});
  for my $argname ('fname','lname','phone') {
    last if !$args{$argname};
    push @userargs, $args{$argname};
  }
##fixme:  also underlying in DNSDB::updateUser():  no way to just update this or that attribute;
#         have to pass them all in to be overwritten
  my ($code,$msg) = DNSDB::addUser($dbh, @userargs);
  die $msg if $code eq 'FAIL';
}

sub delUser {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  die "Missing UID\n" if !$args{uid};
  my ($code,$msg) = DNSDB::delUser($dbh, $args{uid});
  die $msg if $code eq 'FAIL';
}

#sub userFullName {
#sub userStatus {
#sub getUserData {

sub getSOA {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  my %ret = DNSDB::getSOA($dbh, $args{def}, $args{id});
  if (!$ret{recid}) {
    if ($args{def} eq 'y') {
      die "No default SOA record in group\n";
    } else {
      die "No SOA record in domain\n";
    }
  }
  return \%ret;
}

sub getRecLine {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  my $ret = DNSDB::getRecLine($dbh, $args{def}, $args{id});

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub getDomRecs {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

#bleh
  $args{nrecs} = 'all' if !$args{nrecs};
  $args{nstart} = 0 if !$args{nstart};
## for order, need to map input to column names
  $args{order} = 'host' if !$args{order};
  $args{direction} = 'ASC' if !$args{direction};

  my $ret = DNSDB::getDomRecs($dbh, (defrec => $args{defrec}, revrec => $args{revrec}, id => $args{id},
	offset => $args{offset}, sortby => $args{sortby}, sortorder => $args{sortorder},
	filter => $args{filter}) );

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub getRecCount {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  return DNSDB::getRecCount($dbh, $args{id});
}

sub addRec {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  # note dist, weight, port are not reequired on all types;  will be ignored if not needed.
  my ($code, $msg) = DNSDB::addRec($dbh, $args{def}, $args{domid}, $args{host}, $DNSDB::typemap{$args{type}},
	$args{val}, $args{ttl}, $args{dist}, $args{weight}, $args{port});

  die $msg if $code eq 'FAIL';
}

sub updateRec {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  # note dist, weight, port are not reequired on all types;  will be ignored if not needed.
  my ($code, $msg) = DNSDB::updateRec($dbh, $args{def}, $args{recid}, $args{host}, $DNSDB::typemap{$args{type}},
	$args{val}, $args{ttl}, $args{dist}, $args{weight}, $args{port});

  die $msg if $code eq 'FAIL';
}

sub delRec {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  # note dist, weight, port are not reequired on all types;  will be ignored if not needed.
  my ($code, $msg) = DNSDB::delRec($dbh, $args{def}, $args{recid});

  die $msg if $code eq 'FAIL';
}

#sub getParents {

sub domStatus {
  my %args = @_;

  # Make sure we've got all the local bits we need
  die "Missing remote username\n" if !$args{rpcuser};		# for logging
  die "Missing remote system name\n" if !$args{rpcsystem};	# for logging
  die "Access denied\n" if !_aclcheck($args{rpcsystem});

  my @arglist = ($dbh, $args{domid});
  push @arglist, $args{status} if defined($args{status});

  my $status = DNSDB::domStatus(@arglist);
}

#sub importAXFR {
#sub export {
#sub __export_tiny {

sub get_method_list {
  my @methods = keys %{$methods};
  return \@methods;
}
