#!/usr/bin/perl -w -T
# XMLRPC interface to manipulate most DNS DB entities
##
# $Id: dns-rpc.cgi 407 2012-10-05 16:45:23Z kdeugau $
# Copyright 2012 Kris Deugau <kdeugau@deepnet.cx>
# 
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version. 
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##

use strict;
use warnings;

# don't remove!  required for GNU/FHS-ish install from tarball
use lib '.';	##uselib##

use DNSDB;	# note we're not importing subs;  this lets us (ab)use the same sub names here for convenience
use Data::Dumper;

#use Frontier::RPC2;
use Frontier::Responder;

## We need to handle a couple of things globally, rather than pasting the same bit into *every* sub.
## So, let's subclass Frontier::RPC2 + Frontier::Responder, so we can override the single sub in each
## that needs kicking
#### hmm.  put this in a separate file?
#package DNSDB::RPC;
#our @ISA = ("Frontier::RPC2", "Frontier::Responder");
#package main;

DNSDB::loadConfig(rpcflag => 1);

# need to create a DNSDB object too
my ($dbh,$msg) = DNSDB::connectDB($DNSDB::config{dbname}, $DNSDB::config{dbuser},
	$DNSDB::config{dbpass}, $DNSDB::config{dbhost});

DNSDB::initGlobals($dbh);

my $methods = {
	'dnsdb.addDomain'	=> \&addDomain,
	'dnsdb.delZone'		=> \&delZone,
	'dnsdb.addRDNS'		=> \&addRDNS,
	'dnsdb.addGroup'	=> \&addGroup,
	'dnsdb.delGroup'	=> \&delGroup,
	'dnsdb.addUser'		=> \&addUser,
	'dnsdb.updateUser'	=> \&updateUser,
	'dnsdb.delUser'		=> \&delUser,
	'dnsdb.getSOA'		=> \&getSOA,
	'dnsdb.getRecLine'	=> \&getRecLine,
	'dnsdb.getDomRecs'	=> \&getDomRecs,
	'dnsdb.getRecCount'	=> \&getRecCount,
	'dnsdb.addRec'		=> \&addRec,
	'dnsdb.updateRec'	=> \&updateRec,
	'dnsdb.delRec'		=> \&delRec,
	'dnsdb.zoneStatus'	=> \&zoneStatus,

	'dnsdb.getMethods'	=> \&get_method_list
};

my $res = Frontier::Responder->new(
	methods => $methods
	);

# "Can't do that" errors
if (!$dbh) {
  print "Content-type: text/xml\n\n".$res->{_decode}->encode_fault(5, $msg);
  exit;
}
##fixme:  fail on missing rpcuser/rpcsystem args

print $res->answer;

exit;

##
## Subs below here
##

# Utility subs
sub _aclcheck {
  my $subsys = shift;
  return 1 if grep /$ENV{REMOTE_ADDR}/, @{$DNSDB::config{rpcacl}{$subsys}};
  return 0;
}

# Let's see if we can factor these out of the RPC method subs
sub _commoncheck {
  my $argref = shift;
  my $needslog = shift;

  die "Missing remote system name\n" if !$argref->{rpcsystem};
  die "Access denied\n" if !_aclcheck($argref->{rpcsystem});
  if ($needslog) {
    die "Missing remote username\n" if !$argref->{rpcuser};
    die "Couldn't set userdata for logging\n"
	unless DNSDB::initRPC($dbh, (username => $argref->{rpcuser}, rpcsys => $argref->{rpcsystem},
		fullname => ($argref->{fullname} ? $argref->{fullname} : $argref->{rpcuser}) ) );
  }
}

#sub connectDB {
#sub finish {
#sub initGlobals {
#sub initPermissions {
#sub getPermissions {
#sub changePermissions {
#sub comparePermissions {
#sub changeGroup {
#sub _log {

sub addDomain {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my ($code, $msg) = DNSDB::addDomain($dbh, $args{domain}, $args{group}, $args{state});
  die $msg if $code eq 'FAIL';
  return $msg;	# domain ID
}

sub delZone {
  my %args = @_;

  _commoncheck(\%args, 'y');
  die "Need forward/reverse zone flag\n" if !$args{revrec};

  my ($code,$msg);
  # Let's be nice;  delete based on zone id OR zone name.  Saves an RPC call round-trip, maybe.
  if ($args{zone} =~ /^\d+$/) {
    ($code,$msg) = DNSDB::delZone($dbh, $args{zone}, $args{revrec});
  } else {
    my $zoneid;
    $zoneid = DNSDB::domainID($dbh, $args{zone}) if $args{revrec} eq 'n';
    $zoneid = DNSDB::revID($dbh, $args{zone}) if $args{revrec} eq 'y';
    die "Can't find zone: $DNSDB::errstr\n" if !$zoneid;
    ($code,$msg) = DNSDB::delZone($dbh, $zoneid, $args{revrec});
  }
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub domainName {}
#sub revName {}
#sub domainID {}
#sub revID {}

sub addRDNS {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my ($code, $msg) = DNSDB::addRDNS($dbh, $args{revzone}, $args{revpatt}, $args{group}, $args{state});
  die $msg if $code eq 'FAIL';
  return $msg;	# domain ID
}

#sub getZoneCount {}
#sub getZoneList {}
#sub getZoneLocation {}

sub addGroup {
  my %args = @_;

  _commoncheck(\%args, 'y');
  die "Missing new group name\n" if !$args{groupname};
  die "Missing parent group ID\n" if !$args{parent_id};

# not sure how to usefully represent permissions via RPC.  :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  my $perms = {domain_edit => 1, domain_create => 1, domain_delete => 1,
	record_edit => 1, record_create => 1, record_delete => 1
	};
## optional $inhert arg?
  my ($code,$msg) = DNSDB::addGroup($dbh, $args{groupname}, $args{parent_id}, $perms);
  die $msg if $code eq 'FAIL';
  return $msg;
}

sub delGroup {
  my %args = @_;

  _commoncheck(\%args, 'y');
  die "Missing group ID or name to remove\n" if !$args{group};

  my ($code,$msg);
  # Let's be nice;  delete based on groupid OR group name.  Saves an RPC call round-trip, maybe.
  if ($args{group} =~ /^\d+$/) {
    ($code,$msg) = DNSDB::delGroup($dbh, $args{group});
  } else {
    my $grpid = DNSDB::groupID($dbh, $args{group});
    die "Can't find group\n" if !$grpid;
    ($code,$msg) = DNSDB::delGroup($dbh, $grpid);
  }
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub getChildren {}
#sub groupName {}
#sub getGroupCount {}
#sub getGroupList {}
#sub groupID {}

sub addUser {
  my %args = @_;

  _commoncheck(\%args, 'y');

# not sure how to usefully represent permissions from any further out from DNSDB.pm :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  my $perms = {domain_edit => 1, domain_create => 1, domain_delete => 1,
	record_edit => 1, record_create => 1, record_delete => 1
	};
  # bend and twist;  get those arguments in in the right order!
  $args{type} = 'u' if !$args{type};
  $args{permstring} = 'i' if !defined($args{permstring});
  my @userargs = ($args{username}, $args{group}, $args{pass}, $args{state}, $args{type}, $args{permstring});
  for my $argname ('fname','lname','phone') {
    last if !$args{$argname};
    push @userargs, $args{$argname};
  }
  my ($code,$msg) = DNSDB::addUser($dbh, @userargs);
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub getUserCount {}
#sub getUserList {}
#sub getUserDropdown {}
#sub checkUser {}

sub updateUser {
  my %args = @_;

  _commoncheck(\%args, 'y');

  die "Missing UID\n" if !$args{uid};

# not sure how to usefully represent permissions from any further out from DNSDB.pm :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  my $perms = {domain_edit => 1, domain_create => 1, domain_delete => 1,
	record_edit => 1, record_create => 1, record_delete => 1
	};
  # bend and twist;  get those arguments in in the right order!
  my @userargs = ($args{uid}, $args{username}, $args{group}, $args{pass}, $args{state}, $args{type});
  for my $argname ('fname','lname','phone') {
    last if !$args{$argname};
    push @userargs, $args{$argname};
  }
##fixme:  also underlying in DNSDB::updateUser():  no way to just update this or that attribute;
#         have to pass them all in to be overwritten
  my ($code,$msg) = DNSDB::addUser($dbh, @userargs);
  die $msg if $code eq 'FAIL';
}

sub delUser {
  my %args = @_;

  _commoncheck(\%args, 'y');

  die "Missing UID\n" if !$args{uid};
  my ($code,$msg) = DNSDB::delUser($dbh, $args{uid});
  die $msg if $code eq 'FAIL';
}

#sub userFullName {}
#sub userStatus {}
#sub getUserData {}

#sub addLoc {}
#sub updateLoc {}
#sub delLoc {}
#sub getLoc {}
#sub getLocCount {}
#sub getLocList {}
#sub getLocDropdown {}

sub getSOA {
  my %args = @_;

  _commoncheck(\%args);

  my %ret = DNSDB::getSOA($dbh, $args{def}, $args{id});
  if (!$ret{recid}) {
    if ($args{def} eq 'y') {
      die "No default SOA record in group\n";
    } else {
      die "No SOA record in domain\n";
    }
  }
  return \%ret;
}

#sub updateSOA {}

sub getRecLine {
  my %args = @_;

  _commoncheck(\%args);

  my $ret = DNSDB::getRecLine($dbh, $args{def}, $args{id});

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub getDomRecs {
  my %args = @_;

  _commoncheck(\%args);

  # set some optional args
  $args{nrecs} = 'all' if !$args{nrecs};
  $args{nstart} = 0 if !$args{nstart};
## for order, need to map input to column names
  $args{order} = 'host' if !$args{order};
  $args{direction} = 'ASC' if !$args{direction};

  my $ret = DNSDB::getDomRecs($dbh, (defrec => $args{defrec}, revrec => $args{revrec}, id => $args{id},
	offset => $args{offset}, sortby => $args{sortby}, sortorder => $args{sortorder},
	filter => $args{filter}) );

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub getRecCount {
  my %args = @_;

  _commoncheck(\%args);

  # set some optional args
  $args{nrecs} = 'all' if !$args{nrecs};
  $args{nstart} = 0 if !$args{nstart};
## for order, need to map input to column names
  $args{order} = 'host' if !$args{order};
  $args{direction} = 'ASC' if !$args{direction};

  my $ret = DNSDB::getRecCount($dbh, (defrec => $args{defrec}, revrec => $args{revrec}, id => $args{id},
	offset => $args{offset}, sortby => $args{sortby}, sortorder => $args{sortorder},
	filter => $args{filter}) );

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub addRec {
  my %args = @_;

  _commoncheck(\%args, 'y');

  # note dist, weight, port are not required on all types;  will be ignored if not needed.
  my ($code, $msg) = DNSDB::addRec($dbh, $args{def}, $args{domid}, $args{host}, $DNSDB::typemap{$args{type}},
	$args{val}, $args{ttl}, $args{dist}, $args{weight}, $args{port});

  die $msg if $code eq 'FAIL';
}

sub updateRec {
  my %args = @_;

  _commoncheck(\%args, 'y');

  # note dist, weight, port are not required on all types;  will be ignored if not needed.
  my ($code, $msg) = DNSDB::updateRec($dbh, $args{def}, $args{recid}, $args{host}, $DNSDB::typemap{$args{type}},
	$args{val}, $args{ttl}, $args{dist}, $args{weight}, $args{port});

  die $msg if $code eq 'FAIL';
}

sub delRec {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my ($code, $msg) = DNSDB::delRec($dbh, $args{def}, $args{recid});

  die $msg if $code eq 'FAIL';
}

#sub getLogCount {}
#sub getLogEntries {}
#sub getTypelist {}
#sub parentID {}
#sub isParent {}

sub zoneStatus {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my @arglist = ($dbh, $args{zoneid});
  push @arglist, $args{status} if defined($args{status});

  my $status = DNSDB::zoneStatus(@arglist);
}

#sub importAXFR {}
#sub importBIND {}
#sub import_tinydns {}
#sub export {}
#sub __export_tiny {}
#sub _printrec_tiny {}
#sub mailNotify {}

sub get_method_list {
  my @methods = keys %{$methods};
  return \@methods;
}
