#!/usr/bin/perl -w -T
# XMLRPC interface to manipulate most DNS DB entities
##
# $Id: dns-rpc.cgi 452 2013-01-11 17:21:13Z kdeugau $
# Copyright 2012 Kris Deugau <kdeugau@deepnet.cx>
# 
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version. 
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##

use strict;
use warnings;

# don't remove!  required for GNU/FHS-ish install from tarball
use lib '.';	##uselib##

use DNSDB;	# note we're not importing subs;  this lets us (ab)use the same sub names here for convenience
use Data::Dumper;

#use Frontier::RPC2;
use Frontier::Responder;

## We need to handle a couple of things globally, rather than pasting the same bit into *every* sub.
## So, let's subclass Frontier::RPC2 + Frontier::Responder, so we can override the single sub in each
## that needs kicking
#### hmm.  put this in a separate file?
#package DNSDB::RPC;
#our @ISA = ("Frontier::RPC2", "Frontier::Responder");
#package main;

DNSDB::loadConfig(rpcflag => 1);

# need to create a DNSDB object too
my ($dbh,$msg) = DNSDB::connectDB($DNSDB::config{dbname}, $DNSDB::config{dbuser},
	$DNSDB::config{dbpass}, $DNSDB::config{dbhost});

DNSDB::initGlobals($dbh);

my $methods = {
	'dnsdb.addDomain'	=> \&addDomain,
	'dnsdb.delZone'		=> \&delZone,
	'dnsdb.addRDNS'		=> \&addRDNS,
	'dnsdb.addGroup'	=> \&addGroup,
	'dnsdb.delGroup'	=> \&delGroup,
	'dnsdb.addUser'		=> \&addUser,
	'dnsdb.updateUser'	=> \&updateUser,
	'dnsdb.delUser'		=> \&delUser,
	'dnsdb.getLocDropdown'	=> \&getLocDropdown,
	'dnsdb.getSOA'		=> \&getSOA,
	'dnsdb.getRecLine'	=> \&getRecLine,
	'dnsdb.getDomRecs'	=> \&getDomRecs,
	'dnsdb.getRecCount'	=> \&getRecCount,
	'dnsdb.addRec'		=> \&addRec,
	'dnsdb.updateRec'	=> \&updateRec,
	'dnsdb.delRec'		=> \&delRec,
#sub getLogCount {}
#sub getLogEntries {}
	'dnsdb.getRevPattern'	=> \&getRevPattern,
	'dnsdb.zoneStatus'	=> \&zoneStatus,
	'dnsdb.getZonesByCIDR'	=> \&getZonesByCIDR,

	'dnsdb.getMethods'	=> \&get_method_list
};

my $res = Frontier::Responder->new(
	methods => $methods
	);

# "Can't do that" errors
if (!$dbh) {
  print "Content-type: text/xml\n\n".$res->{_decode}->encode_fault(5, $msg);
  exit;
}
##fixme:  fail on missing rpcuser/rpcsystem args

print $res->answer;

exit;

##
## Subs below here
##

# Utility subs
sub _aclcheck {
  my $subsys = shift;
  return 1 if grep /$ENV{REMOTE_ADDR}/, @{$DNSDB::config{rpcacl}{$subsys}};
  return 0;
}

# Let's see if we can factor these out of the RPC method subs
sub _commoncheck {
  my $argref = shift;
  my $needslog = shift;

  die "Missing remote system name\n" if !$argref->{rpcsystem};
  die "Access denied\n" if !_aclcheck($argref->{rpcsystem});
  if ($needslog) {
    die "Missing remote username\n" if !$argref->{rpcuser};
    die "Couldn't set userdata for logging\n"
	unless DNSDB::initRPC($dbh, (username => $argref->{rpcuser}, rpcsys => $argref->{rpcsystem},
		fullname => ($argref->{fullname} ? $argref->{fullname} : $argref->{rpcuser}) ) );
  }
}

#sub connectDB {
#sub finish {
#sub initGlobals {
#sub initPermissions {
#sub getPermissions {
#sub changePermissions {
#sub comparePermissions {
#sub changeGroup {
#sub _log {

sub addDomain {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my ($code, $msg) = DNSDB::addDomain($dbh, $args{domain}, $args{group}, $args{state});
  die $msg if $code eq 'FAIL';
  return $msg;	# domain ID
}

sub delZone {
  my %args = @_;

  _commoncheck(\%args, 'y');
  die "Need forward/reverse zone flag\n" if !$args{revrec};

  my ($code,$msg);
  # Let's be nice;  delete based on zone id OR zone name.  Saves an RPC call round-trip, maybe.
  if ($args{zone} =~ /^\d+$/) {
    ($code,$msg) = DNSDB::delZone($dbh, $args{zone}, $args{revrec});
  } else {
    my $zoneid;
    $zoneid = DNSDB::domainID($dbh, $args{zone}) if $args{revrec} eq 'n';
    $zoneid = DNSDB::revID($dbh, $args{zone}) if $args{revrec} eq 'y';
    die "Can't find zone: $DNSDB::errstr\n" if !$zoneid;
    ($code,$msg) = DNSDB::delZone($dbh, $zoneid, $args{revrec});
  }
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub domainName {}
#sub revName {}
#sub domainID {}
#sub revID {}

sub addRDNS {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my ($code, $msg) = DNSDB::addRDNS($dbh, $args{revzone}, $args{revpatt}, $args{group}, $args{state}, $args{defloc});
  die "$msg\n" if $code eq 'FAIL';
  return $msg;	# domain ID
}

#sub getZoneCount {}
#sub getZoneList {}
#sub getZoneLocation {}

sub addGroup {
  my %args = @_;

  _commoncheck(\%args, 'y');
  die "Missing new group name\n" if !$args{groupname};
  die "Missing parent group ID\n" if !$args{parent_id};

# not sure how to usefully represent permissions via RPC.  :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  my $perms = {domain_edit => 1, domain_create => 1, domain_delete => 1,
	record_edit => 1, record_create => 1, record_delete => 1
	};
## optional $inhert arg?
  my ($code,$msg) = DNSDB::addGroup($dbh, $args{groupname}, $args{parent_id}, $perms);
  die $msg if $code eq 'FAIL';
  return $msg;
}

sub delGroup {
  my %args = @_;

  _commoncheck(\%args, 'y');
  die "Missing group ID or name to remove\n" if !$args{group};

  my ($code,$msg);
  # Let's be nice;  delete based on groupid OR group name.  Saves an RPC call round-trip, maybe.
  if ($args{group} =~ /^\d+$/) {
    ($code,$msg) = DNSDB::delGroup($dbh, $args{group});
  } else {
    my $grpid = DNSDB::groupID($dbh, $args{group});
    die "Can't find group\n" if !$grpid;
    ($code,$msg) = DNSDB::delGroup($dbh, $grpid);
  }
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub getChildren {}
#sub groupName {}
#sub getGroupCount {}
#sub getGroupList {}
#sub groupID {}

sub addUser {
  my %args = @_;

  _commoncheck(\%args, 'y');

# not sure how to usefully represent permissions via RPC.  :/
# not to mention, permissions are checked at the UI layer, not the DB layer.
  # bend and twist;  get those arguments in in the right order!
  $args{type} = 'u' if !$args{type};
  $args{permstring} = 'i' if !defined($args{permstring});
  my @userargs = ($args{username}, $args{group}, $args{pass}, $args{state}, $args{type}, $args{permstring});
  for my $argname ('fname','lname','phone') {
    last if !$args{$argname};
    push @userargs, $args{$argname};
  }
  my ($code,$msg) = DNSDB::addUser($dbh, @userargs);
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub getUserCount {}
#sub getUserList {}
#sub getUserDropdown {}
#sub checkUser {}

sub updateUser {
  my %args = @_;

  _commoncheck(\%args, 'y');

  die "Missing UID\n" if !$args{uid};

  # bend and twist;  get those arguments in in the right order!
  $args{type} = 'u' if !$args{type};
  my @userargs = ($args{uid}, $args{username}, $args{group}, $args{pass}, $args{state}, $args{type});
  for my $argname ('fname','lname','phone') {
    last if !$args{$argname};
    push @userargs, $args{$argname};
  }
##fixme:  also underlying in DNSDB::updateUser():  no way to just update this or that attribute;
#         have to pass them all in to be overwritten
  my ($code,$msg) = DNSDB::updateUser($dbh, @userargs);
  die $msg if $code eq 'FAIL';
  return $msg;
}

sub delUser {
  my %args = @_;

  _commoncheck(\%args, 'y');

  die "Missing UID\n" if !$args{uid};
  my ($code,$msg) = DNSDB::delUser($dbh, $args{uid});
  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub userFullName {}
#sub userStatus {}
#sub getUserData {}

#sub addLoc {}
#sub updateLoc {}
#sub delLoc {}
#sub getLoc {}
#sub getLocCount {}
#sub getLocList {}

sub getLocDropdown {
  my %args = @_;

  _commoncheck(\%args);
  $args{defloc} = '' if !$args{defloc};

  my $ret = DNSDB::getLocDropdown($dbh, $args{group}, $args{defloc});
  return $ret;
}

sub getSOA {
  my %args = @_;

  _commoncheck(\%args);

  my $ret = DNSDB::getSOA($dbh, $args{defrec}, $args{revrec}, $args{id});
  if (!$ret) {
    if ($args{defrec} eq 'y') {
      die "No default SOA record in group\n";
    } else {
      die "No SOA record in zone\n";
    }
  }
  return $ret;
}

#sub updateSOA {}

sub getRecLine {
  my %args = @_;

  _commoncheck(\%args);

  my $ret = DNSDB::getRecLine($dbh, $args{defrec}, $args{revrec}, $args{id});

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub getDomRecs {
  my %args = @_;

  _commoncheck(\%args);

  # set some optional args
  $args{nrecs} = 'all' if !$args{nrecs};
  $args{nstart} = 0 if !$args{nstart};
## for order, need to map input to column names
  $args{order} = 'host' if !$args{order};
  $args{direction} = 'ASC' if !$args{direction};

  my $ret = DNSDB::getDomRecs($dbh, (defrec => $args{defrec}, revrec => $args{revrec}, id => $args{id},
	offset => $args{offset}, sortby => $args{sortby}, sortorder => $args{sortorder},
	filter => $args{filter}) );

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub getRecCount {
  my %args = @_;

  _commoncheck(\%args);

  # set some optional args
  $args{nrecs} = 'all' if !$args{nrecs};
  $args{nstart} = 0 if !$args{nstart};
## for order, need to map input to column names
  $args{order} = 'host' if !$args{order};
  $args{direction} = 'ASC' if !$args{direction};

  my $ret = DNSDB::getRecCount($dbh, $args{defrec}, $args{revrec}, $args{id}, $args{filter});

  die $DNSDB::errstr if !$ret;

  return $ret;
}

sub addRec {
  my %args = @_;

  _commoncheck(\%args, 'y');

  # add records in the zone's default location if none is specified
  if (!$args{location} && $args{defrec} eq 'n') {
    $args{location} = DNSDB::getZoneLocation($dbh, $args{revrec}, $args{parent_id});
  }

  # callers may often not care about TTLs
  if (!$args{ttl}) {
    my $tmp = DNSDB::getSOA($dbh, $args{defrec}, $args{revrec}, $args{parent_id});
    $args{ttl} = $tmp->{minttl};
  }

  my @recargs = ($dbh, $args{defrec}, $args{revrec}, $args{parent_id},
	\$args{name}, \$args{type}, \$args{address}, $args{ttl}, $args{location});
  if ($args{type} == $DNSDB::reverse_typemap{MX} or $args{type} == $DNSDB::reverse_typemap{SRV}) {
    push @recargs, $args{distance};
    if ($args{type} == $DNSDB::reverse_typemap{SRV}) {
      push @recargs, $args{weight};
      push @recargs, $args{port};
    }
  }

  my ($code, $msg) = DNSDB::addRec(@recargs);

  die $msg if $code eq 'FAIL';
  return $msg;
}

sub updateRec {
  my %args = @_;

  _commoncheck(\%args, 'y');

  # get old line, so we can update only the bits that the caller passed to change
  # note we subbed address for val since it's a little more caller-friendly
  my $oldrec = DNSDB::getRecLine($dbh, $args{defrec}, $args{revrec}, $args{id});
  foreach my $field (qw(name type address ttl location distance weight port)) {
    $args{$field} = $oldrec->{$field} if !$args{$field} && defined($oldrec->{$field});
  }

  # note dist, weight, port are not required on all types;  will be ignored if not needed.
  # parent_id is the "primary" zone we're updating;  necessary for forward/reverse voodoo
  my ($code, $msg) = DNSDB::updateRec($dbh, $args{defrec}, $args{revrec}, $args{id}, $args{parent_id},
	\$args{name}, \$args{type}, \$args{address}, $args{ttl}, $args{location},
	$args{distance}, $args{weight}, $args{port});

  die $msg if $code eq 'FAIL';
  return $msg;
}

sub delRec {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my ($code, $msg) = DNSDB::delRec($dbh, $args{defrec}, $args{recrev}, $args{id});

  die $msg if $code eq 'FAIL';
  return $msg;
}

#sub getLogCount {}
#sub getLogEntries {}

sub getRevPattern {
  my %args = @_;

  _commoncheck(\%args, 'y');

  return DNSDB::getRevPattern($dbh, $args{cidr}, $args{group});
}

#sub getTypelist {}
#sub parentID {}
#sub isParent {}

sub zoneStatus {
  my %args = @_;

  _commoncheck(\%args, 'y');

  my @arglist = ($dbh, $args{zoneid});
  push @arglist, $args{status} if defined($args{status});

  my $status = DNSDB::zoneStatus(@arglist);
}

# Get a list of hashes referencing the reverse zone(s) for a passed CIDR block
sub getZonesByCIDR {
  my %args = @_;

  _commoncheck(\%args, 'y');

  return DNSDB::getZonesByCIDR($dbh, %args);
}

#sub importAXFR {}
#sub importBIND {}
#sub import_tinydns {}
#sub export {}
#sub __export_tiny {}
#sub _printrec_tiny {}
#sub mailNotify {}

sub get_method_list {
  my @methods = keys %{$methods};
  return \@methods;
}
