# dns/trunk/DNSDB.pm
# Abstraction functions for DNS administration
##
# $Id: DNSDB.pm 708 2016-03-09 19:09:07Z kdeugau $
# Copyright 2008-2013 Kris Deugau <kdeugau@deepnet.cx>
# 
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version. 
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##

package DNSDB;

use strict;
use warnings;
use Exporter;
use DBI;
use Net::DNS;
use Crypt::PasswdMD5;
use Digest::MD5 qw(md5_hex);
use Net::SMTP;
use NetAddr::IP 4.027 qw(:lower);
use POSIX;
use Fcntl qw(:flock);
use Time::TAI64 qw(:tai64);

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

$VERSION	= 1.3;	##VERSION##
@ISA		= qw(Exporter);
@EXPORT_OK	= qw(
	&initGlobals &login &initActionLog
	&getPermissions &changePermissions &comparePermissions
	&changeGroup
	&connectDB &finish
	&addDomain &delZone &domainName &revName &domainID &revID &addRDNS
	&getZoneCount &getZoneList &getZoneLocation
	&addGroup &delGroup &getChildren &groupName
	&getGroupCount &getGroupList
	&addUser &updateUser &delUser &userFullName &userStatus &getUserData
	&getUserCount &getUserList &getUserDropdown
	&addLoc &updateLoc &delLoc &getLoc
	&getLocCount &getLocList &getLocDropdown
	&getSOA	&updateSOA &getRecLine &getRecList &getRecCount
	&addRec &updateRec &delRec
	&getLogCount &getLogEntries
	&getRevPattern
	&getTypelist
	&parentID
	&isParent
	&zoneStatus &getZonesByCIDR &importAXFR
	&export
	&mailNotify
	%typemap %reverse_typemap
	@permtypes $permlist %permchains
	);

@EXPORT		= qw(%typemap %reverse_typemap @permtypes $permlist %permchains);
%EXPORT_TAGS	= ( ALL => [qw(
		&initGlobals &login &initActionLog
		&getPermissions &changePermissions &comparePermissions
		&changeGroup
		&connectDB &finish
		&addDomain &delZone &domainName &revName &domainID &revID &addRDNS
		&getZoneCount &getZoneList &getZoneLocation
		&addGroup &delGroup &getChildren &groupName
		&getGroupCount &getGroupList
		&addUser &updateUser &delUser &userFullName &userStatus &getUserData
		&getUserCount &getUserList &getUserDropdown
		&addLoc &updateLoc &delLoc &getLoc
		&getLocCount &getLocList &getLocDropdown
		&getSOA &updateSOA &getRecLine &getRecList &getRecCount
		&addRec &updateRec &delRec
		&getLogCount &getLogEntries
		&getRevPattern
		&getTypelist
		&parentID
		&isParent
		&zoneStatus &getZonesByCIDR &importAXFR
		&export
		&mailNotify
		%typemap %reverse_typemap
		@permtypes $permlist %permchains
		)]
	);

our $errstr = '';
our $resultstr = '';

# Arguably defined wholly in the db, but little reason to change without supporting code changes
# group_view, user_view permissions? separate rDNS permission(s)?
our @permtypes = qw (
	group_edit	group_create	group_delete
	user_edit	user_create	user_delete
	domain_edit	domain_create	domain_delete
	record_edit	record_create	record_delete	record_locchg
	location_edit	location_create	location_delete	location_view
	self_edit	admin
);
our $permlist = join(',',@permtypes);

# Some permissions more or less require certain others.
our %permchains = (
	user_edit	=> 'self_edit',
	location_edit	=> 'location_view',
	location_create	=> 'location_view',
	location_delete	=> 'location_view',
	record_locchg	=> 'location_view',
);

# DNS record type map and reverse map.
# loaded from the database, from http://www.iana.org/assignments/dns-parameters
our %typemap;
our %reverse_typemap;

## (Semi)private variables

# Hash of functions for validating record types.  Filled in initGlobals() since
# it relies on visibility flags from the rectypes table in the DB
my %validators;

# Entity-relationship reference hashes.
my %par_tbl = (
		group	=> 'groups',
		user	=> 'users',
		defrec	=> 'default_records',
		defrevrec	=> 'default_rev_records',
		domain	=> 'domains',
		revzone	=> 'revzones',
		record	=> 'records'
	);
my %id_col = (
		group	=> 'group_id',
		user	=> 'user_id',
		defrec	=> 'record_id',
		defrevrec	=> 'record_id',
		domain	=> 'domain_id',
		revzone	=> 'rdns_id',
		record	=> 'record_id'
	);
my %par_col = (
		group	=> 'parent_group_id',
		user	=> 'group_id',
		defrec	=> 'group_id',
		defrevrec	=> 'group_id',
		domain	=> 'group_id',
		revzone	=> 'group_id',
		record	=> 'domain_id'
	);
my %par_type = (
		group	=> 'group',
		user	=> 'group',
		defrec	=> 'group',
		defrevrec	=> 'group',
		domain	=> 'group',
		revzone	=> 'group',
		record	=> 'domain'
	);

##
## Constructor and destructor
##

sub new {
  my $this = shift;
  my $class = ref($this) || $this;
  my %args = @_;

  # Prepopulate a basic config.  Note some of these *will* cause errors if left unset.
  # note:  add appropriate stanzas in __cfgload() to parse these
  my %defconfig = (
		# The only configuration options not loadable from a config file.
		configfile => "/etc/dnsdb/dnsdb.conf",  ##CFG_LEAF##

		# Database connection info
		dbname	=> 'dnsdb',
		dbuser	=> 'dnsdb',
		dbpass	=> 'secret',
		dbhost	=> '',

		# Email notice settings
		mailhost	=> 'smtp.example.com',
		mailnotify	=> 'dnsdb@example.com',	# to
		mailsender	=> 'dnsdb@example.com',	# from
		mailname	=> 'DNS Administration',
		orgname		=> 'Example Corp',
		domain		=> 'example.com',

		# Template directory
		templatedir	=> 'templates/',
# fmeh.  this is a real web path, not a logical internal one.  hm..
#		cssdir	=> 'templates/',
		sessiondir	=> 'session/',
		exportcache	=> 'cache/',
		usecache	=> 1,	# do we bother using the cache above?

		# Session params
		timeout		=> '1h',	# passed as-is to CGI::Session

		# Other miscellanea
		log_failures	=> 1,	# log all evarthing by default
		perpage		=> 15,
		maxfcgi		=> 10,	# reasonable default?
		force_refresh	=> 1,
		lowercase	=> 0,	# mangle as little as possible by default
		# show IPs and CIDR blocks as-is for reverse zones.  valid values are
		# 'none' (default, show natural IP or CIDR)
		# 'zone' (zone name, wherever used)
		# 'record' (IP or CIDR values in reverse record lists)
		# 'all' (all IP values in any reverse zone view)
		showrev_arpa	=> 'none',
		# Two options for template record expansion:
		template_skip_0	=> 0,	# publish .0 by default
		template_skip_255	=> 0,	# publish .255 by default
		# allow TXT records to be dealt with mostly automatically by DNS server?
		autotxt		=> 1,
	);

  # Config file parse calls.
  # If we are passed a blank argument for $args{configfile},
  #   we should NOT parse the default config file - we will
  #   rely on hardcoded defaults OR caller-specified values.
  # If we are passed a non-blank argument, parse that file.
  # If no config file is specified, parse the default one.
  my %siteconfig;
  if (defined($args{configfile})) {
    if ($args{configfile}) {
      return if !__cfgload($args{configfile}, \%siteconfig);
    }
  } else {
    return if !__cfgload($defconfig{configfile}, \%siteconfig);
  }

  # Assemble the object.  Apply configuration hashes in order of precedence.
  my $self = {
	# Hardcoded defaults
	%defconfig,
	# Default config file OR caller-specified one, loaded above
	%siteconfig,
	# Caller-specified arguments
	%args
	};
  bless $self, $class;

  # Several settings are booleans.  Handle multiple possible ways of setting them.
  for my $boolopt ('log_failures', 'force_refresh', 'lowercase', 'usecache',
	'template_skip_0', 'template_skip_255', 'autotxt') {
    if ($self->{$boolopt} ne '1' && $self->{$boolopt} ne '0') {
      # true/false, on/off, yes/no all valid.
      if ($self->{$boolopt} =~ /^(?:true|false|t|f|on|off|yes|no)$/) {
        if ($self->{$boolopt} =~ /(?:true|t|on|yes)/) {
         $self->{$boolopt} = 1;
        } else {
         $self->{$boolopt} = 0;
        }
      } else {
        warn "Bad $boolopt setting $self->{$boolopt}, using default\n";
        $self->{$boolopt} = $defconfig{$boolopt};
      }
    }
  }

  # Enum-ish option(s)
  if (!grep /$self->{showrev_arpa}/, ('none','zone','record','all')) {
    warn "Bad showrev_arpa setting $self->{showrev_arpa}, using default\n";
    $self->{showrev_arpa} = 'none';
  }

  # Try to connect to the DB, and initialize a number of handy globals.
  $self->{dbh} = connectDB($self->{dbname}, $self->{dbuser}, $self->{dbpass}, $self->{dbhost}) or return;
  $self->initGlobals();

  return $self;
}

sub DESTROY {
  my $self = shift;
  $self->{dbh}->disconnect if $self->{dbh};
}

sub errstr { $DNSDB::errstr; }

##
## utility functions
##

## DNSDB::_rectable()
# Takes default+rdns flags, returns appropriate table name
sub _rectable {
  my $def = shift;
  my $rev = shift;

  return 'records' if $def ne 'y';
  return 'default_records' if $rev ne 'y';
  return 'default_rev_records';
} # end _rectable()

## DNSDB::_recparent()
# Takes default+rdns flags, returns appropriate parent-id column name
sub _recparent {
  my $def = shift;
  my $rev = shift;

  return 'group_id' if $def eq 'y';
  return 'rdns_id' if $rev eq 'y';
  return 'domain_id';
} # end _recparent()

## DNSDB::_ipparent()
# Check an IP to be added in a reverse zone to see if it's really in the requested parent.
# Takes default and reverse flags, IP (fragment) to check, parent zone ID,
# and a reference to a NetAddr::IP object (also used to pass back a fully-reconstructed IP for
# database insertion)
sub _ipparent {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $defrec = shift;
  my $revrec = shift;
  my $val = shift;
  my $id = shift;
  my $addr = shift;

  return if $revrec ne 'y';	# this sub not useful in forward zones

  $$addr = NetAddr::IP->new($$val);	 #necessary?

  # subsub to split, reverse, and overlay an IP fragment on a netblock
  sub __rev_overlay {
    my $splitme = shift;	# ':' or '.', m'lud?
    my $parnet = shift;
    my $val = shift;
    my $addr = shift;

    my $joinme = $splitme;
    $splitme = '\.' if $splitme eq '.';
    my @working = reverse(split($splitme, $parnet->addr));
    my @parts = reverse(split($splitme, $$val));
    for (my $i = 0; $i <= $#parts; $i++) {
      $working[$i] = $parts[$i];
    }
    my $checkme = NetAddr::IP->new(join($joinme, reverse(@working))) or return 0;
    return 0 unless $checkme->within($parnet);
    $$addr = $checkme;	# force "correct" IP to be recorded.
    return 1;
  }

  my ($parstr) = $dbh->selectrow_array("SELECT revnet FROM revzones WHERE rdns_id = ?", undef, ($id));
  my $parnet = NetAddr::IP->new($parstr);

  # Fail early on v6-in-v4 or v4-in-v6.  We're not accepting these ATM.
  return 0 if $parnet->addr =~ /\./ && $$val =~ /:/;
  return 0 if $parnet->addr =~ /:/ && $$val =~ /\./;

  if ($$addr && ($$val =~ /^[\da-fA-F][\da-fA-F:]+[\da-fA-F]$/ || $$val =~ m|/\d+$|)) {
    # the only case where NetAddr::IP's acceptance of legitimate IPs is "correct" is for a proper IPv6 address,
    # or a netblock (only expected on templates)
    # the rest we have to restructure before fiddling.  *sigh*
    return 1 if $$addr->within($parnet);
  } else {
    # We don't have a complete IP in $$val (yet)... unless we have a netblock
    if ($parnet->addr =~ /:/) {
      $$val =~ s/^:+//;	 # gotta strip'em all...
      return __rev_overlay(':', $parnet, $val, $addr);
    }
    if ($parnet->addr =~ /\./) {
      $$val =~ s/^\.+//;
      return __rev_overlay('.', $parnet, $val, $addr);
    }
    # should be impossible to get here...
  }
  # ... and here.
  # can't do nuttin' in forward zones
} # end _ipparent()

## DNSDB::_maybeip()
# Wrapper for quick "does this look like an IP address?" regex, so we don't make dumb copy-paste mistakes
sub _maybeip {
  my $izzit = shift;  # reference
  return 1 if $$izzit =~ m,^(?:[\d\./]+|[0-9a-fA-F:/]+)$,;
}

## DNSDB::_inrev()
# Check if a given "hostname" is within a given reverse zone
# Takes a reference to the "hostname" and the reverse zone CIDR as a NetAddr::IP
# Returns true/false.  Sets $errstr on errors.
sub _inrev {
  my $self = shift;
  my $dbh = $self->{dbh};
  # References, since we might munge them
  my $fq = shift;
  my $zone = shift;

  # set default error
  $errstr = "$$fq not within $zone";

  # Unlike forward zones, we will not coerce the data into the reverse zone - an A record
  # in a reverse zone is already silly enough without appending a mess of 1.2.3.in-addr.arpa
  # (or worse, 1.2.3.4.5.6.7.8.ip6.arpa) on the end of the nominal "hostname".
  # We're also going to allow the "hostname" to be stored as .arpa or IP, because of
  # non-IP FQDNs in .arpa
  if ($$fq =~ /\.arpa$/) {
    # "FQDN" could be any syntactically legitimate string, but it must be within the formal
    # .arpa zone.  Note we're not validating these for correct reverse-IP values.
    # yes, we really need the v6 branch on the end here.
    $zone = _ZONE($zone, 'ZONE', 'r', '.').($zone->{isv6} ? '.ip6.arpa' : '.in-addr.arpa');
    return unless $$fq =~ /$zone$/;
  } else {
    # in most cases we should be getting a real IP as the "FQDN" to test
    my $addr = new NetAddr::IP $$fq if _maybeip($fq);

    # "FQDN" should be a valid IP address.  Normalize formatting if so.
    if (!$addr) {
      $errstr = "$$fq is not a valid IP address";
      return;
    }
    return if !$zone->contains($addr);
    ($$fq = $addr) =~ s{/(?:32|128)$}{};
  }
  return 1;
} # end _inrev()

## DNSDB::_hostparent()
# A little different than _ipparent above;  this tries to *find* the parent zone of a hostname
# Takes a hostname.
# Returns the domain ID of the parent domain if one was found.
sub _hostparent {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $hname = shift;

  $hname =~ s/^\*\.//;	# this should be impossible to find in the domains table.
  my @hostbits = split /\./, $hname;
  my $sth = $dbh->prepare("SELECT count(*),domain_id FROM domains WHERE lower(domain) = lower(?) GROUP BY domain_id");
  foreach (@hostbits) {
    $sth->execute($hname);
    my ($found, $parid) = $sth->fetchrow_array;
    if ($found) {
      return $parid;
    }
    $hname =~ s/^$_\.//;
  }
} # end _hostparent()

## DNSDB::_log()
# Log an action
# Takes a log entry hash containing at least:
#  group_id, log entry
# and optionally one or more of:
#  domain_id, rdns_id
# The %userdata hash provides the user ID, username, and fullname
sub _log {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  $args{rdns_id} = 0 if !$args{rdns_id};
  $args{domain_id} = 0 if !$args{domain_id};

##fixme:  farm out the actual logging to different subs for file, syslog, internal, etc based on config
#  if ($self->{log_channel} eq 'sql') {
  $dbh->do("INSERT INTO log (domain_id,rdns_id,group_id,entry,user_id,email,name) VALUES (?,?,?,?,?,?,?)",
	undef,
	($args{domain_id}, $args{rdns_id}, $args{group_id}, $args{entry},
		$self->{loguserid}, $self->{logusername}, $self->{logfullname}) );
#  } elsif ($self->{log_channel} eq 'file') {
#  } elsif ($self->{log_channel} eq 'syslog') {
#  }
} # end _log


##
## Record validation subs.
##

## All of these subs take substantially the same arguments:
# a hash containing at least the following keys:
#  - defrec (default/live flag)
#  - revrec (forward/reverse flag)
#  - id (parent entity ID)
#  - host (hostname)
#  - rectype
#  - val (IP, hostname [CNAME/MX/SRV] or text)
#  - addr (NetAddr::IP object from val.  May be undef.)
# MX and SRV record validation also expect distance, and SRV records expect weight and port as well.
# host, rectype, and addr should be references as these may be modified in validation

# A record
sub _validate_1 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

# only for strict type restrictions
#  return ('FAIL', 'Reverse zones cannot contain A records') if $args{revrec} eq 'y';

  if ($args{revrec} eq 'y') {
    # Get the revzone, so we can see if ${$args{val}} is in that zone
    my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');

    return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);

    # ${$args{val}} is either a valid IP or a string ending with the .arpa zone name;
    # now check if it's a well-formed FQDN
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) &&
	${$args{val}} =~ /\.arpa$/;

    # Check IP is well-formed, and that it's a v4 address
    # Fail on "compact" IPv4 variants, because they are not consistent and predictable.
    return ('FAIL',"A record must be a valid IPv4 address")
	unless ${$args{host}} =~ m{^\d+\.\d+\.\d+\.\d+(/\d+)?$};
    $args{addr} = new NetAddr::IP ${$args{host}};
    return ('FAIL',"A record must be a valid IPv4 address")
	unless $args{addr} && !$args{addr}->{isv6};
    # coerce IP/value to normalized form for storage
    ${$args{host}} = $args{addr}->addr;

    # I'm just going to ignore the utterly barmy idea of an A record in the *default*
    # records for a reverse zone;  it's bad enough to find one in funky legacy data.

  } else {
    # revrec ne 'y'

    # Coerce all hostnames to end in ".DOMAIN" for group/default records,
    # or the intended parent domain for live records.
    my $pname = ($args{defrec} eq 'y' ? 'DOMAIN' : $self->domainName($args{id}));
    ${$args{host}} =~ s/\.*$/\.$pname/ if (${$args{host}} ne '@' && ${$args{host}} !~ /$pname$/i);

    # Check if it's a proper formal .arpa name for an IP, and renormalize it to the IP
    # value if so.  Done mainly for symmetry with PTR/A+PTR, and saves a conversion on export.
    if (${$args{val}} =~ /\.arpa$/) {
      my ($code,$tmp) = _zone2cidr(${$args{val}});
      if ($code ne 'FAIL') {
        ${$args{val}} = $tmp->addr;
        $args{addr} = $tmp;
      }
    }
    # Check IP is well-formed, and that it's a v4 address
    # Fail on "compact" IPv4 variants, because they are not consistent and predictable.
    return ('FAIL',"A record must be a valid IPv4 address")
	unless ${$args{val}} =~ /^\d+\.\d+\.\d+\.\d+$/;
    $args{addr} = new NetAddr::IP ${$args{val}};
    return ('FAIL',"A record must be a valid IPv4 address")
	unless $args{addr} && !$args{addr}->{isv6};
    # coerce IP/value to normalized form for storage
    ${$args{val}} = $args{addr}->addr;
  }

  return ('OK','OK');
} # done A record

# NS record
sub _validate_2 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # NS target check - IP addresses not allowed.  Must be a more or less well-formed hostname.
  if ($args{revrec} eq 'y') {
    return ('FAIL', "NS records cannot point directly to an IP address")
      if ${$args{host}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;
##enhance:  Look up the passed value to see if it exists.  Ooo, fancy.
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
  } else {
    return ('FAIL', "NS records cannot point directly to an IP address")
      if ${$args{val}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;
##enhance:  Look up the passed value to see if it exists.  Ooo, fancy.
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
  }

  # Check that the target of the record is within the parent.
  if ($args{defrec} eq 'n') {
    # Check if IP/address/zone/"subzone" is within the parent
    if ($args{revrec} eq 'y') {
      # Get the revzone, so we can see if ${$args{val}} is in that zone
      my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');

      # Note the NS record may or may not be for the zone itself, it may be a pointer for a subzone
      return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);

      # ${$args{val}} is either a valid IP or a string ending with the .arpa zone name;
      # now check if it's a well-formed FQDN
##enhance or ##fixme
# convert well-formed .arpa names to IP addresses to match old "strict" validation design
      return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) &&
        ${$args{val}} =~ /\.arpa$/;
    } else {
      # Forcibly append the domain name if the hostname being added does not end with the current domain name
      my $pname = $self->domainName($args{id});
      ${$args{host}} =~ s/\.*$/\.$pname/ if (${$args{host}} ne '@' && ${$args{host}} !~ /$pname$/i);
    }
  } else {
    # Default reverse NS records should always refer to the implied parent.  
    if ($args{revrec} eq 'y') {
      ${$args{val}} = 'ZONE';
    } else {
      ${$args{host}} = 'DOMAIN';
    }    
  }

  return ('OK','OK');
} # done NS record

# CNAME record
sub _validate_5 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # CNAMEs in reverse zones shouldn't be handled manually, they should be generated on
  # export by use of the "delegation" type.  For the masochistic, and those importing
  # legacy data from $deity-knows-where, we'll support them.

  if ($args{revrec} eq 'y') {
    # CNAME target check - IP addresses not allowed.  Must be a more or less well-formed hostname.
    return ('FAIL', "CNAME records cannot point directly to an IP address")
      if ${$args{host}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;

    if ($args{defrec} eq 'n') {
      # Get the revzone, so we can see if ${$args{val}} is in that zone
      my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');
      return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);
      # CNAMEs can not be used for parent nodes;  just leaf nodes with no other record types
      # note that this WILL probably miss some edge cases.
      if (${$args{val}} =~ /^[\d.\/]+$/) {
        # convert IP "hostname" to .arpa
        my $tmphn = _ZONE(NetAddr::IP->new(${$args{val}}), 'ZONE', 'r', '.');
        my $tmpz = _ZONE($revzone, 'ZONE', 'r', '.');
        return ('FAIL', "The bare zone may not be a CNAME") if $tmphn eq $tmpz;
##enhance: look up the target name and publish that instead on export
      }
    }

##enhance or ##fixme
# convert well-formed .arpa names to IP addresses to match old "strict" validation design
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) &&
      ${$args{val}} =~ /\.arpa$/;

##enhance:  Look up the passed value to see if it exists.  Ooo, fancy.
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
  } else {
    # CNAME target check - IP addresses not allowed.  Must be a more or less well-formed hostname.
    return ('FAIL', "CNAME records cannot point directly to an IP address")
      if ${$args{val}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;

    # Make sure target is a well-formed hostname
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec});

    # Forcibly append the domain name if the hostname being added does not end with the current domain name
    my $pname = ($args{defrec} eq 'y' ? 'DOMAIN' : $self->domainName($args{id}));
    ${$args{host}} =~ s/\.*$/\.$pname/ if ${$args{host}} !~ /$pname$/i;

    # CNAMEs can not be used for parent nodes;  just leaf nodes with no other record types
    # Enforce this for the zone name
    return ('FAIL', "The bare zone name may not be a CNAME") if ${$args{host}} eq $pname || ${$args{host}} =~ /^\@/;

##enhance:  Look up the passed value to see if it exists.  Ooo, fancy.
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
  }

  return ('OK','OK');
} # done CNAME record

# SOA record
sub _validate_6 {
  # Smart monkeys won't stick their fingers in here;  we have
  # separate dedicated routines to deal with SOA records.
  return ('OK','OK');
} # done SOA record

# PTR record
sub _validate_12 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;
  my $warnflag = '';

  if ($args{defrec} eq 'y') {
    if ($args{revrec} eq 'y') {
      if (${$args{val}} =~ /^[\d.]+$/) {
        # v4 or bare number
        if (${$args{val}} =~ /^\d+\.\d+\.\d+\.\d+$/) {
          # probable full IP.  pointless but harmless.  validate/normalize.
          my $tmp = NetAddr::IP->new(${$args{val}})->addr
            or return ('FAIL', "${$args{val}} is not a valid IP address");
          ${$args{val}} = $tmp;
          $warnflag = "${$args{val}} will only be added to a small number of zones\n";
        } elsif (${$args{val}} =~ /^\d+$/) {
          # bare number.  This can be expanded to either a v4 or v6 zone
          ${$args{val}} =~ s/^\.*/ZONE./ unless ${$args{val}} =~ /^ZONE/;
        } else {
          # $deity-only-knows what kind of gibberish we've been given.  Only usable as a formal .arpa name.
          # Append ARPAZONE to be replaced with the formal .arpa zone name when converted to a live record.
          ${$args{val}} =~ s/\.*$/.ARPAZONE/ unless ${$args{val}} =~ /ARPAZONE$/;
        }
      } elsif (${$args{val}} =~ /^[a-fA-F0-9:]+$/) {
        # v6 or fragment;  pray it's not complete gibberish
        ${$args{val}} =~ s/^:*/ZONE::/ unless ${$args{val}} =~ /^ZONE/;
      } else {
        # $deity-only-knows what kind of gibberish we've been given.  Only usable as a formal .arpa name.
        # Append ARPAZONE to be replaced with the formal .arpa zone name when converted to a live record.
        ${$args{val}} .= ".ARPAZONE" unless ${$args{val}} =~ /ARPAZONE$/;
      }
    } else {
      return ('FAIL', "PTR records are not supported in default record sets for forward zones (domains)");
    }
  } else {
    if ($args{revrec} eq 'y') {
      # Get the revzone, so we can see if ${$args{val}} is in that zone
      my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');

      return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);

      if (${$args{val}} =~ /\.arpa$/) {
        # Check that it's well-formed
        return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec});

        # Check if it's a proper formal .arpa name for an IP, and renormalize it to the IP
        # value if so.  I can't see why someone would voluntarily work with those instead of
        # the natural IP values but what the hey.
        my ($code,$tmp) = _zone2cidr(${$args{val}});
        ${$args{val}} = $tmp->addr if $code ne 'FAIL';
      } else {
        # not a formal .arpa name, so it should be an IP value.  Validate...
        return ('FAIL', "${$args{val}} is not a valid IP value")
            unless ${$args{val}} =~ /^(?:\d+\.\d+\.\d+\.\d+|[a-fA-F0-9:]+)$/;
        $args{addr} = NetAddr::IP->new(${$args{val}})
            or return ('FAIL', "IP/value looks like an IP address but isn't valid");
        # ... and normalize.
        ${$args{val}} = $args{addr}->addr;
      }
      # Validate PTR target for form.
      # %blank% skips the IP when expanding a template record
      return ('FAIL', $errstr)
	unless _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) ||
		lc(${$args{host}}) eq '%blank%';
    } else { # revrec ne 'y'
      # Fetch the domain and append if the passed hostname isn't within it.
      my $pname = ($args{defrec} eq 'y' ? 'DOMAIN' : $self->domainName($args{id}));
      ${$args{host}} =~ s/\.*$/\.$pname/ if (${$args{host}} ne '@' && ${$args{host}} !~ /$pname$/i);
      # Validate hostname and target for form
      return ('FAIL', $errstr) if ! _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
      return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
    }
  }

# Multiple PTR records do NOT generally do what most people believe they do,
# and tend to fail in the most awkward way possible.  Check and warn.

  my $chkbase = ${$args{val}};;
  my $hostcol = 'val';	# Reverse zone hostnames are stored "backwards"
  if ($args{revrec} eq 'n') {	# PTRs in forward zones should be rare.
    $chkbase = ${$args{host}};
    $hostcol = 'host';
  }
  my @checkvals = ($chkbase);
  if ($chkbase =~ /,/) {
    # push . and :: variants into checkvals if $chkbase has ,
    my $tmp;
    ($tmp = $chkbase) =~ s/,/./;
    push @checkvals, $tmp;
    ($tmp = $chkbase) =~ s/,/::/;
    push @checkvals, $tmp;
  }

  my $pcsth = $dbh->prepare("SELECT count(*) FROM "._rectable($args{defrec},$args{revrec})." WHERE $hostcol = ?");
  foreach my $checkme (@checkvals) {
    if ($args{update}) {
      # $args{update} contains the ID of the record being updated.  If the list of records that matches
      # the new hostname specification doesn't include this, the change effectively adds a new PTR that's
      # the same as one or more existing ones.
      my @ptrs = @{ $dbh->selectcol_arrayref("SELECT record_id FROM "._rectable($args{defrec},$args{revrec}).
	" WHERE val = ?", undef, ($checkme)) };
      $warnflag .= "PTR record for $checkme already exists;  adding another will probably not do what you want"
	if @ptrs && (!grep /^$args{update}$/, @ptrs);
    } else {
      # New record.  Always warn if a PTR exists
      # Don't warn when a matching A record exists tho
      my ($ptrcount) = $dbh->selectrow_array("SELECT count(*) FROM "._rectable($args{defrec},$args{revrec}).
	" WHERE $hostcol = ? AND (type=12 OR type=65280 OR type=65281)", undef, ($checkme));
      $warnflag .= "PTR record for $checkme already exists;  adding another will probably not do what you want"
	if $ptrcount;
    }
  }

  return ('WARN',$warnflag) if $warnflag;

  return ('OK','OK');
} # done PTR record

# MX record
sub _validate_15 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

# only for strict type restrictions
#  return ('FAIL', 'Reverse zones cannot contain MX records') if $args{revrec} eq 'y';

  return ('FAIL', "Distance is required for MX records") unless defined(${$args{dist}});
  ${$args{dist}} =~ s/\s*//g;
  return ('FAIL',"Distance is required, and must be numeric") unless ${$args{dist}} =~ /^\d+$/;

  ${$args{fields}} = "distance,";
  push @{$args{vallist}}, ${$args{dist}};

  if ($args{revrec} eq 'n') {
    # MX target check - IP addresses not allowed.  Must be a more or less well-formed hostname.
    return ('FAIL', "MX records cannot point directly to an IP address")
      if ${$args{val}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;

    # Coerce all hostnames to end in ".DOMAIN" for group/default records,
    # or the intended parent domain for live records.
    my $pname = ($args{defrec} eq 'y' ? 'DOMAIN' : $self->domainName($args{id}));
    ${$args{host}} =~ s/\.*$/\.$pname/ if (${$args{host}} ne '@' && ${$args{host}} !~ /$pname$/i);
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
  } else {
    # MX target check - IP addresses not allowed.  Must be a more or less well-formed hostname.
    return ('FAIL', "MX records cannot point directly to an IP address")
      if ${$args{host}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;

    # MX records in reverse zones get stricter treatment.  The UI bars adding them in
    # reverse record sets, but we "need" to allow editing existing ones.  And we'll allow
    # editing them if some loon manually munges one into a default reverse record set.
    if ($args{defrec} eq 'n') {
      # Get the revzone, so we can see if ${$args{val}} is in that zone
      my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');
      return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);
    }

##enhance or ##fixme
# convert well-formed .arpa names to IP addresses to match old "strict" validation design
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) &&
      ${$args{val}} =~ /\.arpa$/;

##enhance:  Look up the passed value to see if it exists.  Ooo, fancy.
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});

  }

  return ('OK','OK');
} # done MX record

# TXT record
sub _validate_16 {
  my $self = shift;

  my %args = @_;

  if ($args{revrec} eq 'n') {
    # Coerce all hostnames to end in ".DOMAIN" for group/default records,
    # or the intended parent domain for live records.
    my $pname = ($args{defrec} eq 'y' ? 'DOMAIN' : $self->domainName($args{id}));
    ${$args{host}} =~ s/\.*$/\.$pname/ if (${$args{host}} ne '@' && ${$args{host}} !~ /$pname$/i);
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
  } else {
    # We don't coerce reverse "hostnames" into the zone, mainly because we store most sane
    # records as IP values, not .arpa names.
    if ($args{defrec} eq 'n') {
      # Get the revzone, so we can see if ${$args{val}} is in that zone
      my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');
      return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);
    }

##enhance or ##fixme
# convert well-formed .arpa names to IP addresses to match old "strict" validation design
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) &&
      ${$args{val}} =~ /\.arpa$/;
  }

  # Could arguably put a WARN return here on very long (>512) records
  return ('OK','OK');
} # done TXT record

# RP record
sub _validate_17 {
  # Probably have to validate these separately some day.  Call _validate_16() above since
  # they're otherwise very similar
  return _validate_16(@_);
} # done RP record

# AAAA record
# Almost but not quite an exact duplicate of A record
sub _validate_28 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

# only for strict type restrictions
#  return ('FAIL', 'Reverse zones cannot contain AAAA records') if $args{revrec} eq 'y';

  if ($args{revrec} eq 'y') {
    # Get the revzone, so we can see if ${$args{val}} is in that zone
    my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');

    return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);

    # ${$args{val}} is either a valid IP or a string ending with the .arpa zone name;
    # now check if it's a well-formed FQDN
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) &&
	${$args{val}} =~ /\.arpa$/;

    # Check IP is well-formed, and that it's a v4 address
    # Fail on "compact" IPv4 variants, because they are not consistent and predictable.
    return ('FAIL',"AAAA record must be a valid IPv6 address")
	unless ${$args{host}} =~ /^[a-fA-F0-9:]+$/;
    $args{addr} = new NetAddr::IP ${$args{host}};
    return ('FAIL',"AAAA record must be a valid IPv6 address")
	unless $args{addr} && $args{addr}->{isv6};
    # coerce IP/value to normalized form for storage
    ${$args{host}} = $args{addr}->addr;

    # I'm just going to ignore the utterly barmy idea of an AAAA record in the *default*
    # records for a reverse zone;  it's bad enough to find one in funky legacy data.

  } else {
    # revrec ne 'y'

    # Coerce all hostnames to end in ".DOMAIN" for group/default records,
    # or the intended parent domain for live records.
    my $pname = ($args{defrec} eq 'y' ? 'DOMAIN' : $self->domainName($args{id}));
    ${$args{host}} =~ s/\.*$/\.$pname/ if (${$args{host}} ne '@' && ${$args{host}} !~ /$pname$/i);

    # Check if it's a proper formal .arpa name for an IP, and renormalize it to the IP
    # value if so.  Done mainly for symmetry with PTR/AAAA+PTR, and saves a conversion on export.
    if (${$args{val}} =~ /\.arpa$/) {
      my ($code,$tmp) = _zone2cidr(${$args{val}});
      if ($code ne 'FAIL') {
        ${$args{val}} = $tmp->addr;
        $args{addr} = $tmp;
      }
    }
    # Check IP is well-formed, and that it's a v6 address
    return ('FAIL',"AAAA record must be a valid IPv6 address")
	unless ${$args{val}} =~ /^[a-fA-F0-9:]+$/;
    $args{addr} = new NetAddr::IP ${$args{val}};
    return ('FAIL',"AAAA record must be a valid IPv6 address")
	unless $args{addr} && $args{addr}->{isv6};
    # coerce IP/value to normalized form for storage
    ${$args{val}} = $args{addr}->addr;
  }

  return ('OK','OK');
} # done AAAA record

# SRV record
sub _validate_33 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

# Not absolutely true but WTF use is an SRV record for a reverse zone?
#  return ('FAIL', 'Reverse zones cannot contain SRV records') if $args{revrec} eq 'y';

  # Key additional record parts.  Always required.
  return ('FAIL',"Distance, port and weight are required for SRV records")
	unless defined(${$args{weight}}) && defined(${$args{port}}) && defined(${$args{dist}});
  ${$args{dist}} =~ s/\s*//g;
  ${$args{weight}} =~ s/\s*//g;
  ${$args{port}} =~ s/\s*//g;
  return ('FAIL',"Distance, port and weight are required, and must be numeric")
	unless ${$args{weight}} =~ /^\d+$/ && ${$args{port}} =~ /^\d+$/;

  ${$args{fields}} = "distance,weight,port,";
  push @{$args{vallist}}, (${$args{dist}}, ${$args{weight}}, ${$args{port}});

  if ($args{revrec} eq 'n') {
    # Coerce all hostnames to end in ".DOMAIN" for group/default records,
    # or the intended parent domain for live records.
    my $pname = ($args{defrec} eq 'y' ? 'DOMAIN' : $self->domainName($args{id}));
    ${$args{host}} =~ s/\.*$/\.$pname/ if ${$args{host}} !~ /$pname$/i;

##enhance:  Rejig so that we can pass back a WARN red flag, instead of
# hard-failing, since it seems that purely from the DNS record perspective,
# SRV records without underscores are syntactically valid
    # Not strictly true, but SRV records not following this convention won't be found.
    return ('FAIL',"SRV records must begin with _service._protocol [${$args{host}}]")
	unless ${$args{host}} =~ /^_[A-Za-z-]+\._[A-Za-z]+\.[a-zA-Z0-9-]+/;

    # SRV target check - IP addresses not allowed.  Must be a more or less well-formed hostname.
    return ('FAIL', "SRV records cannot point directly to an IP address")
      if ${$args{val}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;
  } else {
    # hm.  we can't do anything sane with IP values here;  part of the record data is in
    # fact encoded in the "hostname".  enforce .arpa names?  OTOH, SRV records in a reverse
    # zone are pretty silly.

##enhance:  Rejig so that we can pass back a WARN red flag, instead of
# hard-failing, since it seems that purely from the DNS record perspective,
# SRV records without underscores are syntactically valid
    # Not strictly true, but SRV records not following this convention won't be found.
    return ('FAIL',"SRV records must begin with _service._protocol [${$args{host}}]")
	unless ${$args{val}} =~ /^_[A-Za-z]+\._[A-Za-z]+\.[a-zA-Z0-9-]+/;

    # SRV target check - IP addresses not allowed.  Must be a more or less well-formed hostname.
    return ('FAIL', "SRV records cannot point directly to an IP address")
      if ${$args{host}} =~ /^(?:[\d.]+|[0-9a-fA-F:]+)$/;

    # SRV records in reverse zones get stricter treatment.  The UI bars adding them in
    # reverse record sets, but we "need" to allow editing existing ones.  And we'll allow
    # editing them if some loon manually munges one into a default reverse record set.
    if ($args{defrec} eq 'n') {
      # Get the revzone, so we can see if ${$args{val}} is in that zone
      my $revzone = new NetAddr::IP $self->revName($args{id}, 'y');
      return ('FAIL', $errstr) if !$self->_inrev($args{val}, $revzone);
    }

##enhance or ##fixme
# convert well-formed .arpa names to IP addresses to match old "strict" validation design
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{val}}, ${$args{rectype}}, $args{defrec}, $args{revrec}) &&
      ${$args{val}} =~ /\.arpa$/;

##enhance:  Look up the passed value to see if it exists.  Ooo, fancy.
    return ('FAIL', $errstr) if ! _check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});

  }

  return ('OK','OK');
} # done SRV record

# Now the custom types

# A+PTR record.  With a very little bit of magic we can also use this sub to validate AAAA+PTR.  Whee!
sub _validate_65280 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  my $code = 'OK';
  my $msg = 'OK';

  if ($args{defrec} eq 'n') {
    # live record;  revrec determines whether we validate the PTR or A component first.

    # Fail early on non-IP gibberish in ${$args{val}}.  Arguably .arpa names might be acceptable
    # but that gets stupid in forward zones, since these records are shared.
    return ('FAIL', "$typemap{${$args{rectype}}} record must be a valid IPv4 address")
      if ${$args{rectype}} == 65280 && ${$args{val}} !~ m{^\d+\.\d+\.\d+\.\d+(?:/\d+)?$};
    return ('FAIL', "$typemap{${$args{rectype}}} record must be a valid IPv6 address")
      if ${$args{rectype}} == 65281 && ${$args{val}} !~ m{^[a-fA-F0-9:]+(?:/\d+)?$};
    # If things are not OK, this should prevent Stupid in the error log.
    $args{addr} = new NetAddr::IP ${$args{val}}
      or return ('FAIL', "$typemap{${$args{rectype}}} record must be a valid IPv".
			(${$args{rectype}} == 65280 ? '4' : '6')." address");
    ${$args{val}} = $args{addr}->addr;

    if ($args{revrec} eq 'y') {
      ($code,$msg) = $self->_validate_12(%args);
      return ($code,$msg) if $code eq 'FAIL';

      # check A+PTR is really v4
      return ('FAIL',"$typemap{${$args{rectype}}} record must be a valid IPv4 address")
	if ${$args{rectype}} == 65280 && $args{addr}->{isv6};
      # check AAAA+PTR is really v6
      return ('FAIL',"$typemap{${$args{rectype}}} record must be a valid IPv6 address")
	if ${$args{rectype}} == 65281 && !$args{addr}->{isv6};

      # Check if the reqested domain exists.  If not, coerce the type down to PTR and warn.
      if (!(${$args{domid}} = $self->_hostparent(${$args{host}}))) {
	my $addmsg = "Record ".($args{update} ? 'updated' : 'added').
		" as PTR instead of $typemap{${$args{rectype}}};  domain not found for ${$args{host}}";
	$msg .= "\n$addmsg" if $code eq 'WARN';
	$msg = $addmsg if $code eq 'OK';
        ${$args{rectype}} = $reverse_typemap{PTR};
        return ('WARN', $msg);
      }

      # Add domain ID to field list and values
      ${$args{fields}} .= "domain_id,";
      push @{$args{vallist}}, ${$args{domid}};

    } else {
      ($code,$msg) = $self->_validate_1(%args) if ${$args{rectype}} == 65280;
      ($code,$msg) = $self->_validate_28(%args) if ${$args{rectype}} == 65281;
      return ($code,$msg) if $code eq 'FAIL';

      # Check if the requested reverse zone exists - note, an IP fragment won't
      # work here since we don't *know* which parent to put it in.
      # ${$args{val}} has been validated as a valid IP by now, in one of the above calls.
      my ($revid) = $dbh->selectrow_array("SELECT rdns_id FROM revzones WHERE revnet >> ?".
	" ORDER BY masklen(revnet) DESC", undef, (${$args{val}}));
      if (!$revid) {
        $msg = "Record ".($args{update} ? 'updated' : 'added')." as ".(${$args{rectype}} == 65280 ? 'A' : 'AAAA').
		" instead of $typemap{${$args{rectype}}};  reverse zone not found for ${$args{val}}";
	${$args{rectype}} = (${$args{rectype}} == 65280 ? $reverse_typemap{A} : $reverse_typemap{AAAA});
	return ('WARN', $msg);
      }

      # Check for duplicate PTRs.  Note we don't have to play games with $code and $msg, because
      # by definition there can't be duplicate PTRs if the reverse zone isn't managed here.
      if ($args{update}) {
	# Record update.  There should usually be an existing PTR (the record being updated)
	my @ptrs = @{ $dbh->selectcol_arrayref("SELECT record_id FROM "._rectable($args{defrec},$args{revrec}).
		" WHERE val = ?", undef, (${$args{val}})) };
	if (@ptrs && (!grep /^$args{update}$/, @ptrs)) {
	  $msg = "PTR record for ${$args{val}} already exists;  adding another will probably not do what you want";
	  $code = 'WARN';
	}
      } else {
	# New record.  Always warn if a PTR exists
	my ($ptrcount) = $dbh->selectrow_array("SELECT count(*) FROM "._rectable($args{defrec},$args{revrec}).
		" WHERE val = ?", undef, (${$args{val}}));
	$msg = "PTR record for ${$args{val}} already exists;  adding another will probably not do what you want"
		if $ptrcount;
	$code = 'WARN' if $ptrcount;
      }

      # Add the reverse zone ID to the fieldlist
      ${$args{fields}} .= "rdns_id,";
      push @{$args{vallist}}, $revid;

      # Coerce the hostname back to the domain;  this is so it displays and manipulates
      # sanely in the reverse zone.
      if (${$args{host}} eq '@') {
        ${$args{host}} = $self->domainName($args{id});  # errors?  What errors?
      }
    } # revrec ne 'y'

  } else {	# defrec eq 'y'

    if ($args{revrec} eq 'y') {
      ($code,$msg) = $self->_validate_12(%args);
      return ($code,$msg) if $code eq 'FAIL';
      if (${$args{rectype}} == 65280) {
	return ('FAIL',"A+PTR record must be a valid IPv4 address or fragment")
		if ${$args{val}} =~ /:/;
	${$args{val}} =~ s/^ZONE,/ZONE./;       # Clean up after uncertain IP-fragment-type from _validate_12
      } elsif (${$args{rectype}} == 65281) {
	return ('FAIL',"AAAA+PTR record must be a valid IPv6 address or fragment")
		if ${$args{val}} =~ /\./;
	${$args{val}} =~ s/^ZONE,/ZONE::/;      # Clean up after uncertain IP-fragment-type from _validate_12
      }
    } else {
      # This is easy.  I also can't see a real use-case for A/AAAA+PTR in *all* forward
      # domains, since you wouldn't be able to substitute both domain and reverse zone
      # sanely, and you'd end up with guaranteed over-replicated PTR records that would
      # confuse the hell out of pretty much anything that uses them.
##fixme: make this a config flag?
      return ('FAIL', "$typemap{${$args{rectype}}} records not allowed in default domains");
    }
  }

  return ($code, $msg);
} # done A+PTR record

# AAAA+PTR record
# A+PTR above has been magicked to handle AAAA+PTR as well.
sub _validate_65281 {
  return _validate_65280(@_);
} # done AAAA+PTR record

# PTR template record
sub _validate_65282 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # we're *this* >.< close to being able to just call _validate_12... unfortunately we can't, quite.
  if ($args{revrec} eq 'y') {
    if ($args{defrec} eq 'n') {
      return ('FAIL', "Template block ${$args{val}} is not within ".$self->revName($args{id}))
	unless $self->_ipparent($args{defrec}, $args{revrec}, $args{val}, $args{id}, \$args{addr});
##fixme:  warn if $args{val} is not /31 or larger block?
      ${$args{val}} = "$args{addr}";
    } else {
      if (${$args{val}} =~ /\./) {
	# looks like a v4 or fragment
	if (${$args{val}} =~ m|^\d+\.\d+\.\d+\.\d+(?:/\d+)?$|) {
	  # woo!  a complete IP!  validate it and normalize, or fail.
	  $args{addr} = NetAddr::IP->new(${$args{val}})
		or return ('FAIL', "IP/value looks like IPv4 but isn't valid");
	  ${$args{val}} = "$args{addr}";
	} else {
	  ${$args{val}} =~ s/^\.*/ZONE./ unless ${$args{val}} =~ /^ZONE/;
	}
      } elsif (${$args{val}} =~ /[a-f:]/) {
	# looks like a v6 or fragment
	${$args{val}} =~ s/^:*/ZONE::/ if !$args{addr} && ${$args{val}} !~ /^ZONE/;
	if ($args{addr}) {
	  if ($args{addr}->addr =~ /^0/) {
	    ${$args{val}} =~ s/^:*/ZONE::/ unless ${$args{val}} =~ /^ZONE/;
	  } else {
	    ${$args{val}} = "$args{addr}";
	  }
	}
      } else {
	# bare number (probably).  These could be v4 or v6, so we'll
	# expand on these on creation of a reverse zone.
	${$args{val}} = "ZONE,${$args{val}}" unless ${$args{val}} =~ /^ZONE/;
      }
    }
##fixme:  validate %-patterns?

# Unlike single PTR records, there is absolutely no way to sanely support multiple
# PTR templates for the same block, since they expect to expand to all the individual
# IPs on export.  Nested templates should be supported though.

    my @checkvals = (${$args{val}});
    if (${$args{val}} =~ /,/) {
      # push . and :: variants into checkvals if val has ,
      my $tmp;
      ($tmp = ${$args{val}}) =~ s/,/./;
      push @checkvals, $tmp;
      ($tmp = ${$args{val}}) =~ s/,/::/;
      push @checkvals, $tmp;
    }
##fixme:  this feels wrong still - need to restrict template pseudorecords to One Of Each
# Per Netblock such that they don't conflict on export
    my $typeck;
# type 65282 -> ptr template -> look for any of 65282, 65283, 65284
    $typeck = 'type=65283 OR type=65284' if ${$args{rectype}} == 65282;
# type 65283 -> a+ptr template -> v4 -> look for 65282 or 65283
    $typeck = 'type=65283' if ${$args{rectype}} == 65282;
# type 65284 -> aaaa+ptr template -> v6 -> look for 65282 or 65284
    $typeck = 'type=65284' if ${$args{rectype}} == 65282;
    my $pcsth = $dbh->prepare("SELECT count(*) FROM "._rectable($args{defrec},$args{revrec})." WHERE val = ? ".
	"AND (type=65282 OR $typeck)");
    foreach my $checkme (@checkvals) {
      $pcsth->execute($checkme);
      my ($rc) = $pcsth->fetchrow_array;
      return ('FAIL', "Only one template pseudorecord may exist for a given IP block") if $rc > 1;
    }

  } else {
    return ('FAIL', "Forward zones cannot contain PTR records");
  }

  return ('OK','OK');
} # done PTR template record

# A+PTR template record
sub _validate_65283 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  my ($code,$msg) = ('OK','OK');

##fixme:  need to fiddle things since A+PTR templates are acceptable in live
# forward zones but not default records
  if ($args{defrec} eq 'n') {
    if ($args{revrec} eq 'n') {

      # Coerce all hostnames to end in ".DOMAIN" for group/default records,
      # or the intended parent domain for live records.
      my $pname = $self->domainName($args{id});
      ${$args{host}} =~ s/\.*$/\.$pname/ if ${$args{host}} !~ /$pname$/i;

      # check for form;  note this checks both normal and "other" hostnames.
      return ('FAIL', $errstr)
	if !_check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});

      # Check if the requested reverse zone exists - note, an IP fragment won't
      # work here since we don't *know* which parent to put it in.
      # ${$args{val}} has been validated as a valid IP by now, in one of the above calls.
      my ($revid) = $dbh->selectrow_array("SELECT rdns_id FROM revzones WHERE revnet >> ?".
	" ORDER BY masklen(revnet) DESC", undef, (${$args{val}}));
      # Fail if no match;  we can't coerce a PTR-template type down to not include the PTR bit currently.
      if (!$revid) {
        $msg = "Can't ".($args{update} ? 'update' : 'add')." ${$args{host}}/${$args{val}} as ".
		"$typemap{${$args{rectype}}}:  reverse zone not found for ${$args{val}}";
##fixme:  add A template, AAAA template types?
#	${$args{rectype}} = (${$args{rectype}} == 65280 ? $reverse_typemap{A} : $reverse_typemap{AAAA});
	return ('FAIL', $msg);
      }

      # Add reverse zone ID to field list and values
      ${$args{fields}} .= "rdns_id,";
      push @{$args{vallist}}, $revid;

    } else {
      return ('FAIL', "IP or IP fragment ${$args{val}} is not within ".$self->revName($args{id}))
	unless $self->_ipparent($args{defrec}, $args{revrec}, $args{val}, $args{id}, \$args{addr});
      ${$args{val}} = "$args{addr}";

      if (!(${$args{domid}} = $self->_hostparent(${$args{host}}))) {
	my $addmsg = "Record ".($args{update} ? 'updated' : 'added').
		" as PTR template instead of $typemap{${$args{rectype}}};  domain not found for ${$args{host}}";
	$msg .= "\n$addmsg" if $code eq 'WARN';
	$msg = $addmsg if $code eq 'OK';
        ${$args{rectype}} = 65282;
        return ('WARN', $msg);
      }

      # Add domain ID to field list and values
      ${$args{fields}} .= "domain_id,";
      push @{$args{vallist}}, ${$args{domid}};
    }

  } else {
    my ($code,$msg) = $self->_validate_65282(%args);
    return ($code, $msg) if $code eq 'FAIL';
    # get domain, check against ${$args{name}}
  }

  return ('OK','OK');
} # done A+PTR template record

# AAAA+PTR template record
# Not sure this can be handled sanely due to the size of IPv6 address space
# _validate_65283 above should handle v6 template records fine.  It's on export we've got trouble.
sub _validate_65284 {
  my $self = shift;
  my %args = @_;

  # do a quick check on the form of the hostname part;  this is effectively a
  # "*.0.0.f.ip6.arpa" hostname, not an actual expandable IP template pattern
  # like with 65283.
  return ('FAIL', $errstr)
	if !_check_hostname_form(${$args{host}}, ${$args{rectype}}, $args{defrec}, $args{revrec});
  return $self->_validate_65283(%args);
} # done AAAA+PTR template record

# Delegation record
# This is essentially a specialized clone of the NS record, primarily useful
# for delegating IPv4 sub-/24 reverse blocks
sub _validate_65285 {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

# Almost, but not quite, identical to NS record validation.

  # Check that the target of the record is within the parent.
  # Yes, host<->val are mixed up here;  can't see a way to avoid it.  :(
  if ($args{defrec} eq 'n') {
    # Check if IP/address/zone/"subzone" is within the parent
    if ($args{revrec} eq 'y') {
      my $tmpip = NetAddr::IP->new(${$args{val}});
      my $pname = $self->revName($args{id});
      return ('FAIL',"${$args{val}} not within $pname")
	 unless $self->_ipparent($args{defrec}, $args{revrec}, $args{val}, $args{id}, \$tmpip);
      # Normalize
      ${$args{val}} = "$tmpip";
    } else {
      my $pname = $self->domainName($args{id});
      ${$args{host}} =~ s/\.*$/\.$pname/ if ${$args{host}} !~ /$pname$/i;
    }
  } else {
    return ('FAIL',"Delegation records are not permitted in default record sets");
  }
  return ('OK','OK');
}

# Subs not specific to a particular record type

# Convert $$host and/or $$val to lowercase as appropriate.
# Should only be called if $self->{lowercase} is true.
# $rectype is also a reference for caller convenience
sub _caseclean {
  my ($rectype, $host, $val, $defrec, $revrec) = @_;

  # Can't case-squash default records, due to DOMAIN, ZONE, and ADMINDOMAIN templating
  return if $defrec eq 'y';

  if ($typemap{$$rectype} eq 'TXT' || $typemap{$$rectype} eq 'SPF') {
    # TXT records should preserve user entry in the string.
    # SPF records are a duplicate of TXT with a new record type value (99)
    $$host = lc($$host) if $revrec eq 'n';	# only lowercase $$host on live forward TXT;  preserve TXT content
    $$val = lc($$val) if $revrec eq 'y';	# only lowercase $$val on live reverse TXT;  preserve TXT content
  } else {
    # Non-TXT, live records, are fully case-insensitive
    $$host = lc($$host);
    $$val = lc($$val);
  } # $typemap{$$rectype} else

} # _caseclean()


##
## Record data substitution subs
##

# Replace ZONE in hostname, or create (most of) the actual proper zone name
sub _ZONE {
  my $zone = shift;
  my $string = shift;
  my $fr = shift || 'f';	# flag for forward/reverse order?  nb: ignored for IP
  my $sep = shift || '-';	# Separator character - unlikely we'll ever need more than . or -

  my $prefix;

  $string =~ s/,/./ if !$zone->{isv6};
  $string =~ s/,/::/ if $zone->{isv6};

  # Subbing ZONE in the host.  We need to properly ID the netblock range
  # The subbed text should have "network IP with trailing zeros stripped" for
  # blocks lined up on octet (for v4) or hex-quad (for v6) boundaries
  # For blocks that do NOT line up on these boundaries, we take the most
  # significant octet or 16-bit chunk of the "broadcast" IP and append it
  # after a double-dash
  # ie:
  # 8.0.0.0/6 -> 8.0.0.0 -> 11.255.255.255;  sub should be 8--11
  # 10.0.0.0/12 -> 10.0.0.0 -> 10.0.0.0 -> 10.15.255.255;  sub should be 10-0--15
  # 192.168.4.0/22 -> 192.168.4.0 -> 192.168.7.255;  sub should be 192-168-4--7
  # 192.168.0.8/29 -> 192.168.0.8 -> 192.168.0.15;  sub should be 192-168-0-8--15
  # Similar for v6

  if (!$zone->{isv6}) { # IPv4

    $prefix = $zone->network->addr;	# Just In Case someone managed to slip in
					# a funky subnet that had host bits set.
    my $bc = $zone->broadcast->addr;

    if ($zone->masklen > 24) {
      $bc =~ s/^\d+\.\d+\.\d+\.//;
    } elsif ($zone->masklen > 16) {
      $prefix =~ s/\.0$//;
      $bc =~ s/^\d+\.\d+\.//;
    } elsif ($zone->masklen > 8) {
      $bc =~ s/^\d+\.//;
      $prefix =~ s/\.0\.0$//;
    } else {
      $prefix =~ s/\.0\.0\.0$//;
    }
    if ($zone->masklen % 8) {
      $bc =~ s/(\.255)+$//;
      $prefix .= "--$bc";	#"--".zone->masklen;	# use range or mask length?
    }
    if ($fr eq 'f') {
      $prefix =~ s/\.+/$sep/g;
    } else {
      $prefix = join($sep, reverse(split(/\./, $prefix)));
    }

  } else { # IPv6

    if ($fr eq 'f') {

      $prefix = $zone->network->addr;	# Just In Case someone managed to slip in
					# a funky subnet that had host bits set.
      my $bc = $zone->broadcast->addr;
      if (($zone->masklen % 16) != 0) {
        # Strip trailing :0 off $prefix, and :ffff off the broadcast IP
        for (my $i=0; $i<(7-int($zone->masklen / 16)); $i++) {
          $prefix =~ s/:0$//;
          $bc =~ s/:ffff$//;
        }
        # Strip the leading 16-bit chunks off the front of the broadcast IP
        $bc =~ s/^([a-f0-9]+:)+//;
        # Append the remaining 16-bit chunk to the prefix after "--"
        $prefix .= "--$bc";
      } else {
        # Strip off :0 from the end until we reach the netblock length.
        for (my $i=0; $i<(8-$zone->masklen / 16); $i++) {
	  $prefix =~ s/:0$//;
        }
      }
      # Actually deal with the separator
      $prefix =~ s/:/$sep/g;

    } else {	# $fr eq 'f'

      $prefix = $zone->network->full;	# Just In Case someone managed to slip in
					# a funky subnet that had host bits set.
      my $bc = $zone->broadcast->full;
      $prefix =~ s/://g;	# clean these out since they're not spaced right for this case
      $bc =~ s/://g;
      # Strip trailing 0 off $prefix, and f off the broadcast IP, to match the mask length
      for (my $i=0; $i<(31-int($zone->masklen / 4)); $i++) {
        $prefix =~ s/0$//;
        $bc =~ s/f$//;
      }
      # Split and reverse the order of the nibbles in the network/broadcast IPs
      # trim another 0 for nibble-aligned blocks first, but only if we really have a block, not an IP
      $prefix =~ s/0$// if $zone->masklen % 4 == 0 && $zone->masklen != 128;
      my @nbits = reverse split //, $prefix;
      my @bbits = reverse split //, $bc;
      # Handle the sub-nibble case.  Eww.  I feel dirty supporting this...
      $nbits[0] = "$nbits[0]-$bbits[0]" if ($zone->masklen % 4) != 0;
      # Glue it back together
      $prefix = join($sep, @nbits);

    }	# $fr ne 'f'

  } # $zone->{isv6}

  # Do the substitution, finally
  $string =~ s/ZONE/$prefix/;
  $string =~ s/--/-/ if $sep ne '-';	# - as separator needs extra help for sub-octet v4 netblocks
  return $string;
} # done _ZONE()

# Not quite a substitution sub, but placed here as it's basically the inverse of above;
# given the .arpa zone name, return the CIDR netblock the zone is for.
# Supports v4 non-octet/non-classful netblocks as per the method outlined in the Grasshopper Book (2nd Ed p217-218)
# Does NOT support non-quad v6 netblocks via the same scheme;  it shouldn't ever be necessary.
# Takes a nominal .arpa zone name, returns a success code and NetAddr::IP, or a fail code and message
sub _zone2cidr {
  my $zone = shift;

  my $cidr;
  my $tmpcidr;
  my $warnmsg = '';

  if ($zone =~ /\.in-addr\.arpa\.?$/) {
    # v4 revzone, formal zone name type
    my $tmpzone = $zone;
    return ('FAIL', "Non-numerics in apparent IPv4 reverse zone name [$tmpzone]")
        if $tmpzone !~ m{^(?:\d+[/-])?[\d\.]+\.in-addr\.arpa\.?$};
    $tmpzone =~ s/\.in-addr\.arpa\.?//;

    # Snag the octet pieces
    my @octs = split /\./, $tmpzone;

    # Map result of a range manipulation to a mask length change.  Cheaper than finding the 2-root of $octets[0]+1.
    # Note we will not support /31 blocks, mostly due to issues telling "24-31" -> .24/29 apart from
    # "24-31" -> .24/31", with a litte bit of "/31 is icky".
    my %maskmap = (  3 => 2,  7 => 3, 15 => 4, 31 => 5, 63 => 6, 127 => 7,
		    30 => 2, 29 => 3, 28 => 4, 27 => 5, 26 => 6,  25 => 7
	);

    # Handle "range" blocks, eg, 80-83.168.192.in-addr.arpa (192.168.80.0/22)
    # Need to take the size of the range to offset the basic octet-based mask length,
    # and make sure the first number in the range gets used as the network address for the block
    # Alternate form:  The second number is actually the real netmask, not the end of the range.
    my $masklen = 0;
    if ($octs[0] =~ m{^((\d+)[/-](\d+))$}) {	# take the range...
      if (24 < $3 && $3 < 31) {
	# we have a real netmask
	$masklen = -$maskmap{$3};
      } else {
	# we have a range.  NB:  only real CIDR ranges are supported
	$masklen -= $maskmap{-(eval $1)};	# find the mask base...
      }
      $octs[0] = $2;	# set the base octet of the range...
    }
    @octs = reverse @octs;	# We can reverse the octet pieces now that we've extracted and munged any ranges

# arguably we should only allow sub-octet range/mask in-addr.arpa
# specifications in the least significant octet, but the code is
# simpler if we deal with sub-octet delegations at any level.

    # Now we find the "true" mask with the aid of the "base" calculated above
    if ($#octs == 0) {
      $masklen += 8;
      $tmpcidr = "$octs[0].0.0.0/$masklen";	# really hope we don't see one of these very often.
    } elsif ($#octs == 1) {
      $masklen += 16;
      $tmpcidr = "$octs[0].$octs[1].0.0/$masklen";
    } elsif ($#octs == 2) {
      $masklen += 24;
      $tmpcidr = "$octs[0].$octs[1].$octs[2].0/$masklen";
    } else {
      $masklen += 32;
      $tmpcidr = "$octs[0].$octs[1].$octs[2].$octs[3]/$masklen";
    }

  } elsif ($zone =~ /\.ip6\.arpa\.?$/) {
    # v6 revzone, formal zone name type
    my $tmpzone = $zone;
##fixme:  if-n-when we decide we can support sub-nibble v6 zone names, we'll need to change this segment
    return ('FAIL', "Non-hexadecimals in apparent IPv6 reverse zone name [$tmpzone]")
        if $tmpzone !~ /^[a-fA-F\d\.]+\.ip6\.arpa\.?$/;
    $tmpzone =~ s/\.ip6\.arpa\.?//;
    my @quads = reverse(split(/\./, $tmpzone));
    $warnmsg .= "Apparent sub-/64 IPv6 reverse zone\n" if $#quads > 15;
    my $nc;
    foreach (@quads) {
      $tmpcidr .= $_;
      $tmpcidr .= ":" if ++$nc % 4 == 0 && $nc < $#quads;
    }
    my $nq = 1 if $nc % 4 != 0;
    my $mask = $nc * 4;	# need to do this here because we probably increment it below
    while ($nc++ % 4 != 0) {
      $tmpcidr .= "0";
    }
    # polish it off with trailing ::/mask if this is a CIDR block instead of an IP
    $tmpcidr .= "::/$mask" if $mask != 128;
  }

  # Just to be sure, use NetAddr::IP to validate.  Saves a lot of nasty regex watching for valid octet values.
  return ('FAIL', "Invalid zone $zone (apparent netblock $tmpcidr)")
	unless $cidr = NetAddr::IP->new($tmpcidr);

  if ($warnmsg) {
    $errstr = $warnmsg;
    return ('WARN', $cidr);
  }
  return ('OK', $cidr);
##fixme:  use wantarray() to decide what to return?
} # done _zone2cidr()

# Record template %-parameter expansion, IPv4.  Note that IPv6 doesn't
# really have a sane way to handle this type of expansion at the moment
# due to the size of the address space.
# Takes a reference to a template string to be expanded, an IP to use in the replacement,
# an optional netblock for the %ngb (net, gw, bcast) expansion, and an optional index
# number for %c "n'th usable IP in block/range" patterns.
# Alters the template string referred.
sub _template4_expand {
  # ugh pthui
  my $self;
  $self = shift if ref($_[0]) eq 'DNSDB';

  my $tmpl = shift;
  my $ip = shift;
  my $subnet = shift;   # for %ngb and %c
  my $ipindex = shift;  # for %c

  # blank $tmpl on config template_skip_0 or template_skip_255, unless we have a %ngb
  if ($$tmpl !~ /\%-?n-?g-?b\%/) {
    if ( ($ip =~ /\.0$/ && $self->{template_skip_0}) ||
         ($ip =~ /\.255$/ && $self->{template_skip_255}) ) {
      $$tmpl = '';
      return;
    }
  }

  my @ipparts = split /\./, $ip;
  my @iphex;
  my @ippad;
  for (@ipparts) {
    push @iphex, sprintf("%x", $_);
    push @ippad, sprintf("%0.3u", $_);
  }

  # Two or three consecutive separator characters (_ or -) should be rare - users that use them
  # anywhere other than punycoded internationalized domains get to keep the pieces when it breaks.
  # We clean up the ones that we may inadvertently generate after replacing %c and %ngb%
  my ($thrsep) = ($$tmpl =~ /[_-]{3}/);
  my ($twosep) = ($$tmpl =~ /[_-]{2}/);

  # Take the simplest path to pattern substitution;  replace only exactly the %c or %ngb%
  # patterns as-is.  Then check after to see if we've caused doubled separator characters (- or _)
  # and eliminate them, but only if the original template didn't have them already.  Also
  # unconditionally drop separator characters immediately before a dot;  these do not always
  # strictly make the label invalid but almost always, and any exceptions should never show up
  # in a template record that expands to "many" real records anyway.

  # %ngb and %c require a netblock
  if ($subnet) {
    # extract the fragments
    my ($ngb,$n,$g,$b) = ($$tmpl =~ /(\%(-?n)(-?g)(-?b)\%)/);
    my ($c) = ($$tmpl =~ /(\%-?c)/);  my $nld = '';  my $cld = '';
    $c = '' if !$c;
    my $skipgw = ($c =~ /\%-c/ ? 0 : 1);
    my $ipkill = 0;

##fixme: still have one edge case not handled well:
# %c%n-gb%
# do we drop the record as per -g, or publish the record with an index of 1 as per %c?
# arguably this is a "that's a STUPID question!" case

    if ($c) {
      # "n'th usable IP in the block" pattern.  We need the caller to provide an index
      # number otherwise we see exponential time growth because we have to iterate over
      # the whole block to map the IP back to an index.  :/
      # NetAddr::IP does not have a method for asking "what index is IP <foo> at?"

      # no index, or index == 0, (AKA network address), or IP == broadcast, blank the index fragment
      if (!$ipindex || ($$subnet->broadcast->addr eq $ip)) {
        $$tmpl =~ s/$c//;
      } else {
        # if we have %c, AKA "skip the gateway", and we're on the nominal gateway IP, blank the index fragment
        if ($skipgw && $$subnet->first->addr eq $ip) {
          $$tmpl =~ s/$c//;
        }
        # else replace the index fragment with the passed index minus $skipgw, so that we can start the
        # resulting index at 1 on net+2
        else {
          $$tmpl =~ s/$c/($ipindex-$skipgw)/e;
        }
      }
    } # if ($c)

    if ($ngb) {
      # individually check the network, standard gateway (net+1) IP, and broadcast IP
      # blank $$tmpl if n, g, or b was prefixed with - (this allows "hiding" net/gw/bcast entries)

      if ($$subnet->network->addr eq $ip) {
        if ($n eq '-n') {
          $$tmpl = '';
        } else {
          $$tmpl =~ s/$ngb/net/;
          $ipkill = 1;
        }
      } elsif ($$subnet->first->addr eq $ip) {
        if ($g eq '-g') {
          $$tmpl = '';
        } else {
          $$tmpl =~ s/$ngb/gw/;
          $ipkill = 1;
        }
      } elsif ($$subnet->broadcast->addr eq $ip) {
        if ($b eq '-b') {
          $$tmpl = '';
        } else {
          $$tmpl =~ s/$ngb/bcast/;
          $ipkill = 1;
        }
      } else {
        $$tmpl =~ s/$ngb//;
      }
    }

    # We don't (usually) want to expand the IP-related patterns on the -net, -gw, or -bcast IPs.
    # Arguably this is another place for another config knob, or possibly further extension of
    # the template pattern to control it on a per-subnet basis.
    if ($ipkill) {
      # kill common IP patterns
      $$tmpl =~ s/\%[_.-]?[irdh]//;
      # kill IP octet patterns
      $$tmpl =~ s/\%[1234][dh0](?:[_.-]\%[1234][dh0]){0,3}//;
    }

    # and now clean up to make sure we leave a valid DNS label... mostly.  Should arguably
    # split on /\./ and process each label separately.
    $$tmpl =~ s/([_-]){3}/$1/ if !$thrsep;
    $$tmpl =~ s/([_-]){2}/$1/ if !$twosep;
    $$tmpl =~ s/[_-]\././;

  } # if ($subnet)

  # IP substitutions in template records:
  #major patterns:
  #dashed IP, forward and reverse
  #underscoreed IP, forward and reverse
  #dotted IP, forward and reverse (even if forward is... dumb)
  # -> %r for reverse, %i for forward, leading -, _, or . to indicate separator, defaults to -
  # %r or %-r	=> %4d-%3d-%2d-%1d
  # %_r		=> %4d_%3d_%2d_%1d
  # %.r		=> %4d.%3d.%2d.%1d
  # %i or %-i	=> %1d-%2d-%3d-%4d
  # %_i		=> %1d_%2d_%3d_%4d
  # %.i		=> %1d.%2d.%3d.%4d
  $$tmpl =~ s/\%r/\%4d-\%3d-\%2d-\%1d/g;
  $$tmpl =~ s/\%([-._])r/\%4d$1\%3d$1\%2d$1\%1d/g;
  $$tmpl =~ s/\%i/\%1d-\%2d-\%3d-\%4d/g;
  $$tmpl =~ s/\%([-._])i/\%1d$1\%2d$1\%3d$1\%4d/g;

  #hex-coded IP
  # %h
  $$tmpl =~ s/\%h/$iphex[0]$iphex[1]$iphex[2]$iphex[3]/g;

  #IP as decimal-coded 32-bit value
  # %d
  my $iptmp = $ipparts[0]*256*256*256 + $ipparts[1]*256*256 + $ipparts[2]*256 + $ipparts[3];
  $$tmpl =~ s/\%d/$iptmp/g;

  #minor patterns (per-octet)
  # %[1234][dh0]
  #octet
  #hex-coded octet
  #0-padded octet
  $$tmpl =~ s/\%([1234])d/$ipparts[$1-1]/g;
  $$tmpl =~ s/\%([1234])h/$iphex[$1-1]/g;
  $$tmpl =~ s/\%([1234])0/$ippad[$1-1]/g;

} # _template4_expand()

# Broad syntactic check on the hostname.  Checks for valid characters, correctly-expandable template patterns.
# Takes the hostname, type, and live/default and forward/reverse flags
# Returns true/false, sets errstr on failures
sub _check_hostname_form {
  my ($hname,$rectype,$defrec,$revrec) = @_;

  if ($hname =~ /\%/ && ($rectype == 65282 || $rectype == 65283) ) {
    my $tmphost = $hname;
    # we don't actually need to test with the real IP passed;  that saves a bit of fiddling.
    DNSDB::_template4_expand(\$tmphost, '10.10.10.10');
    if ($tmphost =~ /\%/ || lc($tmphost) !~ /^(?:\*\.)?(?:[0-9a-z_.-]+)$/) {
      $errstr = "Invalid template $hname";
      return;
    }
  } elsif ($rectype == $reverse_typemap{CNAME} && $revrec eq 'y') {
    # Allow / in reverse CNAME hostnames for sub-/24 delegation
    if (lc($hname) !~ m|^[0-9a-z_./-]+$|) {
      # error message is deliberately restrictive;  special cases are SPECIAL and not for general use
      $errstr = "Hostnames may not contain anything other than (0-9 a-z . _)";
      return;
    }
  } elsif ($revrec eq 'y') {
    # Reverse zones don't support @ in hostnames
    if (lc($hname) !~ /^(?:\*\.)?[0-9a-z_.-]+$/) {
      # error message is deliberately restrictive;  special cases are SPECIAL and not for general use
      $errstr = "Hostnames may not contain anything other than (0-9 a-z . _)";
      return;
    }
  } else {
    if (lc($hname) !~ /^(?:\*\.)?(?:[0-9a-z_.-]+|@)$/) {
      # Don't mention @, because it would be far too wordy to explain the nuance of @
      $errstr = "Hostnames may not contain anything other than (0-9 a-z . _)";
      return;
    }
  }
  return 1;
} # _check_hostname_form()


##
## Initialization and cleanup subs
##

## DNSDB::__cfgload()
# Private sub to parse a config file and load it into %config
# Takes a filename and a hashref to put the parsed entries in
sub __cfgload {
  $errstr = '';
  my $cfgfile = shift;
  my $cfg = shift;

  if (open CFG, "<$cfgfile") {
    while (<CFG>) {
      chomp;
      s/^\s*//;
      next if /^#/;
      next if /^$/;
# hmm.  more complex bits in this file might require [heading] headers, maybe?
#    $mode = $1 if /^\[(a-z)+]/;
    # DB connect info
      $cfg->{dbname}	= $1 if /^dbname\s*=\s*([a-z0-9_.-]+)/i;
      $cfg->{dbuser}	= $1 if /^dbuser\s*=\s*([a-z0-9_.-]+)/i;
      $cfg->{dbpass}	= $1 if /^dbpass\s*=\s*([a-z0-9_.-]+)/i;
      $cfg->{dbhost}	= $1 if /^dbhost\s*=\s*([a-z0-9_.-]+)/i;
      # Mail settings
      $cfg->{mailhost}		= $1 if /^mailhost\s*=\s*([a-z0-9_.-]+)/i;
      $cfg->{mailnotify}	= $1 if /^mailnotify\s*=\s*([a-z0-9_.\@-]+)/i;
      $cfg->{mailsender}	= $1 if /^mailsender\s*=\s*([a-z0-9_.\@-]+)/i;
      $cfg->{mailname}		= $1 if /^mailname\s*=\s*([a-z0-9\s_.-]+)/i;
      $cfg->{orgname}		= $1 if /^orgname\s*=\s*([a-z0-9\s_.,'-]+)/i;
      $cfg->{domain}		= $1 if /^domain\s*=\s*([a-z0-9_.-]+)/i;
      # session - note this is fed directly to CGI::Session
      $cfg->{timeout}		= $1 if /^[tT][iI][mM][eE][oO][uU][tT]\s*=\s*(\d+[smhdwMy]?)/;
      $cfg->{sessiondir}	= $1 if m{^sessiondir\s*=\s*([a-z0-9/_.-]+)}i;
      # misc
      $cfg->{log_failures}	= $1 if /^log_failures\s*=\s*([a-z01]+)/i;
      $cfg->{perpage}		= $1 if /^perpage\s*=\s*(\d+)/i;
      $cfg->{exportcache}	= $1 if m{^exportcache\s*=\s*([a-z0-9/_.-]+)}i;
      $cfg->{usecache}		= $1 if m{^usecache\s*=\s*([a-z01]+)}i;
      $cfg->{force_refresh}	= $1 if /^force_refresh\s*=\s*([a-z01]+)/i;
      $cfg->{lowercase}		= $1 if /^lowercase\s*=\s*([a-z01]+)/i;
      $cfg->{showrev_arpa}	= $1 if /^showrev_arpa\s*=\s*([a-z]+)/i;
      $cfg->{template_skip_0}	= $1 if /^template_skip_0\s*=\s*([a-z01]+)/i;
      $cfg->{template_skip_255}	= $1 if /^template_skip_255\s*=\s*([a-z01]+)/i;
      $cfg->{autotxt}		= $1 if /^force_refresh\s*=\s*([a-z01]+)/i;
# not supported in dns.cgi yet
#      $cfg->{templatedir}	= $1 if m{^templatedir\s*=\s*([a-z0-9/_.-]+)}i;
#      $cfg->{templateoverride}	= $1 if m{^templateoverride\s*=\s*([a-z0-9/_.-]+)}i;
      # RPC options
      $cfg->{rpcmode}		= $1 if /^rpc_mode\s*=\s*(socket|HTTP|XMLRPC)\s*$/i;
      $cfg->{maxfcgi}		= $1 if /^max_fcgi_requests\s*=\s*(\d+)\s*$/i;
      if (my ($tmp) = /^rpc_iplist\s*=\s*(.+)/i) {
        my @ips = split /[,\s]+/, $tmp;
        my $rpcsys = shift @ips;
        push @{$cfg->{rpcacl}{$rpcsys}}, @ips;
      }
    }
    close CFG;
  } else {
    $errstr = "Couldn't load configuration file $cfgfile: $!";
    return;
  }
  return 1;
} # end __cfgload()


## DNSDB::connectDB()
# Creates connection to DNS database.
# Requires the database name, username, and password.
# Returns a handle to the db or undef on failure.
# Set up for a PostgreSQL db;  could be any transactional DBMS with the
# right changes.
# Called by new();  not intended to be called publicly.
sub connectDB {
  $errstr = '';
  my $dbname = shift;
  my $user = shift;
  my $pass = shift;
  my $dbh;
  my $DSN = "DBI:Pg:dbname=$dbname";

  my $host = shift;
  $DSN .= ";host=$host" if $host;

# Note that we want to autocommit by default, and we will turn it off locally as necessary.
# We may not want to print gobbledygook errors;  YMMV.  Have to ponder that further.
  $dbh = DBI->connect($DSN, $user, $pass, {
	AutoCommit => 1,
	PrintError => 0
	});
  if (!$dbh) {
    $errstr = $DBI::errstr;
    return;
  }
#) if(!$dbh);

  local $dbh->{RaiseError} = 1;

  eval {
##fixme:  initialize the DB if we can't find the table (since, by definition, there's
# nothing there if we can't select from it...)
    my $tblsth = $dbh->prepare("SELECT count(*) FROM pg_catalog.pg_class WHERE relkind='r' AND relname=?");
    my ($tblcount) = $dbh->selectrow_array($tblsth, undef, ('misc'));
#  return (undef,$DBI::errstr) if $dbh->err;

#if ($tblcount == 0) {
#  # create tables one at a time, checking for each.
#  return (undef, "check table misc missing");
#}

# Return here if we can't select.
# This should retrieve the dbversion key.
    my $sth = $dbh->prepare("SELECT key,value FROM misc WHERE misc_id=1");
    $sth->execute();
#  return (undef,$DBI::errstr) if ($sth->err);

##fixme:  do stuff to the DB on version mismatch
# x.y series should upgrade on $DNSDB::VERSION > misc(key=>version)
# DB should be downward-compatible;  column defaults should give sane (if possibly
# useless-and-needs-help) values in columns an older software stack doesn't know about.

# See if the select returned anything (or null data).  This should
# succeed if the select executed, but...
    $sth->fetchrow();
#  return (undef,$DBI::errstr)  if ($sth->err);

    $sth->finish;

  }; # wrapped DB checks
  if ($@) {
    $errstr = $@;
    return;
  }

# If we get here, we should be OK.
  return $dbh;
} # end connectDB


## DNSDB::finish()
# Cleans up after database handles and so on.
# Requires a database handle
sub finish {
  my $self = shift;
  $self->{dbh}->disconnect;
} # end finish


## DNSDB::initGlobals()
# Initialize global variables
# NB: this does NOT include web-specific session variables!
sub initGlobals {
  my $self = shift;
  my $dbh = $self->{dbh};

# load record types from database
  my $sth = $dbh->prepare("SELECT val,name,stdflag FROM rectypes");
  $sth->execute;
  while (my ($recval,$recname,$stdflag) = $sth->fetchrow_array()) {
    $typemap{$recval} = $recname;
    $reverse_typemap{$recname} = $recval;
    # now we fill the record validation function hash
    if ($stdflag < 5) {
      my $fn = "_validate_$recval";
      $validators{$recval} = \&$fn;
    } else {
      my $fn = "sub { return ('FAIL','Type $recval ($recname) not supported'); }";
      $validators{$recval} = eval $fn;
    }
  }
} # end initGlobals


## DNSDB::initRPC()
# Takes a remote username and remote fullname.
# Sets up the RPC logging-pseudouser if needed.
# Sets the %userdata hash for logging.
# Returns undef on failure
sub initRPC {
  my $self = shift;
  my $dbh = $self->{dbh};
  my %args  = @_;

  return if !$args{username};
  return if !$args{fullname};

  $args{username} = "$args{username}/$args{rpcsys}";

  my $tmpuser = $dbh->selectrow_hashref("SELECT username,user_id AS userid,group_id,firstname,lastname,status".
	" FROM users WHERE username=?", undef, ($args{username}) );
  if (!$tmpuser) {
    $dbh->do("INSERT INTO users (username,password,firstname,type) VALUES (?,'RPC',?,'R')", undef,
	($args{username}, $args{fullname}) );
    $tmpuser = $dbh->selectrow_hashref("SELECT username,user_id AS userid,group_id,firstname,lastname,status".
	" FROM users WHERE username=?", undef, ($args{username}) );
  }
  $tmpuser->{lastname} = '' if !$tmpuser->{lastname};
  $self->{loguserid} = $tmpuser->{userid};
  $self->{logusername} = $tmpuser->{username};
  $self->{logfullname} = "$tmpuser->{firstname} $tmpuser->{lastname} ($args{rpcsys})";
  return 1 if $tmpuser;
} # end initRPC()


## DNSDB::login()
# Takes a database handle, username and password
# Returns a userdata hash (UID, GID, username, fullname parts) if username exists,
# password matches the one on file, and account is not disabled
# Returns undef otherwise
sub login {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $user = shift;
  my $pass = shift;

  my $userinfo = $dbh->selectrow_hashref("SELECT user_id,group_id,password,firstname,lastname,status".
	" FROM users WHERE username=?",
	undef, ($user) );
  return if !$userinfo;
  return if !$userinfo->{status};

  if ($userinfo->{password} =~ m|^\$1\$([A-Za-z0-9/.]+)\$|) {
    # native passwords (crypt-md5)
    return if $userinfo->{password} ne unix_md5_crypt($pass,$1);
  } elsif ($userinfo->{password} =~ /^[0-9a-f]{32}$/) {
    # VegaDNS import (hex-coded MD5)
    return if $userinfo->{password} ne md5_hex($pass);
  } else {
    # plaintext (convenient now and then)
    return if $userinfo->{password} ne $pass;
  }

  return $userinfo;
} # end login()


## DNSDB::initActionLog()
# Set up action logging.  Takes a database handle and user ID
# Sets some internal globals and Does The Right Thing to set up a logging channel.
# This sets up _log() to spew out log entries to the defined channel without worrying
# about having to open a file or a syslog channel
##fixme Need to call _initActionLog_blah() for various logging channels, configured
# via dnsdb.conf, in $self->{log_channel} or something
# See https://secure.deepnet.cx/trac/dnsadmin/ticket/21
sub initActionLog {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $uid = shift;

  return if !$uid;

  # snag user info for logging.  there's got to be a way to not have to pass this back
  # and forth from a caller, but web usage means no persistence we can rely on from
  # the server side.
  my ($username,$fullname) = $dbh->selectrow_array("SELECT username, firstname || ' ' || lastname".
	" FROM users WHERE user_id=?", undef, ($uid));
##fixme: errors are unpossible!

  $self->{logusername} = $username;
  $self->{loguserid} = $uid;
  $self->{logfullname} = $fullname;

  # convert to real check once we have other logging channels
  # if ($self->{log_channel} eq 'sql') {
  #   Open Log, Sez Me!
  # }

} # end initActionLog


## DNSDB::getPermissions()
# Get permissions from DB
# Requires DB handle, group or user flag, ID, and hashref.
sub getPermissions {
  my $self = shift;
  my $dbh = $self->{dbh};

  my $type = shift;
  my $id = shift;
  my $hash = shift;

  my $sql = qq(
	SELECT
	p.admin,p.self_edit,
	p.group_create,p.group_edit,p.group_delete,
	p.user_create,p.user_edit,p.user_delete,
	p.domain_create,p.domain_edit,p.domain_delete,
	p.record_create,p.record_edit,p.record_delete,p.record_locchg,
	p.location_create,p.location_edit,p.location_delete,p.location_view
	FROM permissions p
	);
  if ($type eq 'group') {
    $sql .= qq(
	JOIN groups g ON g.permission_id=p.permission_id
	WHERE g.group_id=?
	);
  } else {
    $sql .= qq(
	JOIN users u ON u.permission_id=p.permission_id
	WHERE u.user_id=?
	);
  }

  my $sth = $dbh->prepare($sql);

##fixme?  we don't trap other plain SELECT errors
  $sth->execute($id);

#  my $permref = $sth->fetchrow_hashref;
#  return $permref;
#  $hash = $permref;
# Eww.  Need to learn how to forcibly drop a hashref onto an existing hash.
  ($hash->{admin},$hash->{self_edit},
	$hash->{group_create},$hash->{group_edit},$hash->{group_delete},
	$hash->{user_create},$hash->{user_edit},$hash->{user_delete},
	$hash->{domain_create},$hash->{domain_edit},$hash->{domain_delete},
	$hash->{record_create},$hash->{record_edit},$hash->{record_delete},$hash->{record_locchg},
	$hash->{location_create},$hash->{location_edit},$hash->{location_delete},$hash->{location_view}
	) = $sth->fetchrow_array;

} # end getPermissions()


## DNSDB::changePermissions()
# Update an ACL entry
# Takes a db handle, type, owner-id, and hashref for the changed permissions.
sub changePermissions {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $type = shift;
  my $id = shift;
  my $newperms = shift;
  my $inherit = shift || 0;

  my $resultmsg = '';

  # see if we're switching from inherited to custom.  for bonus points,
  # snag the permid and parent permid anyway, since we'll need the permid
  # to set/alter custom perms, and both if we're switching from custom to
  # inherited.
  my $sth = $dbh->prepare("SELECT (u.permission_id=g.permission_id) AS was_inherited,u.permission_id,g.permission_id,".
	($type eq 'user' ? 'u.group_id,u.username' : 'u.parent_group_id,u.group_name').
	" FROM ".($type eq 'user' ? 'users' : 'groups')." u ".
	" JOIN groups g ON u.".($type eq 'user' ? '' : 'parent_')."group_id=g.group_id ".
	" WHERE u.".($type eq 'user' ? 'user' : 'group')."_id=?");
  $sth->execute($id);

  my ($wasinherited,$permid,$parpermid,$parid,$name) = $sth->fetchrow_array;

# hack phtoui
# group id 1 is "special" in that it's it's own parent (err...  possibly.)
# may make its parent id 0 which doesn't exist, and as a bonus is Perl-false.
  $wasinherited = 0 if ($type eq 'group' && $id == 1);

  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  # Wrap all the SQL in a transaction
  eval {
    if ($inherit) {

      $dbh->do("UPDATE ".($type eq 'user' ? 'users' : 'groups')." SET inherit_perm='t',permission_id=? ".
	"WHERE ".($type eq 'user' ? 'user' : 'group')."_id=?", undef, ($parpermid, $id) );
      $dbh->do("DELETE FROM permissions WHERE permission_id=?", undef, ($permid) );

    } else {

      if ($wasinherited) {	# munge new permission entry in if we're switching from inherited perms
##fixme: need to add semirecursive bit to properly munge inherited permission ID on subgroups and users
# ... if'n'when we have groups with fully inherited permissions.
        # SQL is coo
	$dbh->do("INSERT INTO permissions ($permlist,".($type eq 'user' ? 'user' : 'group')."_id) ".
		"SELECT $permlist,? FROM permissions WHERE permission_id=?", undef, ($id,$permid) );
	($permid) = $dbh->selectrow_array("SELECT permission_id FROM permissions ".
		"WHERE ".($type eq 'user' ? 'user' : 'group')."_id=?", undef, ($id) );
	$dbh->do("UPDATE ".($type eq 'user' ? 'users' : 'groups')." SET inherit_perm='f',permission_id=? ".
		"WHERE ".($type eq 'user' ? 'user' : 'group')."_id=?", undef, ($permid, $id) );
      }

      # and now set the permissions we were passed
      foreach (@permtypes) {
	if (defined ($newperms->{$_})) {
	  $dbh->do("UPDATE permissions SET $_=? WHERE permission_id=?", undef, ($newperms->{$_},$permid) );
	}
      }

    } # (inherited->)? custom

    if ($type eq 'user') {
      $resultmsg = "Updated permissions for user $name";
    } else {
      $resultmsg = "Updated default permissions for group $name";
    }
    $self->_log(group_id => ($type eq 'user' ? $parid : $id), entry => $resultmsg);
    $dbh->commit;
  }; # end eval
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    return ('FAIL',"Error changing permissions: $msg");
  }

  return ('OK',$resultmsg);
} # end changePermissions()


## DNSDB::comparePermissions()
# Compare two permission hashes
# Returns '>', '<', '=', '!'
sub comparePermissions {
  my $self = shift;
  my $p1 = shift;
  my $p2 = shift;

  my $retval = '=';	# assume equality until proven otherwise

  no warnings "uninitialized";

  foreach (@permtypes) {
    next if $p1->{$_} == $p2->{$_};	# equal is good
    if ($p1->{$_} && !$p2->{$_}) {
      if ($retval eq '<') {	# if we've already found an unequal pair where
        $retval = '!';		# $p2 has more access, and we now find a pair
        last;			# where $p1 has more access, the overall access
      }				# is neither greater or lesser, it's unequal.
      $retval = '>';
    }
    if (!$p1->{$_} && $p2->{$_}) {
      if ($retval eq '>') {	# if we've already found an unequal pair where
        $retval = '!';		# $p1 has more access, and we now find a pair
        last;			# where $p2 has more access, the overall access
      }				# is neither greater or lesser, it's unequal.
      $retval = '<';
    }
  }
  return $retval;
} # end comparePermissions()


## DNSDB::changeGroup()
# Change group ID of an entity
# Takes a database handle, entity type, entity ID, and new group ID
sub changeGroup {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $type = shift;
  my $id = shift;
  my $newgrp = shift;

##fixme:  fail on not enough args
  #return ('FAIL', "Missing 

  return ('FAIL', "Can't change the group of a $type")
	unless grep /^$type$/, ('domain','revzone','user','group');	# could be extended for defrecs?

  # Collect some names for logging and messages
  my $entname;
  if ($type eq 'domain') {
    $entname = $self->domainName($id);
  } elsif ($type eq 'revzone') {
    $entname = $self->revName($id);
  } elsif ($type eq 'user') {
    $entname = $self->userFullName($id, '%u');
  } elsif ($type eq 'group') {
    $entname = $self->groupName($id);
  }

  my ($oldgid) = $dbh->selectrow_array("SELECT group_id FROM $par_tbl{$type} WHERE $id_col{$type}=?",
	undef, ($id));
  my $oldgname = $self->groupName($oldgid);
  my $newgname = $self->groupName($newgrp);

  return ('FAIL', "Can't move things into a group that doesn't exist") if !$newgname;

  return ('WARN', "Nothing to do, new group is the same as the old group") if $oldgid == $newgrp;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  eval {
    $dbh->do("UPDATE $par_tbl{$type} SET group_id=? WHERE $id_col{$type}=?", undef, ($newgrp, $id));
    # Log the change in both the old and new groups
    $self->_log(group_id => $oldgid, entry => "Moved $type $entname from $oldgname to $newgname");
    $self->_log(group_id => $newgrp, entry => "Moved $type $entname from $oldgname to $newgname");
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $self->_log(group_id => $oldgid, entry => "Error moving $type $entname to $newgname: $msg");
      $dbh->commit;	# since we enabled transactions earlier
    }
    return ('FAIL',"Error moving $type $entname to $newgname: $msg");
  }

  return ('OK',"Moved $type $entname from $oldgname to $newgname");
} # end changeGroup()


##
## Processing subs
##

## DNSDB::addDomain()
# Add a domain
# Takes a database handle, domain name, numeric group, boolean(ish) state (active/inactive),
# and a default location indicator
# Returns a status code and message
sub addDomain {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $domain = shift;
  return ('FAIL',"Domain must not be blank\n") if !$domain;
  my $group = shift;
  return ('FAIL',"Group must be specified\n") if !defined($group);
  my $state = shift;
  return ('FAIL',"Domain status must be specified\n") if !defined($state);
  my $defloc = shift || '';

  $state = 1 if $state =~ /^active$/;
  $state = 1 if $state =~ /^on$/;
  $state = 0 if $state =~ /^inactive$/;
  $state = 0 if $state =~ /^off$/;

  return ('FAIL',"Invalid domain status") if $state !~ /^\d+$/;

  $domain = lc($domain) if $self->{lowercase};

  return ('FAIL', "Invalid characters in domain") if $domain !~ /^[a-zA-Z0-9_.-]+$/;

  my $sth = $dbh->prepare("SELECT domain_id FROM domains WHERE lower(domain) = lower(?) AND default_location = ?");
  my $dom_id;

# quick check to start to see if we've already got one
  $sth->execute($domain, $defloc);
  ($dom_id) = $sth->fetchrow_array;

  return ('FAIL', "Domain already exists") if $dom_id;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  # Wrap all the SQL in a transaction
  eval {
    # insert the domain...
    $dbh->do("INSERT INTO domains (domain,group_id,status,default_location) VALUES (?,?,?,?)", undef,
        ($domain, $group, $state, $defloc));

    # get the ID...
    ($dom_id) = $dbh->selectrow_array("SELECT domain_id FROM domains WHERE lower(domain) = lower(?) AND default_location = ?",
	undef, ($domain, $defloc));

    $self->_log(domain_id => $dom_id, group_id => $group,
	entry => "Added ".($state ? 'active' : 'inactive')." domain $domain");

    # ... and now we construct the standard records from the default set.  NB:  group should be variable.
    my $sth = $dbh->prepare("SELECT host,type,val,distance,weight,port,ttl FROM default_records WHERE group_id=?");
    my $sth_in = $dbh->prepare("INSERT INTO records (domain_id,host,type,val,distance,weight,port,ttl,location)".
	" VALUES ($dom_id,?,?,?,?,?,?,?,?)");
    $sth->execute($group);
    while (my ($host, $type, $val, $dist, $weight, $port, $ttl) = $sth->fetchrow_array()) {
      $host =~ s/DOMAIN/$domain/g;
      $val =~ s/DOMAIN/$domain/g;
      _caseclean(\$type, \$host, \$val, 'n', 'n') if $self->{lowercase};
      $sth_in->execute($host, $type, $val, $dist, $weight, $port, $ttl, $defloc);
      if ($typemap{$type} eq 'SOA') {
	my @tmp1 = split /:/, $host;
	my @tmp2 = split /:/, $val;
	$self->_log(domain_id => $dom_id, group_id => $group,
		entry => "[new $domain] Added SOA record [contact $tmp1[0]] [master $tmp1[1]] ".
		"[refresh $tmp2[0]] [retry $tmp2[1]] [expire $tmp2[2]] [minttl $tmp2[3]], TTL $ttl");
      } else {
	my $logentry = "[new $domain] Added record '$host $typemap{$type}";
	$logentry .= " [distance $dist]" if $typemap{$type} eq 'MX';
	$logentry .= " [priority $dist] [weight $weight] [port $port]" if $typemap{$type} eq 'SRV';
	$self->_log(domain_id => $dom_id, group_id => $group,
		entry => $logentry." $val', TTL $ttl");
      }
    }

    # once we get here, we should have suceeded.
    $dbh->commit;
  }; # end eval

  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    $self->_log(group_id => $group, entry => "Failed adding domain $domain ($msg)")
	if $self->{log_failures};
    $dbh->commit;	# since we enabled transactions earlier
    return ('FAIL',$msg);
  } else {
    return ('OK',$dom_id);
  }
} # end addDomain


## DNSDB::delZone()
# Delete a forward or reverse zone.
# Takes a database handle, zone ID, and forward/reverse flag.
# for now, just delete the records, then the domain.
# later we may want to archive it in some way instead (status code 2, for example?)
sub delZone {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $zoneid = shift;
  my $revrec = shift;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  return ('FAIL', 'Need a zone identifier to look up') if !$zoneid;

  my $msg = '';
  my $failmsg = '';
  my $zone = ($revrec eq 'n' ? $self->domainName($zoneid) : $self->revName($zoneid));

  return ('FAIL', ($revrec eq 'n' ? 'Domain' : 'Reverse zone')." ID $zoneid doesn't exist") if !$zone;

  # Set this up here since we may use if if $self->{log_failures} is enabled
  my %loghash;
  $loghash{domain_id} = $zoneid if $revrec eq 'n';
  $loghash{rdns_id} = $zoneid if $revrec eq 'y';
  $loghash{group_id} = $self->parentID(
	id => $zoneid, type => ($revrec eq 'n' ? 'domain' : 'revzone'), revrec => $revrec);

  # Wrap all the SQL in a transaction
  eval {
    # Disentangle custom record types before removing the
    # ones that are only in the zone to be deleted
    if ($revrec eq 'n') {
      my $sth = $dbh->prepare("UPDATE records SET type=?,domain_id=0 WHERE domain_id=? AND type=?");
      $failmsg = "Failure converting multizone types to single-zone";
      $sth->execute($reverse_typemap{PTR}, $zoneid, 65280);
      $sth->execute($reverse_typemap{PTR}, $zoneid, 65281);
      $sth->execute(65282, $zoneid, 65283);
      $sth->execute(65282, $zoneid, 65284);
      $failmsg = "Failure removing domain records";
      $dbh->do("DELETE FROM records WHERE domain_id=?", undef, ($zoneid));
      $failmsg = "Failure removing domain";
      $dbh->do("DELETE FROM domains WHERE domain_id=?", undef, ($zoneid));
    } else {
      my $sth = $dbh->prepare("UPDATE records SET type=?,rdns_id=0 WHERE rdns_id=? AND type=?");
      $failmsg = "Failure converting multizone types to single-zone";
      $sth->execute($reverse_typemap{A}, $zoneid, 65280);
      $sth->execute($reverse_typemap{AAAA}, $zoneid, 65281);
# We don't have an "A template" or "AAAA template" type, although it might be useful for symmetry.
#      $sth->execute(65286?, $zoneid, 65283);
#      $sth->execute(65286?, $zoneid, 65284);
      $failmsg = "Failure removing reverse records";
      $dbh->do("DELETE FROM records WHERE rdns_id=?", undef, ($zoneid));
      $failmsg = "Failure removing reverse zone";
      $dbh->do("DELETE FROM revzones WHERE rdns_id=?", undef, ($zoneid));
    }

    $msg = "Deleted ".($revrec eq 'n' ? 'domain' : 'reverse zone')." $zone";
    $loghash{entry} = $msg;
    $self->_log(%loghash);

    # once we get here, we should have suceeded.
    $dbh->commit;
  }; # end eval

  if ($@) {
    $msg = $@;
    eval { $dbh->rollback; };
    $loghash{entry} = "Error deleting $zone: $msg ($failmsg)";
    if ($self->{log_failures}) {
      $self->_log(%loghash);
      $dbh->commit;	# since we enabled transactions earlier
    }
    return ('FAIL', $loghash{entry});
  } else {
    return ('OK', $msg);
  }

} # end delZone()


## DNSDB::domainName()
# Return the domain name based on a domain ID
# Takes a database handle and the domain ID
# Returns the domain name or undef on failure
sub domainName {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $domid = shift;
  my ($domname) = $dbh->selectrow_array("SELECT domain FROM domains WHERE domain_id=?", undef, ($domid) );
  $errstr = $DBI::errstr if !$domname;
  return $domname if $domname;
} # end domainName()


## DNSDB::revName()
# Return the reverse zone name based on an rDNS ID
# Takes a database handle and the rDNS ID, and an optional flag to force return of the CIDR zone
# instead of the formal .arpa zone name
# Returns the reverse zone name or undef on failure
sub revName {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $revid = shift;
  my $cidrflag = shift || 'n';
  my ($revname) = $dbh->selectrow_array("SELECT revnet FROM revzones WHERE rdns_id=?", undef, ($revid) );
  $errstr = $DBI::errstr if !$revname;
  my $tmp = new NetAddr::IP $revname;
  $revname = _ZONE($tmp, 'ZONE', 'r', '.').($tmp->{isv6} ? '.ip6.arpa' : '.in-addr.arpa')
	if ($self->{showrev_arpa} eq 'zone' || $self->{showrev_arpa} eq 'all') && $cidrflag eq 'n';
  return $revname if $revname;
} # end revName()


## DNSDB::domainID()
# Takes a database handle and domain name
# Returns the domain ID number
sub domainID {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $domain = shift;
  my ($domid) = $dbh->selectrow_array("SELECT domain_id FROM domains WHERE lower(domain) = lower(?)",
	undef, ($domain) );
  if (!$domid) {
    if ($dbh->err) {
      $errstr = $DBI::errstr;
    } else {
      $errstr = "Domain $domain not present";
    }
  }
  return $domid if $domid;
} # end domainID()


## DNSDB::revID()
# Takes a database handle and reverse zone name
# Returns the rDNS ID number
sub revID {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $revzone = shift;
  my ($revid) = $dbh->selectrow_array("SELECT rdns_id FROM revzones WHERE revnet=?", undef, ($revzone) );
  if (!$revid) {
    if ($dbh->err) {
      $errstr = $DBI::errstr;
    } else {
      $errstr = "Reverse zone $revzone not present";
    }
  }
  return $revid if $revid;
} # end revID()


## DNSDB::addRDNS
# Adds a reverse DNS zone
# Takes a database handle, CIDR block, reverse DNS pattern, numeric group,
# and boolean(ish) state (active/inactive)
# Returns a status code and message
sub addRDNS {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $zone = shift;

  # Autodetect formal .arpa zones
  if ($zone =~ /\.arpa\.?$/) {
    my $code;
    ($code,$zone) = _zone2cidr($zone);
    return ('FAIL', $zone) if $code eq 'FAIL';
  }
  $zone = NetAddr::IP->new($zone);

  return ('FAIL',"Zone name must be a valid CIDR netblock") unless ($zone && $zone->addr !~ /^0/);
  my $revpatt = shift;	# construct a custom (A/AAAA+)? PTR template record
  my $group = shift;
  my $state = shift;
  my $defloc = shift || '';

  $state = 1 if $state =~ /^active$/;
  $state = 1 if $state =~ /^on$/;
  $state = 0 if $state =~ /^inactive$/;
  $state = 0 if $state =~ /^off$/;

  return ('FAIL',"Invalid zone status") if $state !~ /^\d+$/;

# quick check to start to see if we've already got one
  my ($rdns_id) = $dbh->selectrow_array("SELECT rdns_id FROM revzones WHERE revnet = ? AND default_location = ?",
	undef, ("$zone", $defloc));

  return ('FAIL', "Zone already exists") if $rdns_id;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  my $warnstr = '';
  my $defttl = 3600;	# 1 hour should be reasonable.  And unless things have gone horribly
			# wrong, we should have a value to override this anyway.

  # Wrap all the SQL in a transaction
  eval {
    # insert the zone...
    $dbh->do("INSERT INTO revzones (revnet,group_id,status,default_location) VALUES (?,?,?,?)", undef,
	($zone, $group, $state, $defloc) );

    # get the ID...
    ($rdns_id) = $dbh->selectrow_array("SELECT currval('revzones_rdns_id_seq')");

    $self->_log(rdns_id => $rdns_id, group_id => $group,
	entry => "Added ".($state ? 'active' : 'inactive')." reverse zone $zone");

    # ... and now we construct the standard records from the default set.  NB:  group should be variable.
    my $sth = $dbh->prepare("SELECT host,type,val,ttl FROM default_rev_records WHERE group_id=?");
    my $sth_in = $dbh->prepare("INSERT INTO records (rdns_id,domain_id,host,type,val,ttl,location)".
	" VALUES ($rdns_id,?,?,?,?,?,?)");
    $sth->execute($group);
    while (my ($host,$type,$val,$ttl) = $sth->fetchrow_array()) {
      # Silently skip v4/v6 mismatches.  This is not an error, this is expected.
      if ($zone->{isv6}) {
	next if ($type == 65280 || $type == 65283);
      } else {
	next if ($type == 65281 || $type == 65284);
      }

      $host =~ s/ADMINDOMAIN/$self->{domain}/g;

      # Check to make sure the IP stubs will fit in the zone.  Under most usage failures here should be rare.
      # On failure, tack a note on to a warning string and continue without adding this record.
      # While we're at it, we substitute $zone for ZONE in the value.
      if ($val eq 'ZONE') {
	# If we've got a pattern, we skip the default record version on (A+)PTR-template types
	next if $revpatt && ($type == 65282 || $type == 65283);	
##fixme?  do we care if we have multiple whole-zone templates?
	$val = $zone->network;
      } elsif ($val =~ /ZONE/) {
	my $tmpval = $val;
	$tmpval =~ s/ZONE//;
	# Bend the rules and allow single-trailing-number PTR or PTR template records to be inserted
	# as either v4 or v6.  May make this an off-by-default config flag
	# Note that the origin records that may trigger this **SHOULD** already have ZONE,\d
	if ($type == 12 || $type == 65282) {
	  $tmpval =~ s/[,.]/::/ if ($tmpval =~ /^[,.]\d+$/ && $zone->{isv6});
	  $tmpval =~ s/[,:]+/./ if ($tmpval =~ /^(?:,|::)\d+$/ && !$zone->{isv6});
	}
	my $addr;
	if ($self->_ipparent('n', 'y', \$tmpval, $rdns_id, \$addr)) {
	  $val = $addr->addr;
	} else {
	  $warnstr .= "\nDefault record '$val $typemap{$type} $host' doesn't fit in $zone, skipping";
	  next;
	}
      }

      # Substitute $zone for ZONE in the hostname, but only for non-NS records.
      # NS records get this substitution on the value instead.
      $host = _ZONE($zone, $host) if $type != 2;

      # Fill in the forward domain ID if we can find it, otherwise:
      # Coerce type down to PTR or PTR template if we can't
      my $domid = 0;
      if ($type >= 65280) {
	if (!($domid = $self->_hostparent($host))) {
	  $warnstr .= "\nRecord added as PTR instead of $typemap{$type};  domain not found for $host";
	  $type = $reverse_typemap{PTR};
	  $domid = 0;	# just to be explicit.
	}
      }

      _caseclean(\$type, \$host, \$val, 'n', 'y') if $self->{lowercase};

      $sth_in->execute($domid,$host,$type,$val,$ttl,$defloc);

      if ($typemap{$type} eq 'SOA') {
	my @tmp1 = split /:/, $host;
	my @tmp2 = split /:/, $val;
	$self->_log(rdns_id => $rdns_id, group_id => $group,
		entry => "[new $zone] Added SOA record [contact $tmp1[0]] [master $tmp1[1]] ".
		"[refresh $tmp2[0]] [retry $tmp2[1]] [expire $tmp2[2]] [minttl $tmp2[3]], TTL $ttl");
	$defttl = $tmp2[3];
      } else {
	my $logentry = "[new $zone] Added record '$host $typemap{$type} $val', TTL $ttl";
	$logentry .= ", default location ".$self->getLoc($defloc)->{description} if $defloc;
	$self->_log(rdns_id => $rdns_id, domain_id => $domid, group_id => $group, entry => $logentry);
      }
    }

    # Generate record based on provided pattern.  
    if ($revpatt) {
      my $host;
      my $type = ($zone->{isv6} ? 65284 : 65283);
      my $val = $zone->network;

      # Substitute $zone for ZONE in the hostname.
      $host = _ZONE($zone, $revpatt);

      my $domid = 0;
      if (!($domid = $self->_hostparent($host))) {
	$warnstr .= "\nDefault pattern added as PTR template instead of $typemap{$type};  domain not found for $host";
	$type = 65282;
	$domid = 0;	# just to be explicit.
      }

      $sth_in->execute($domid,$host,$type,$val,$defttl,$defloc);
      my $logentry = "[new $zone] Added record '$host $typemap{$type}";
      $self->_log(rdns_id => $rdns_id, domain_id => $domid, group_id => $group,
	entry => $logentry." $val', TTL $defttl from pattern");
    }

    # If there are warnings (presumably about default records skipped for cause) log them
    $self->_log(rdns_id => $rdns_id, group_id => $group, entry => "Warning(s) adding $zone:$warnstr")
	if $warnstr;

    # once we get here, we should have suceeded.
    $dbh->commit;
  }; # end eval

  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    $self->_log(group_id => $group, entry => "Failed adding reverse zone $zone ($msg)")
	if $self->{log_failures};
    $dbh->commit;	# since we enabled transactions earlier
    return ('FAIL',$msg);
  } else {
    my $retcode = 'OK';
    if ($warnstr) {
      $resultstr = $warnstr;
      $retcode = 'WARN';
    }
    return ($retcode, $rdns_id);
  }

} # end addRDNS()


## DNSDB::getZoneCount
# Get count of zones in group or groups
# Takes a database handle and hash containing:
#  - the "current" group
#  - an array of "acceptable" groups
#  - a flag for forward/reverse zones
#  - Optionally accept a "starts with" and/or "contains" filter argument
# Returns an integer count of the resulting zone list.
sub getZoneCount {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  $args{filter} =~ s/\./\[\.\]/g if $args{filter};	# only match literal dots, usually in reverse zones
  push @filterargs, $args{filter} if $args{filter};

  my $sql;
  # Not as compact, and fix-me-twice if the common bits get wrong, but much easier to read
  if ($args{revrec} eq 'n') {
    $sql = "SELECT count(*) FROM domains".
	" WHERE group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND domain ~* ?" : '').
	($args{filter} ? " AND domain ~* ?" : '');
  } else {
    $sql = "SELECT count(*) FROM revzones".
	" WHERE group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND CAST(revnet AS VARCHAR) ~* ?" : '');
#    if ($self->{showrev_arpa} eq 'zone' || $self->{showrev_arpa} eq 'all') {
      # Just In Case the UI is using formal .arpa notation, and someone enters something reversed,
      # we want to match both the formal and natural zone name
      $sql .= ($args{filter} ? " AND (CAST(revnet AS VARCHAR) ~* ? OR CAST(revnet AS VARCHAR) ~* ?)" : '');
      push @filterargs, join('[.]',reverse(split(/\[\.\]/,$args{filter}))) if $args{filter};
#    } else {
#      $sql .= ($args{filter} ? " AND CAST(revnet AS VARCHAR) ~* ?" : '');
#    }
  }
  my ($count) = $dbh->selectrow_array($sql, undef, @filterargs);
  return $count;
} # end getZoneCount()


## DNSDB::getZoneList()
# Get a list of zones in the specified group(s)
# Takes the same arguments as getZoneCount() above
# Returns a reference to an array of hashrefs suitable for feeding to HTML::Template
sub getZoneList {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  my @zonelist;

  $args{sortorder} = 'ASC' if !$args{sortorder} || !grep /^$args{sortorder}$/, ('ASC','DESC');
  $args{offset} = 0 if !$args{offset} || $args{offset} !~ /^(?:all|\d+)$/;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  $args{filter} =~ s/\./\[\.\]/g if $args{filter};	# only match literal dots, usually in reverse zones
  push @filterargs, $args{filter} if $args{filter};

  my $sql;
  # Not as compact, and fix-me-twice if the common bits get wrong, but much easier to read
  if ($args{revrec} eq 'n') {
    $args{sortby} = 'domain' if !$args{sortby} || !grep /^$args{sortby}$/, ('domain','group','status');
    $sql = q(SELECT
		domain_id AS zoneid,
		domain AS zone,
		status,
		groups.group_name AS group,
		l.description AS location
	FROM domains
	LEFT JOIN locations l ON domains.default_location=l.location
	INNER JOIN groups ON domains.group_id=groups.group_id ).
	"WHERE domains.group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND domain ~* ?" : '').
	($args{filter} ? " AND domain ~* ?" : '');
  } else {
##fixme:  arguably startwith here is irrelevant.  depends on the UI though.
    $args{sortby} = 'revnet' if !$args{sortby} || !grep /^$args{sortby}$/, ('revnet','group','status');
    $sql = q(SELECT
		rdns_id AS zoneid,
		revnet AS zone,
		status,
		groups.group_name AS group,
		l.description AS location
	FROM revzones
	LEFT JOIN locations l ON revzones.default_location=l.location
	INNER JOIN groups ON revzones.group_id=groups.group_id ).
	" WHERE revzones.group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND CAST(revnet AS VARCHAR) ~* ?" : '');
#    if ($self->{showrev_arpa} eq 'zone' || $self->{showrev_arpa} eq 'all') {
      # Just In Case the UI is using formal .arpa notation, and someone enters something reversed,
      # we want to match both the formal and natural zone name
      $sql .= ($args{filter} ? " AND (CAST(revnet AS VARCHAR) ~* ? OR CAST(revnet AS VARCHAR) ~* ?)" : '');
      push @filterargs, join('[.]',reverse(split(/\[\.\]/,$args{filter}))) if $args{filter};
#    } else {
#      $sql .= ($args{filter} ? " AND CAST(revnet AS VARCHAR) ~* ?" : '');
#    }
  }
  # A common tail.
  $sql .= " ORDER BY ".($args{sortby} eq 'group' ? 'groups.group_name' : $args{sortby})." $args{sortorder} ".
	($args{offset} eq 'all' ? '' : " LIMIT $self->{perpage}".
	" OFFSET ".$args{offset}*$self->{perpage});

  my @working;
  my $zsth = $dbh->prepare($sql);
  $zsth->execute(@filterargs);
  while (my $zone = $zsth->fetchrow_hashref) {
    if ($args{revrec} eq 'y' && ($self->{showrev_arpa} eq 'zone' || $self->{showrev_arpa} eq 'all')) {
      my $tmp = new NetAddr::IP $zone->{zone};
      $zone->{zone} = DNSDB::_ZONE($tmp, 'ZONE', 'r', '.').($tmp->{isv6} ? '.ip6.arpa' : '.in-addr.arpa');
    }
    push @working, $zone;
  }
  return \@working;
} # end getZoneList()


## DNSDB::getZoneLocation()
# Retrieve the default location for a zone.
# Takes a database handle, forward/reverse flag, and zone ID
sub getZoneLocation {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $revrec = shift;
  my $zoneid = shift;

  my ($loc) = $dbh->selectrow_array("SELECT default_location FROM ".
	($revrec eq 'n' ? 'domains WHERE domain_id = ?' : 'revzones WHERE rdns_id = ?'),
	undef, ($zoneid));
  return $loc;
} # end getZoneLocation()


## DNSDB::addGroup()
# Add a group
# Takes a database handle, group name, parent group, hashref for permissions,
# and optional template-vs-cloneme flag for the default records
# Returns a status code and message
sub addGroup {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $groupname = shift;
  my $pargroup = shift;
  my $permissions = shift;

  # 0 indicates "custom", hardcoded.
  # Any other value clones that group's default records, if it exists.
  my $inherit = shift || 0;	
##fixme:  need a flag to indicate clone records or <?> ?

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  my ($group_id) = $dbh->selectrow_array("SELECT group_id FROM groups WHERE group_name=?", undef, ($groupname));

  return ('FAIL', "Group already exists") if $group_id;

  # Wrap all the SQL in a transaction
  eval {
    $dbh->do("INSERT INTO groups (parent_group_id,group_name) VALUES (?,?)", undef, ($pargroup, $groupname) );

    my ($groupid) = $dbh->selectrow_array("SELECT currval('groups_group_id_seq')");

    # We work through the whole set of permissions instead of specifying them so
    # that when we add a new permission, we don't have to change the code anywhere
    # that doesn't explicitly deal with that specific permission.
    my @permvals;
    foreach (@permtypes) {
      if (!defined ($permissions->{$_})) {
	push @permvals, 0;
      } else {
	push @permvals, $permissions->{$_};
      }
    }
    $dbh->do("INSERT INTO permissions (group_id,$permlist) values (?".',?'x($#permtypes+1).")",
	undef, ($groupid, @permvals) );
    my ($permid) = $dbh->selectrow_array("SELECT currval('permissions_permission_id_seq')");
    $dbh->do("UPDATE groups SET permission_id=$permid WHERE group_id=$groupid");

    # Default records
    my $sthf = $dbh->prepare("INSERT INTO default_records (group_id,host,type,val,distance,weight,port,ttl) ".
	"VALUES ($groupid,?,?,?,?,?,?,?)");
    my $sthr = $dbh->prepare("INSERT INTO default_rev_records (group_id,host,type,val,ttl) ".
	"VALUES ($groupid,?,?,?,?)");
    if ($inherit) {
      # Duplicate records from parent.  Actually relying on inherited records feels
      # very fragile, and it would be problematic to roll over at a later time.
      my $sth2 = $dbh->prepare("SELECT host,type,val,distance,weight,port,ttl FROM default_records WHERE group_id=?");
      $sth2->execute($pargroup);
      while (my @clonedata = $sth2->fetchrow_array) {
	$sthf->execute(@clonedata);
      }
      # And now the reverse records
      $sth2 = $dbh->prepare("SELECT host,type,val,ttl FROM default_rev_records WHERE group_id=?");
      $sth2->execute($pargroup);
      while (my @clonedata = $sth2->fetchrow_array) {
	$sthr->execute(@clonedata);
      }
    } else {
##fixme: Hardcoding is Bad, mmmmkaaaay?
      # reasonable basic defaults for SOA, MX, NS, and minimal hosting
      # could load from a config file, but somewhere along the line we need hardcoded bits.
      $sthf->execute('ns1.example.com:hostmaster.example.com', 6, '10800:3600:604800:10800', 0, 0, 0, 86400);
      $sthf->execute('DOMAIN', 1, '192.168.4.2', 0, 0, 0, 7200);
      $sthf->execute('DOMAIN', 15, 'mx.example.com', 10, 0, 0, 7200);
      $sthf->execute('DOMAIN', 2, 'ns1.example.com', 0, 0, 0, 7200);
      $sthf->execute('DOMAIN', 2, 'ns2.example.com', 0, 0, 0, 7200);
      $sthf->execute('www.DOMAIN', 5, 'DOMAIN', 0, 0, 0, 7200);
      # reasonable basic defaults for generic reverse zone.  Same as initial SQL tabledef.
      $sthr->execute('hostmaster.ADMINDOMAIN:ns1.ADMINDOMAIN', 6, '10800:3600:604800:10800', 86400);
      $sthr->execute('unused-%r.ADMINDOMAIN', 65283, 'ZONE', 3600);
    }

    $self->_log(group_id => $pargroup, entry => "Added group $groupname");

    # once we get here, we should have suceeded.
    $dbh->commit;
  }; # end eval

  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $self->_log(group_id => $pargroup, entry => "Failed to add group $groupname: $msg");
      $dbh->commit;
    }
    return ('FAIL',$msg);
  }

  return ('OK','OK');
} # end addGroup()


## DNSDB::delGroup()
# Delete a group.
# Takes a group ID
# Returns a status code and message
sub delGroup {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $groupid = shift;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

##fixme:  locate "knowable" error conditions and deal with them before the eval
# ... or inside, whatever.
# -> domains still exist in group
# -> ...
  my $failmsg = '';
  my $resultmsg = '';

  # collect some pieces for logging and error messages
  my $groupname = $self->groupName($groupid);
  my $parid = $self->parentID(id => $groupid, type => 'group');

  # Wrap all the SQL in a transaction
  eval {
    # Check for Things in the group
    $failmsg = "Can't remove group $groupname";
    my ($grpcnt) = $dbh->selectrow_array("SELECT count(*) FROM groups WHERE parent_group_id=?", undef, ($groupid));
    die "$grpcnt groups still in group\n" if $grpcnt;
    my ($domcnt) = $dbh->selectrow_array("SELECT count(*) FROM domains WHERE group_id=?", undef, ($groupid));
    die "$domcnt domains still in group\n" if $domcnt;
    my ($revcnt) = $dbh->selectrow_array("SELECT count(*) FROM revzones WHERE group_id=?", undef, ($groupid));
    die "$revcnt reverse zones still in group\n" if $revcnt;
    my ($usercnt) = $dbh->selectrow_array("SELECT count(*) FROM users WHERE group_id=?", undef, ($groupid));
    die "$usercnt users still in group\n" if $usercnt;

    $failmsg = "Failed to delete default records for $groupname";
    $dbh->do("DELETE from default_records WHERE group_id=?", undef, ($groupid));
    $failmsg = "Failed to delete default reverse records for $groupname";
    $dbh->do("DELETE from default_rev_records WHERE group_id=?", undef, ($groupid));
    $failmsg = "Failed to remove group $groupname";
    $dbh->do("DELETE from groups WHERE group_id=?", undef, ($groupid));

    $self->_log(group_id => $parid, entry => "Deleted group $groupname");
    $resultmsg = "Deleted group $groupname";

    # once we get here, we should have suceeded.
    $dbh->commit;
  }; # end eval

  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $self->_log(group_id => $parid, entry => "$failmsg: $msg");
      $dbh->commit;	# since we enabled transactions earlier
    }
    return ('FAIL',"$failmsg: $msg");
  }

  return ('OK',$resultmsg);
} # end delGroup()


## DNSDB::getChildren()
# Get a list of all groups whose parent^n is group <n>
# Takes a database handle, group ID, reference to an array to put the group IDs in,
# and an optional flag to return only immediate children or all children-of-children
# default to returning all children
# Calls itself
sub getChildren {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $rootgroup = shift;
  my $groupdest = shift;
  my $immed = shift || 'all';

  # special break for default group;  otherwise we get stuck.
  if ($rootgroup == 1) {
    # by definition, group 1 is the Root Of All Groups
    my $sth = $dbh->prepare("SELECT group_id FROM groups WHERE NOT (group_id=1)".
	($immed ne 'all' ? " AND parent_group_id=1" : '')." ORDER BY group_name");
    $sth->execute;
    while (my @this = $sth->fetchrow_array) {
      push @$groupdest, @this;
    }
  } else {
    my $sth = $dbh->prepare("SELECT group_id FROM groups WHERE parent_group_id=? ORDER BY group_name");
    $sth->execute($rootgroup);
    return if $sth->rows == 0;
    my @grouplist;
    while (my ($group) = $sth->fetchrow_array) {
      push @$groupdest, $group;
      $self->getChildren($group, $groupdest) if $immed eq 'all';
    }
  }
} # end getChildren()


## DNSDB::groupName()
# Return the group name based on a group ID
# Takes a database handle and the group ID
# Returns the group name or undef on failure
sub groupName {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $groupid = shift;
  my $sth = $dbh->prepare("SELECT group_name FROM groups WHERE group_id=?");
  $sth->execute($groupid);
  my ($groupname) = $sth->fetchrow_array();
  $errstr = $DBI::errstr if !$groupname;
  return $groupname if $groupname;
} # end groupName


## DNSDB::getGroupCount()
# Get count of subgroups in group or groups
# Takes a database handle and hash containing:
#  - the "current" group
#  - an array of "acceptable" groups
#  - Optionally accept a "starts with" and/or "contains" filter argument
# Returns an integer count of the resulting group list.
sub getGroupCount {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  push @filterargs, $args{filter} if $args{filter};

  my $sql = "SELECT count(*) FROM groups ".
	"WHERE parent_group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND group_name ~* ?" : '').
	($args{filter} ? " AND group_name ~* ?" : '');
  my ($count) = $dbh->selectrow_array($sql, undef, (@filterargs) );
  $errstr = $dbh->errstr if !$count;
  return $count;
} # end getGroupCount


## DNSDB::getGroupList()
# Get a list of sub^n-groups in the specified group(s)
# Takes the same arguments as getGroupCount() above
# Returns an arrayref containing hashrefs suitable for feeding straight to HTML::Template
sub getGroupList {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  push @filterargs, $args{filter} if $args{filter};

  # protection against bad or missing arguments
  $args{sortorder} = 'ASC' if !$args{sortorder} || !grep /^$args{sortorder}$/, ('ASC','DESC');
  $args{sortby} = 'group' if !$args{sortby} || $args{sortby} !~ /^[\w_.]+$/;
  $args{offset} = 0 if !$args{offset} || $args{offset} !~ /^(?:all|\d+)$/;

  # munge sortby for columns in database
  $args{sortby} = 'g.group_name' if $args{sortby} eq 'group';
  $args{sortby} = 'g2.group_name' if $args{sortby} eq 'parent';

  my $sql = q(SELECT g.group_id AS groupid, g.group_name AS groupname, g2.group_name AS pgroup
	FROM groups g
	INNER JOIN groups g2 ON g2.group_id=g.parent_group_id
	).
	" WHERE g.parent_group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND g.group_name ~* ?" : '').
	($args{filter} ? " AND g.group_name ~* ?" : '').
	" GROUP BY g.group_id, g.group_name, g2.group_name ".
	" ORDER BY $args{sortby} $args{sortorder} ".
	($args{offset} eq 'all' ? '' : " LIMIT $self->{perpage} OFFSET ".$args{offset}*$self->{perpage});
  my $glist = $dbh->selectall_arrayref($sql, { Slice => {} }, (@filterargs) );
  $errstr = $dbh->errstr if !$glist;

  # LEFT JOINs make the result set balloon beyond sanity just to include counts;
  # this means there's lots of crunching needed to trim the result set back down.
  # So instead we track the order of the groups, and push the counts into the
  # arrayref result separately.
##fixme:  put this whole sub in a transaction?  might be
# needed for accurate results on very busy systems.
##fixme:  large group lists need prepared statements?
#my $ucsth = $dbh->prepare("SELECT count(*) FROM users WHERE group_id=?");
#my $dcsth = $dbh->prepare("SELECT count(*) FROM domains WHERE group_id=?");
#my $rcsth = $dbh->prepare("SELECT count(*) FROM revzones WHERE group_id=?");
  foreach (@{$glist}) {
    my ($ucnt) = $dbh->selectrow_array("SELECT count(*) FROM users WHERE group_id=?", undef, ($$_{groupid}));
    $$_{nusers} = $ucnt;
    my ($dcnt) = $dbh->selectrow_array("SELECT count(*) FROM domains WHERE group_id=?", undef, ($$_{groupid}));
    $$_{ndomains} = $dcnt;
    my ($rcnt) = $dbh->selectrow_array("SELECT count(*) FROM revzones WHERE group_id=?", undef, ($$_{groupid}));
    $$_{nrevzones} = $rcnt;
  }

  return $glist;
} # end getGroupList


## DNSDB::groupID()
# Return the group ID based on the group name
# Takes a database handle and the group name
# Returns the group ID or undef on failure
sub groupID {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $group = shift;
  my ($grpid) = $dbh->selectrow_array("SELECT group_id FROM groups WHERE group_name=?", undef, ($group) );
  $errstr = $DBI::errstr if !$grpid;
  return $grpid if $grpid;
} # end groupID()


## DNSDB::addUser()
# Add a user.
# Takes a DB handle, username, group ID, password, state (active/inactive).
# Optionally accepts:
#   user type (user/admin)	- defaults to user
#   permissions string		- defaults to inherit from group
#      three valid forms:
#	i		     - Inherit permissions
#	c:<user_id>	     - Clone permissions from <user_id>
#	C:<permission list>  - Set these specific permissions
#   first name			- defaults to username
#   last name			- defaults to blank
#   phone			- defaults to blank (could put other data within column def)
# Returns (OK,<uid>) on success, (FAIL,<message>) on failure
sub addUser {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $username = shift;
  my $group = shift;
  my $pass = shift;
  my $state = shift;

  return ('FAIL', "Missing one or more required entries") if !defined($state);
  return ('FAIL', "Username must not be blank") if !$username;

  # Munge in some alternate state values
  $state = 1 if $state =~ /^active$/;
  $state = 1 if $state =~ /^on$/;
  $state = 0 if $state =~ /^inactive$/;
  $state = 0 if $state =~ /^off$/;

  my $type = shift || 'u';	# create limited users by default - fwiw, not sure yet how this will interact with ACLs
  
  my $permstring = shift || 'i';	# default is to inhert permissions from group

  my $fname = shift || $username;
  my $lname = shift || '';
  my $phone = shift || '';	# not going format-check

  my $sth = $dbh->prepare("SELECT user_id FROM users WHERE username=?");
  my $user_id;

# quick check to start to see if we've already got one
  $sth->execute($username);
  ($user_id) = $sth->fetchrow_array;

  return ('FAIL', "User already exists") if $user_id;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  # Wrap all the SQL in a transaction
  eval {
    # insert the user...  note we set inherited perms by default since
    # it's simple and cleans up some other bits of state
##fixme:  need better handling of case of inherited or missing (!!) permissions entries
    my $sth = $dbh->prepare("INSERT INTO users ".
	"(group_id,username,password,firstname,lastname,phone,type,status,permission_id,inherit_perm) ".
	"VALUES (?,?,?,?,?,?,?,?,(SELECT permission_id FROM permissions WHERE group_id=?),'t')");
    $sth->execute($group,$username,unix_md5_crypt($pass),$fname,$lname,$phone,$type,$state,$group);

    # get the ID...
    ($user_id) = $dbh->selectrow_array("SELECT currval('users_user_id_seq')");

# Permissions!  Gotta set'em all!
    die "Invalid permission string $permstring\n"
	if $permstring !~ /^(?:
		i	# inherit
		|c:\d+	# clone
			# custom.  no, the leading , is not a typo
		|C:(?:,(?:group|user|domain|record|location|self)_(?:edit|create|delete|locchg|view))*
		)$/x;
# bleh.  I'd call another function to do my dirty work, but we're in the middle of a transaction already.
    if ($permstring ne 'i') {
      # for cloned or custom permissions, we have to create a new permissions entry.
      my $clonesrc = $group;
      if ($permstring =~ /^c:(\d+)/) { $clonesrc = $1; }
      $dbh->do("INSERT INTO permissions ($permlist,user_id) ".
	"SELECT $permlist,? FROM permissions WHERE permission_id=".
	"(SELECT permission_id FROM permissions WHERE ".($permstring =~ /^c:/ ? 'user' : 'group')."_id=?)",
	undef, ($user_id,$clonesrc) );
      $dbh->do("UPDATE users SET permission_id=".
	"(SELECT permission_id FROM permissions WHERE user_id=?) ".
	"WHERE user_id=?", undef, ($user_id, $user_id) );
    }
    if ($permstring =~ /^C:/) {
      # finally for custom permissions, we set the passed-in permissions (and unset
      # any that might have been brought in by the clone operation above)
      my ($permid) = $dbh->selectrow_array("SELECT permission_id FROM permissions WHERE user_id=?",
	undef, ($user_id) );
      foreach (@permtypes) {
	if ($permstring =~ /,$_/) {
	  $dbh->do("UPDATE permissions SET $_='t' WHERE permission_id=?", undef, ($permid) );
	} else {
	  $dbh->do("UPDATE permissions SET $_='f' WHERE permission_id=?", undef, ($permid) );
	}
      }
    }

    $dbh->do("UPDATE users SET inherit_perm='n' WHERE user_id=?", undef, ($user_id) );

##fixme: add another table to hold name/email for log table?

    $self->_log(group_id => $group, entry => "Added user $username ($fname $lname)");
    # once we get here, we should have suceeded.
    $dbh->commit;
  }; # end eval

  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $self->_log(group_id => $group, entry => "Error adding user $username: $msg");
      $dbh->commit;	# since we enabled transactions earlier
    }
    return ('FAIL',"Error adding user $username: $msg");
  }

  return ('OK',"User $username ($fname $lname) added");
} # end addUser


## DNSDB::getUserCount()
# Get count of users in group
# Takes a database handle and hash containing at least the current group, and optionally:
# - a reference list of secondary groups
# - a filter string
# - a "Starts with" string
sub getUserCount {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  push @filterargs, $args{filter} if $args{filter};

  my $sql = "SELECT count(*) FROM users ".
	"WHERE group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND username ~* ?" : '').
	($args{filter} ? " AND username ~* ?" : '');
  my ($count) = $dbh->selectrow_array($sql, undef, (@filterargs) );
  $errstr = $dbh->errstr if !$count;
  return $count;
} # end getUserCount()


## DNSDB::getUserList()
# Get list of users
# Takes the same arguments as getUserCount() above, plus optional:
# - sort field
# - sort order
# - offset/return-all-everything flag (defaults to $perpage records)
sub getUserList {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  push @filterargs, $args{filter} if $args{filter};

  # better to request sorts on "simple" names, but it means we need to map it to real columns
  my %sortmap = (user => 'u.username', type => 'u.type', group => 'g.group_name', status => 'u.status',
	fname => 'fname');
  $args{sortby} = $sortmap{$args{sortby}};

  # protection against bad or missing arguments
  $args{sortorder} = 'ASC' if !$args{sortorder} || !grep /^$args{sortorder}$/, ('ASC','DESC');
  $args{sortby} = 'u.username' if !$args{sortby} || $args{sortby} !~ /^[\w_.]+$/;
  $args{offset} = 0 if !$args{offset} || $args{offset} !~ /^(?:all|\d+)$/;

  my $sql = "SELECT u.user_id, u.username, u.firstname || ' ' || u.lastname AS fname, u.type, g.group_name, u.status ".
	"FROM users u ".
	"INNER JOIN groups g ON u.group_id=g.group_id ".
	"WHERE u.group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND u.username ~* ?" : '').
	($args{filter} ? " AND u.username ~* ?" : '').
	" AND NOT u.type = 'R' ".
	" ORDER BY $args{sortby} $args{sortorder} ".
	($args{offset} eq 'all' ? '' : " LIMIT $self->{perpage} OFFSET ".$args{offset}*$self->{perpage});
  my $ulist = $dbh->selectall_arrayref($sql, { Slice => {} }, (@filterargs) );
  $errstr = $dbh->errstr if !$ulist;
  return $ulist;
} # end getUserList()


## DNSDB::getUserDropdown()
# Get a list of usernames for use in a dropdown menu.
# Takes a database handle, current group, and optional "tag this as selected" flag.
# Returns a reference to a list of hashrefs suitable to feeding to HTML::Template
sub getUserDropdown {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $grp = shift;
  my $sel = shift || 0;

  my $sth = $dbh->prepare("SELECT username,user_id FROM users WHERE group_id=?");
  $sth->execute($grp);

  my @userlist;
  while (my ($username,$uid) = $sth->fetchrow_array) {
    my %row = (
	username => $username,
	uid => $uid,
	selected => ($sel == $uid ? 1 : 0)
	);
    push @userlist, \%row;
  }
  return \@userlist;
} # end getUserDropdown()


## DNSDB:: updateUser()
# Update general data about user
sub updateUser {
  my $self = shift;
  my $dbh = $self->{dbh};

##fixme:  tweak calling convention so that we can update any given bit of data
  my $uid = shift;
  my $username = shift;
  my $group = shift;
  my $pass = shift;
  my $state = shift;
  my $type = shift || 'u';
  my $fname = shift || $username;
  my $lname = shift || '';
  my $phone = shift || '';	# not going format-check

  my $resultmsg = '';

  # Munge in some alternate state values
  $state = 1 if $state =~ /^active$/;
  $state = 1 if $state =~ /^on$/;
  $state = 0 if $state =~ /^inactive$/;
  $state = 0 if $state =~ /^off$/;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  my $sth;

  # Password can be left blank;  if so we assume there's one on file.
  # Actual blank passwords are bad, mm'kay?
  if (!$pass) {
    ($pass) = $dbh->selectrow_array("SELECT password FROM users WHERE user_id=?", undef, ($uid));
  } else {
    $pass = unix_md5_crypt($pass);
  }

  eval {
    $dbh->do("UPDATE users SET username=?, password=?, firstname=?, lastname=?, phone=?, type=?, status=?".
	" WHERE user_id=?", undef, ($username, $pass, $fname, $lname, $phone, $type, $state, $uid));
    $resultmsg = "Updated user info for $username ($fname $lname)";
    $self->_log(group_id => $group, entry => $resultmsg);
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $self->_log(group_id => $group, entry => "Error updating user $username: $msg");
      $dbh->commit;	# since we enabled transactions earlier
    }
    return ('FAIL',"Error updating user $username: $msg");
  }

  return ('OK',$resultmsg);
} # end updateUser()


## DNSDB::delUser()
# Delete a user.
# Takes a database handle and user ID
# Returns a success/failure code and matching message
sub delUser {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $userid = shift;

  return ('FAIL',"Bad userid") if !defined($userid);

  my $userdata = $self->getUserData($userid);

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  eval {
    $dbh->do("DELETE FROM users WHERE user_id=?", undef, ($userid));
    $self->_log(group_id => $userdata->{group_id},
	entry => "Deleted user ID $userid/".$userdata->{username}.
		" (".$userdata->{firstname}." ".$userdata->{lastname}.")");
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $self->_log(group_id => $userdata->{group_id}, entry => "Error deleting user ID ".
	"$userid/".$userdata->{username}.": $msg");
      $dbh->commit;
    }
    return ('FAIL',"Error deleting user $userid/".$userdata->{username}.": $msg");
  }

  return ('OK',"Deleted user ".$userdata->{username}." (".$userdata->{firstname}." ".$userdata->{lastname}.")");
} # end delUser


## DNSDB::userFullName()
# Return a pretty string!
# Takes a user_id and optional printf-ish string to indicate which pieces where:
# %u for the username
# %f for the first name
# %l for the last name
# All other text in the passed string will be left as-is.
##fixme:  need a "smart" option too, so that missing/null/blank first/last names don't give funky output
sub userFullName {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $userid = shift;
  my $fullformat = shift || '%f %l (%u)';
  my $sth = $dbh->prepare("select username,firstname,lastname from users where user_id=?");
  $sth->execute($userid);
  my ($uname,$fname,$lname) = $sth->fetchrow_array();
  $errstr = $DBI::errstr if !$uname;

  $fullformat =~ s/\%u/$uname/g;
  $fullformat =~ s/\%f/$fname/g;
  $fullformat =~ s/\%l/$lname/g;

  return $fullformat;
} # end userFullName


## DNSDB::userStatus()
# Sets and/or returns a user's status
# Takes a database handle, user ID and optionally a status argument
# Returns undef on errors.
sub userStatus {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $id = shift;
  my $newstatus = shift || 'mu';

  return undef if $id !~ /^\d+$/;

  my $userdata = $self->getUserData($id);

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  if ($newstatus ne 'mu') {
    # ooo, fun!  let's see what we were passed for status
    eval {
      $newstatus = 0 if $newstatus eq 'useroff';
      $newstatus = 1 if $newstatus eq 'useron';
      $dbh->do("UPDATE users SET status=? WHERE user_id=?", undef, ($newstatus, $id));

      $resultstr = ($newstatus ? 'Enabled' : 'Disabled')." user ".$userdata->{username}.
	" (".$userdata->{firstname}." ".$userdata->{lastname}.")";

      my %loghash;
      $loghash{group_id} = $self->parentID(id => $id, type => 'user');
      $loghash{entry} = $resultstr;
      $self->_log(%loghash);

      $dbh->commit;
    };
    if ($@) {
      my $msg = $@;
      eval { $dbh->rollback; };
      $resultstr = '';
      $errstr = $msg;
##fixme: failure logging?
      return;
    }
  }

  my ($status) = $dbh->selectrow_array("SELECT status FROM users WHERE user_id=?", undef, ($id));
  return $status;
} # end userStatus()


## DNSDB::getUserData()
# Get misc user data for display
sub getUserData {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $uid = shift;

  my $sth = $dbh->prepare("SELECT group_id,username,firstname,lastname,phone,type,status,inherit_perm ".
	"FROM users WHERE user_id=?");
  $sth->execute($uid);
  return $sth->fetchrow_hashref();
} # end getUserData()


## DNSDB::addLoc()
# Add a new location.
# Takes a database handle, group ID, short and long description, and a comma-separated
# list of IP addresses.
# Returns ('OK',<location>) on success, ('FAIL',<failmsg>) on failure
sub addLoc {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $grp = shift;
  my $shdesc = shift;
  my $comments = shift;
  my $iplist = shift;

  # $shdesc gets set to the generated location ID if possible, but these can be de-undefined here.
  $comments = '' if !$comments;
  $iplist = '' if !$iplist;

  my $loc;

  # Generate a location ID.  This is, by spec, a two-character widget.  We'll use [a-z][a-z]
  # for now;  676 locations should satisfy all but the largest of the huge networks.
  # Not sure whether these are case-sensitive, or what other rules might apply - in any case
  # the absolute maximum is 16K (256*256) since it's parsed by tinydns as a two-character field.

  # and just to be as clear as possible;  as per http://cr.yp.to/djbdns/tinydns-data.html:

#For versions 1.04 and above: You may include a client location on each line. The line is ignored for clients 
#outside that location. Client locations are specified by % lines:
#
#     %lo:ipprefix
#
#means that IP addresses starting with ipprefix are in location lo. lo is a sequence of one or two ASCII letters.

  # so locations could probably include both upper and lower case.

# add just after "my $origloc = $loc;":
#    # These expand the possible space from 26^2 to 52^2 [* note in testing only 2052 were achieved],
#    # and wrap it around.
#    # Yes, they skip a couple of possibles.  No, I don't care.
#    $loc = 'aA' if $loc eq 'zz';
#    $loc = 'Aa' if $loc eq 'zZ';
#    $loc = 'ZA' if $loc eq 'Zz';
#    $loc = 'aa' if $loc eq 'ZZ';

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

##fixme:  There is probably a far better way to do this.  Sequential increments
# are marginally less stupid that pure random generation though, and the existence
# check makes sure we don't stomp on an imported one.

  eval {
    # Get the "last" location.  Note this is the only use for loc_id, because selecting on location Does Funky Things
    ($loc) = $dbh->selectrow_array("SELECT location FROM locations ORDER BY loc_id DESC LIMIT 1");
    ($loc) = ($loc =~ /^(..)/) if $loc;
    my $origloc = $loc;
    $loc = 'aa' if !$loc;	
    # Make a change...
    $loc++;
    # ... and keep changing if it exists
    while ($dbh->selectrow_array("SELECT count(*) FROM locations WHERE location LIKE ?", undef, ($loc.'%'))) {
      $loc++;
      ($loc) = ($loc =~ /^(..)/);
      die "too many locations in use, can't add another one\n" if $loc eq $origloc;
##fixme: really need to handle this case faster somehow
#if $loc eq $origloc die "<thwap> bad admin:  all locations used, your network is too fragmented";
    }
    # And now we should have a unique location.  tinydns fundamentally limits the
    # number of these but there's no doc on what characters are valid.
    $shdesc = $loc if !$shdesc;
    $dbh->do("INSERT INTO locations (location, group_id, iplist, description, comments) VALUES (?,?,?,?,?)", 
	undef, ($loc, $grp, $iplist, $shdesc, $comments) );
    $self->_log(entry => "Added location ($shdesc, '$iplist')");
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $shdesc = $loc if !$shdesc;
      $self->_log(entry => "Failed adding location ($shdesc, '$iplist'): $msg");
      $dbh->commit;
    }
    return ('FAIL',$msg);
  }

  return ('OK',$loc);
} # end addLoc()


## DNSDB::updateLoc()
# Update details of a location.
# Takes a database handle, location ID, group ID, short description,
# long comments/notes, and comma/space-separated IP list
# Returns a result code and message
sub updateLoc {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $loc = shift;
  my $grp = shift;
  my $shdesc = shift;
  my $comments = shift;
  my $iplist = shift;

  $shdesc = '' if !$shdesc;
  $comments = '' if !$comments;
  $iplist = '' if !$iplist;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  my $oldloc = $self->getLoc($loc);
  my $okmsg = "Updated location (".$oldloc->{description}.", '".$oldloc->{iplist}."') to ($shdesc, '$iplist')";

  eval {
    $dbh->do("UPDATE locations SET group_id=?,iplist=?,description=?,comments=? WHERE location=?",
	undef, ($grp, $iplist, $shdesc, $comments, $loc) );
    $self->_log(entry => $okmsg);
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $shdesc = $loc if !$shdesc;
      $self->_log(entry => "Failed updating location ($shdesc, '$iplist'): $msg");
      $dbh->commit;
    }
    return ('FAIL',$msg);
  }

  return ('OK',$okmsg);
} # end updateLoc()


## DNSDB::delLoc()
sub delLoc {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $loc = shift;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  my $oldloc = $self->getLoc($loc);
  my $olddesc = ($oldloc->{description} ? $oldloc->{description} : $loc);
  my $okmsg = "Deleted location ($olddesc, '".$oldloc->{iplist}."')";

  eval {
    # Check for records with this location first.  Deleting a location without deleting records
    # tagged for that location will render them unpublished without other warning.
    my ($r) = $dbh->selectrow_array("SELECT record_id FROM records WHERE location=? LIMIT 1", undef, ($loc) );
    die "Records still exist in location $olddesc\n" if $r;
    $dbh->do("DELETE FROM locations WHERE location=?", undef, ($loc) );
    $self->_log(entry => $okmsg);
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $self->_log(entry => "Failed to delete location ($olddesc, '$oldloc->{iplist}'): $msg");
      $dbh->commit;
    }
    return ('FAIL', "Failed to delete location ($olddesc, '$oldloc->{iplist}'): $msg");
  }

  return ('OK',$okmsg);
} # end delLoc()


## DNSDB::getLoc()
# Get details about a location/view
# Takes a database handle and location ID.
# Returns a reference to a hash containing the group ID, IP list, description, and comments/notes
sub getLoc {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $loc = shift;

  my $sth = $dbh->prepare("SELECT group_id,iplist,description,comments FROM locations WHERE location=?");
  $sth->execute($loc);
  return $sth->fetchrow_hashref(); 
} # end getLoc()


## DNSDB::getLocCount()
# Get count of locations/views
# Takes a database handle and hash containing at least the current group, and optionally:
# - a reference list of secondary groups
# - a filter string
# - a "Starts with" string
sub getLocCount {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  push @filterargs, $args{filter} if $args{filter};

  my $sql = "SELECT count(*) FROM locations ".
	"WHERE group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND description ~* ?" : '').
	($args{filter} ? " AND description ~* ?" : '');
  my ($count) = $dbh->selectrow_array($sql, undef, (@filterargs) );
  $errstr = $dbh->errstr if !$count;
  return $count;
} # end getLocCount()


## DNSDB::getLocList()
sub getLocList {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # Fail on bad curgroup argument.  There's no sane fallback on this one.
  if (!$args{curgroup} || $args{curgroup} !~ /^\d+$/) {
    $errstr = "Bad or missing curgroup argument";
    return;
  }
  # Fail on bad childlist argument.  This could be sanely ignored if bad, maybe.
  if ($args{childlist} && $args{childlist} !~ /^[\d,]+$/) {
    $errstr = "Bad childlist argument";
    return;
  }

  my @filterargs;
  $args{startwith} = undef if $args{startwith} && $args{startwith} !~ /^(?:[a-z]|0-9)$/;
  push @filterargs, "^$args{startwith}" if $args{startwith};
  push @filterargs, $args{filter} if $args{filter};

  # better to request sorts on "simple" names, but it means we need to map it to real columns
#  my %sortmap = (user => 'u.username', type => 'u.type', group => 'g.group_name', status => 'u.status',
#	fname => 'fname');
#  $args{sortby} = $sortmap{$args{sortby}};

  # protection against bad or missing arguments
  $args{sortorder} = 'ASC' if !$args{sortorder} || !grep /^$args{sortorder}$/, ('ASC','DESC');
  $args{sortby} = 'l.description' if !$args{sortby} || $args{sortby} !~ /^[\w_.]+$/;
  $args{offset} = 0 if !$args{offset} || $args{offset} !~ /^(?:all|\d+)$/;

  my $sql = "SELECT l.location, l.description, l.iplist, g.group_name ".
	"FROM locations l ".
	"INNER JOIN groups g ON l.group_id=g.group_id ".
	"WHERE l.group_id IN ($args{curgroup}".($args{childlist} ? ",$args{childlist}" : '').")".
	($args{startwith} ? " AND l.description ~* ?" : '').
	($args{filter} ? " AND l.description ~* ?" : '').
	" ORDER BY $args{sortby} $args{sortorder} ".
	($args{offset} eq 'all' ? '' : " LIMIT $self->{perpage} OFFSET ".$args{offset}*$self->{perpage});
  my $ulist = $dbh->selectall_arrayref($sql, { Slice => {} }, (@filterargs) );
  $errstr = $dbh->errstr if !$ulist;
  return $ulist;
} # end getLocList()


## DNSDB::getLocDropdown()
# Get a list of location names for use in a dropdown menu.
# Takes a database handle, current group, and optional "tag this as selected" flag.
# Returns a reference to a list of hashrefs suitable to feeding to HTML::Template
sub getLocDropdown {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $grp = shift;
  my $sel = shift || '';

  my $sth = $dbh->prepare(qq(
	SELECT description,location FROM locations
	WHERE group_id=?
	ORDER BY description
	) );
  $sth->execute($grp);

  my @loclist;
  push @loclist, { locname => "(Default/All)", loc => '', selected => ($sel ? 0 : ($sel eq '' ? 1 : 0)) };
  while (my ($locname, $loc) = $sth->fetchrow_array) {
    my %row = (
	locname => $locname,
	loc => $loc,
	selected => ($sel eq $loc ? 1 : 0)
	);
    push @loclist, \%row;
  }
  return \@loclist;
} # end getLocDropdown()


## DNSDB::getSOA()
# Return all suitable fields from an SOA record in separate elements of a hash
# Takes a database handle, default/live flag, domain/reverse flag, and parent ID
sub getSOA {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $def = shift;
  my $rev = shift;
  my $id = shift;

  # (ab)use distance and weight columns to store SOA data?  can't for default_rev_records...
  # - should really attach serial to the zone parent somewhere

  my $sql = "SELECT record_id,host,val,ttl from "._rectable($def,$rev).
	" WHERE "._recparent($def,$rev)." = ? AND type=$reverse_typemap{SOA}";
  my $ret = $dbh->selectrow_hashref($sql, undef, ($id) );
  return if !$ret;
##fixme:  stick a flag somewhere if the record doesn't exist.  by the API, this is an impossible case, but...

  ($ret->{contact},$ret->{prins}) = split /:/, $ret->{host};
  delete $ret->{host};
  ($ret->{refresh},$ret->{retry},$ret->{expire},$ret->{minttl}) = split /:/, $ret->{val};
  delete $ret->{val};

  return $ret;
} # end getSOA()


## DNSDB::updateSOA()
# Update the specified SOA record
# Takes a database handle, default/live flag, forward/reverse flag, and SOA data hash
# Returns a two-element list with a result code and message
sub updateSOA {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $defrec = shift;
  my $revrec = shift;

  my %soa = @_;

  my $oldsoa = $self->getSOA($defrec, $revrec, $soa{id});

  my $msg;
  my %logdata;
  if ($defrec eq 'n') {
    $logdata{domain_id} = $soa{id} if $revrec eq 'n';
    $logdata{rdns_id} = $soa{id} if $revrec eq 'y';
    $logdata{group_id} = $self->parentID(id => $soa{id}, revrec => $revrec,
	type => ($revrec eq 'n' ? 'domain' : 'revzone') );
  } else {
    $logdata{group_id} = $soa{id};
  }
  my $parname = ($defrec eq 'y' ? $self->groupName($soa{id}) :
		($revrec eq 'n' ? $self->domainName($soa{id}) : $self->revName($soa{id})) );

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  eval {
    if (!$oldsoa) {
      # old SOA record is missing for some reason.  create a new one.
      my $sql = "INSERT INTO "._rectable($defrec, $revrec)." ("._recparent($defrec, $revrec).
        ", host, type, val, ttl) VALUES (?,?,6,?,?)";
      $dbh->do($sql, undef, ($soa{id}, "$soa{contact}:$soa{prins}",
        "$soa{refresh}:$soa{retry}:$soa{expire}:$soa{minttl}", $soa{ttl}) );
      $msg = ($defrec eq 'y' ? ($revrec eq 'y' ? 'Default reverse ' : 'Default ') : '').
        "SOA missing for $parname;  added (ns $soa{prins}, contact $soa{contact}, refresh $soa{refresh},".
        " retry $soa{retry}, expire $soa{expire}, minTTL $soa{minttl}, TTL $soa{ttl})";
    } else {
      my $sql = "UPDATE "._rectable($defrec, $revrec)." SET host=?, val=?, ttl=? WHERE record_id=? AND type=6";
      $dbh->do($sql, undef, ("$soa{contact}:$soa{prins}", "$soa{refresh}:$soa{retry}:$soa{expire}:$soa{minttl}",
        $soa{ttl}, $oldsoa->{record_id}) );
      $msg = "Updated ".($defrec eq 'y' ? ($revrec eq 'y' ? 'default reverse ' : 'default ') : '').
	"SOA for $parname: ".
	"(ns $oldsoa->{prins}, contact $oldsoa->{contact}, refresh $oldsoa->{refresh},".
	" retry $oldsoa->{retry}, expire $oldsoa->{expire}, minTTL $oldsoa->{minttl}, TTL $oldsoa->{ttl}) to ".
	"(ns $soa{prins}, contact $soa{contact}, refresh $soa{refresh},".
	" retry $soa{retry}, expire $soa{expire}, minTTL $soa{minttl}, TTL $soa{ttl})";
    }
    $logdata{entry} = $msg;
    $self->_log(%logdata);

    $dbh->commit;
  };
  if ($@) {
    $msg = $@;
    eval { $dbh->rollback; };
    $logdata{entry} = "Error updating ".($defrec eq 'y' ? ($revrec eq 'y' ? 'default reverse zone ' : 'default ') : '').
	"SOA record for $parname: $msg";
    if ($self->{log_failures}) {
      $self->_log(%logdata);
      $dbh->commit;
    }
    return ('FAIL', $logdata{entry});
  } else {
    return ('OK', $msg);
  }
} # end updateSOA()


## DNSDB::getRecLine()
# Return all data fields for a zone record in separate elements of a hash
# Takes a database handle, default/live flag, forward/reverse flag, and record ID
sub getRecLine {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $defrec = shift;
  my $revrec = shift;
  my $id = shift;

##fixme: do we need a knob to twist to switch between unix epoch and postgres time string?
  my $sql = "SELECT record_id,host,type,val,ttl".
	($defrec eq 'n' ? ',location' : '').
	($revrec eq 'n' ? ',distance,weight,port' : '').
	(($defrec eq 'y') ? ',group_id FROM ' : ',domain_id,rdns_id,stamp,stamp < now() AS ispast,expires,stampactive FROM ').
	_rectable($defrec,$revrec)." WHERE record_id=?";
  my $ret = $dbh->selectrow_hashref($sql, undef, ($id) );

  if ($dbh->err) {
    $errstr = $DBI::errstr;
    return undef;
  }

  if (!$ret) {
    $errstr = "No such record";
    return undef;
  }

  # explicitly set a parent id
  if ($defrec eq 'y') {
    $ret->{parid} = $ret->{group_id};
  } else {
    $ret->{parid} = (($revrec eq 'n') ? $ret->{domain_id} : $ret->{rdns_id});
    # and a secondary if we have a custom type that lives in both a forward and reverse zone
    $ret->{secid} = (($revrec eq 'y') ? $ret->{domain_id} : $ret->{rdns_id}) if $ret->{type} > 65279;
  }
  $ret->{address} = $ret->{val};	# because.

  return $ret;
}


##fixme: should use above (getRecLine()) to get lines for below?
## DNSDB::getRecList()
# Return records for a group or zone
# Takes a default/live flag, group or zone ID, start,
# number of records, sort field, and sort order
# Returns a reference to an array of hashes
sub getRecList {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  $args{revrec} = 'n' if !$args{revrec};
  $args{defrec} = 'n' if !$args{defrec};

  # protection against bad or missing arguments
  $args{sortorder} = 'ASC' if !$args{sortorder} || !grep /^$args{sortorder}$/, ('ASC','DESC');
  my $defsort;
  $defsort = 'host' if $args{revrec} eq 'n';     # default sort by host on domain record list
  $defsort = 'val' if $args{revrec} eq 'y';      # default sort by IP on revzone record list
  $args{sortby} = '' if !$args{sortby};
  $args{sortby} = $defsort if !$args{revrec};
  $args{sortby} = $defsort if $args{sortby} !~ /^[\w_,.]+$/;
  $args{offset} = 0 if !$args{offset} || $args{offset} !~ /^(?:all|\d+)$/;  
  my $perpage = ($args{nrecs} ? $args{nrecs} : $self->{perpage});


##fixme:  do we need a knob to twist to switch from unix epoch to postgres time string?
  my @bindvars;
  my $sql = "SELECT r.record_id,";
  # only include the parent info if we don't already know which parent we're in
  $sql .= "r.domain_id,r.rdns_id," unless $args{id};
  $sql .= "r.host,r.type,r.val,r.ttl";
  $sql .= ",l.description AS locname,stamp,r.stamp < now() AS ispast,r.expires,r.stampactive"
	if $args{defrec} eq 'n';
  $sql .= ",r.distance,r.weight,r.port" if $args{revrec} eq 'n';
  $sql .= " FROM "._rectable($args{defrec},$args{revrec})." r ";
  $sql .= "INNER JOIN rectypes t ON r.type=t.val ";	# for sorting by type alphabetically
  $sql .= "LEFT JOIN locations l ON r.location=l.location " if $args{defrec} eq 'n';
  $sql .= "WHERE NOT r.type=$reverse_typemap{SOA}";

  # "normal" record list
  if ($args{id}) {
    $sql .= " AND "._recparent($args{defrec},$args{revrec})." = ?";
    push @bindvars, $args{id};
  }

  # Filtering on host/val (mainly normal record list)
  if ($args{filter}) {
    $sql .= " AND (r.host ~* ? OR r.val ~* ? OR r.host ~* ? OR r.val ~* ?)";
    my $tmp = join('.',reverse(split(/\./,$args{filter})));
    push @bindvars, ($args{filter},$args{filter});
    push @bindvars, ($tmp, $tmp);
  }

  # Filtering on other fields
  foreach (qw(type distance weight port ttl description)) {
    if ($args{$_}) {
      $sql .= " AND $_ ~* ?";
      push @bindvars, $args{$_};
    }
  }

  # whee!  multisort means just passing comma-separated fields in sortby!
  my $newsort = '';
  foreach my $sf (split /,/, $args{sortby}) {
    $sf = "r.$sf";
    # sort on IP, correctly
    $sf =~ s/r\.val/inetlazy(r.val)/;
# hmm.  do we really need to limit this?
#	if $args{revrec} eq 'y' && $args{defrec} eq 'n';
    $sf =~ s/r\.type/t.alphaorder/;  # subtly different from sorting on rectypes.name
    $newsort .= ",$sf";
  }
  $newsort =~ s/^,//;
##enhance: pass in ascending/descending sort per-field
  $sql .= " ORDER BY $newsort $args{sortorder}";
  # ensure consistent ordering by sorting on record_id too
  $sql .= ", record_id $args{sortorder}";

  # Offset/pagination
  $sql .= ($args{offset} eq 'all' ? '' : " LIMIT $perpage OFFSET ".$args{offset}*$perpage);

  my @working;
  my $recsth = $dbh->prepare($sql);
  $recsth->execute(@bindvars);
  while (my $rec = $recsth->fetchrow_hashref) {
    if ($args{revrec} eq 'y' && $args{defrec} eq 'n' &&
	($self->{showrev_arpa} eq 'record' || $self->{showrev_arpa} eq 'all') &&
	$rec->{val} !~ /\.arpa$/ ) {
      # skip all reverse zone .arpa "hostnames" since they're already .arpa names.
##enhance:  extend {showrev_arpa} eq 'record' to specify record types
      my $tmpip = new NetAddr::IP $rec->{val} if $rec->{val} =~ /^(?:[\d.\/]+|[a-fA-F0-9:\/]+)$/;
      $rec->{val} = DNSDB::_ZONE($tmpip, 'ZONE', 'r', '.').($tmpip->{isv6} ? '.ip6.arpa' : '.in-addr.arpa') if $tmpip;
    }
    push @working, $rec;
  }
  return \@working;
} # end getRecList()


## DNSDB::getRecCount()
# Return count of non-SOA records in zone (or default records in a group)
# Takes a database handle, default/live flag, reverse/forward flag, group/domain ID,
# and optional filtering modifier
# Returns the count
sub getRecCount {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  $args{defrec} = 'n' if !$args{defrec};
  $args{revrec} = 'n' if !$args{revrec};

  my @bindvars;
  my $sql = "SELECT count(*) FROM ".
	_rectable($args{defrec},$args{revrec}).
	" r WHERE NOT type=$reverse_typemap{SOA}";
  if ($args{id}) {
    $sql .= " AND "._recparent($args{defrec},$args{revrec})." = ?";
    push @bindvars, $args{id};
  }

  # Filtering on host/val (mainly normal record list)
  if ($args{filter}) {
    $sql .= " AND (r.host ~* ? OR r.val ~* ? OR r.host ~* ? OR r.val ~* ?)";
    my $tmp = join('.',reverse(split(/\./,$args{filter})));
    push @bindvars, ($args{filter},$args{filter});
    push @bindvars, ($tmp, $tmp);
  }

  # Filtering on other fields
  foreach (qw(type distance weight port ttl description)) {
    if ($args{$_}) {
      $sql .= " AND $_ ~* ?";
      push @bindvars, $args{$_};
    }
  }

  my ($count) = $dbh->selectrow_array($sql, undef, (@bindvars) );

  return $count;

} # end getRecCount()


## DNSDB::addRec()
# Add a new record to a domain or a group's default records
# Takes a database handle, default/live flag, group/domain ID,
# host, type, value, and TTL
# Some types require additional detail: "distance" for MX and SRV,
# and weight/port for SRV
# Returns a status code and detail message in case of error
##fixme:  pass a hash with the record data, not a series of separate values
sub addRec {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $defrec = shift;
  my $revrec = shift;
  my $id = shift;	# parent (group_id for defrecs, rdns_id for reverse records,
			# domain_id for domain records)

  my $host = shift;
  my $rectype = shift;	# reference so we can coerce it if "+"-types can't find both zones
  my $val = shift;
  my $ttl = shift;
  my $location = shift;
  $location  = '' if !$location;

  my $expires = shift || '';
  $expires = 1 if $expires eq 'until';	# Turn some special values into the appropriate booleans.
  $expires = 0 if $expires eq 'after';
  my $stamp = shift;
  $stamp = '' if !$stamp;	 # Timestamp should be a string at this point.

  # extra safety net - apparently RPC can squeak this by.  O_o
  return ('FAIL', "host must contain a value") if !$$host;
  return ('FAIL', "val must contain a value") if !$$val;

  # Spaces are evil.
  $$host =~ s/^\s+//;
  $$host =~ s/\s+$//;
  if ($typemap{$$rectype} ne 'TXT') {
    # Leading or trailng spaces could be legit in TXT records.
    $$val =~ s/^\s+//;
    $$val =~ s/\s+$//;
  }

  _caseclean($rectype, $host, $val, $defrec, $revrec) if $self->{lowercase};

  # prep for validation
  my $addr = NetAddr::IP->new($$val) if _maybeip($val);
  $$host =~ s/\.+$//;	# FQDNs ending in . are an internal detail, and really shouldn't be exposed in the UI.

  my $domid = 0;
  my $revid = 0;

  my $retcode = 'OK';	# assume everything will go OK
  my $retmsg = '';

  # do simple validation first
  return ('FAIL', "TTL must be numeric") unless $ttl =~ /^-?\d+$/;

  # Collect these even if we're only doing a simple A record so we can call *any* validation sub
  my $dist = shift;
  my $weight = shift;
  my $port = shift;

  my $fields;
  my @vallist;

  # Call the validation sub for the type requested.
  ($retcode,$retmsg) = $validators{$$rectype}($self, defrec => $defrec, revrec => $revrec, id => $id,
	host => $host, rectype => $rectype, val => $val, addr => $addr,
	dist => \$dist, port => \$port, weight => \$weight,
	fields => \$fields, vallist => \@vallist);

  return ($retcode,$retmsg) if $retcode eq 'FAIL';

  # Set up database fields and bind parameters
  $fields .= "host,type,val,ttl,"._recparent($defrec,$revrec);
  push @vallist, ($$host,$$rectype,$$val,$ttl,$id);

  if ($defrec eq 'n') {
    # locations are not for default records, silly coder!
    $fields .= ",location";
    push @vallist, $location;
    # timestamps are rare.
    if ($stamp) {
      $fields .= ",stamp,expires,stampactive";
      push @vallist, $stamp, $expires, 'y';
    } else {
      $fields .= ",stampactive";
      push @vallist, 'n';
    }
  }

  # a little magic to get the right number of ? placeholders based on how many values we're providing
  my $vallen = '?'.(',?'x$#vallist);

  # Put together the success log entry.  We have to use this horrible kludge
  # because domain_id and rdns_id may or may not be present, and if they are,
  # they're not at a guaranteed consistent index in the array.  wheee!
  my %logdata;
  my @ftmp = split /,/, $fields;
  for (my $i=0; $i <= $#vallist; $i++) {
    $logdata{domain_id} = $vallist[$i] if $ftmp[$i] eq 'domain_id';
    $logdata{rdns_id} = $vallist[$i] if $ftmp[$i] eq 'rdns_id';
  }
  $logdata{group_id} = $id if $defrec eq 'y';
  $logdata{group_id} = $self->parentID(id => $id, type => ($revrec eq 'n' ? 'domain' : 'revzone'), revrec => $revrec)
	if $defrec eq 'n';
  $logdata{entry} = "Added ".($defrec eq 'y' ? 'default record' : 'record');
  # Log reverse records to match the formal .arpa tree
  if ($revrec eq 'y') {
    $logdata{entry} .= " '$$val $typemap{$$rectype} $$host";
  } else {
    $logdata{entry} .= " '$$host $typemap{$$rectype} $$val";
  }

  $logdata{entry} .= " [distance $dist]" if $typemap{$$rectype} eq 'MX';
  $logdata{entry} .= " [priority $dist] [weight $weight] [port $port]"
	if $typemap{$$rectype} eq 'SRV';
  $logdata{entry} .= "', TTL $ttl";
  $logdata{entry} .= ", location ".$self->getLoc($location)->{description} if $location;
  $logdata{entry} .= ($expires ? ', expires at ' : ', valid after ').$stamp if $stamp;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  eval {
    $dbh->do("INSERT INTO "._rectable($defrec, $revrec)." ($fields) VALUES ($vallen)",
	undef, @vallist);
    $self->_log(%logdata);
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $logdata{entry} = "Failed adding ".($defrec eq 'y' ? 'default ' : '').
	"record '$$host $typemap{$$rectype} $$val', TTL $ttl ($msg)";
      $self->_log(%logdata);
      $dbh->commit;
    }
    return ('FAIL',$msg);
  }

  $resultstr = $logdata{entry};
  return ($retcode, $retmsg);

} # end addRec()


## DNSDB::updateRec()
# Update a record
# Takes a database handle, default and reverse flags, record ID, immediate parent ID, and new record data.
# Returns a status code and message
sub updateRec {
  $errstr = '';

  my $self = shift;
  my $dbh = $self->{dbh};

  my $defrec = shift;
  my $revrec = shift;
  my $id = shift;
  my $parid = shift;	# immediate parent entity that we're descending from to update the record

  # all records have these
  my $host = shift;
  my $hostbk = $$host;	# Keep a backup copy of the original, so we can WARN if the update mangles the domain
  my $rectype = shift;
  my $val = shift;
  my $ttl = shift;
  my $location = shift;	# may be empty/null/undef depending on caller
  $location  = '' if !$location;

  my $expires = shift || '';
  $expires = 1 if $expires eq 'until';	# Turn some special values into the appropriate booleans.
  $expires = 0 if $expires eq 'after';
  my $stamp = shift;
  $stamp = '' if !$stamp;	 # Timestamp should be a string at this point.

  # just set it to an empty string;  failures will be caught later.
  $$host = '' if !$$host;

  # Spaces are evil.
  $$host =~ s/^\s+//;
  $$host =~ s/\s+$//;
  if ($typemap{$$rectype} ne 'TXT') {
    # Leading or trailng spaces could be legit in TXT records.
    $$val =~ s/^\s+//;
    $$val =~ s/\s+$//;
  }

  _caseclean($rectype, $host, $val, $defrec, $revrec) if $self->{lowercase};

  # prep for validation
  my $addr = NetAddr::IP->new($$val) if _maybeip($val);
  $$host =~ s/\.+$//;	# FQDNs ending in . are an internal detail, and really shouldn't be exposed in the UI.

  my $domid = 0;
  my $revid = 0;

  my $retcode = 'OK';	# assume everything will go OK
  my $retmsg = '';

  # do simple validation first
  return ('FAIL', "TTL must be numeric") unless $ttl =~ /^-?\d+$/;

  # only MX and SRV will use these
  my $dist = shift || 0;
  my $weight = shift || 0;
  my $port = shift || 0;

  my $fields;
  my @vallist;

  # get old record data so we have the right parent ID
  # and for logging (eventually)
  my $oldrec = $self->getRecLine($defrec, $revrec, $id);

  # Call the validation sub for the type requested.
  # Note the ID to pass here is the *parent*, not the record
  ($retcode,$retmsg) = $validators{$$rectype}($self, defrec => $defrec, revrec => $revrec,
	id => ($defrec eq 'y' ? $oldrec->{group_id} : ($revrec eq 'n' ? $oldrec->{domain_id} : $oldrec->{rdns_id})),
	host => $host, rectype => $rectype, val => $val, addr => $addr,
	dist => \$dist, port => \$port, weight => \$weight,
	fields => \$fields, vallist => \@vallist,
	update => $id);

  return ($retcode,$retmsg) if $retcode eq 'FAIL';

  # Set up database fields and bind parameters.  Note only the optional fields
  # (distance, weight, port, secondary parent ID) are added in the validation call above
  $fields .= "host,type,val,ttl,"._recparent($defrec,$revrec);
  push @vallist, ($$host,$$rectype,$$val,$ttl,
	($defrec eq 'y' ? $oldrec->{group_id} : ($revrec eq 'n' ? $oldrec->{domain_id} : $oldrec->{rdns_id})) );

  if ($defrec eq 'n') {
    # locations are not for default records, silly coder!
    $fields .= ",location";
    push @vallist, $location;
    # timestamps are rare.
    if ($stamp) {
      $fields .= ",stamp,expires,stampactive";
      push @vallist, $stamp, $expires, 'y';
    } else {
      $fields .= ",stampactive";
      push @vallist, 'n';
    }
  }

  # hack hack PTHUI
  # need to forcibly make sure we disassociate a record with a parent it's no longer related to.
  # eg, PTR records may not have a domain parent, or A/AAAA records may not have a revzone parent.
  # needed for crossover types that got coerced down to "standard" types due to data changes
  # need to *avoid* funky records being updated like A/AAAA records in revzones, or PTRs in forward zones.
  if ($defrec eq 'n' && $oldrec->{type} > 65000) {
    if ($$rectype == $reverse_typemap{PTR}) {
      $fields .= ",domain_id";
      push @vallist, 0;
    }
    if ($$rectype == $reverse_typemap{A} || $$rectype == $reverse_typemap{AAAA}) {
      $fields .= ",rdns_id";
      push @vallist, 0;
    }
  }
  # fix fat-finger-originated record type changes
  if ($$rectype == 65285) {  # delegation
    $fields .= ",rdns_id" if $revrec eq 'n';
    $fields .= ",domain_id" if $revrec eq 'y';
    push @vallist, 0;
  }
  # ... and now make sure we *do* associate a record with the "calling" parent
  if ($defrec eq 'n') {
    $domid = $parid if $revrec eq 'n';
    $revid = $parid if $revrec eq 'y';
  }

  # Put together the success log entry.  Horrible kludge from addRec() copied as-is since
  # we don't know whether the passed arguments or retrieved values for domain_id and rdns_id
  # will be maintained (due to "not-in-zone" validation changes)
  my %logdata;
  $logdata{domain_id} = $domid;
  $logdata{rdns_id} = $revid;
  my @ftmp = split /,/, $fields;
  for (my $i=0; $i <= $#vallist; $i++) {
    $logdata{domain_id} = $vallist[$i] if $ftmp[$i] eq 'domain_id';
    $logdata{rdns_id} = $vallist[$i] if $ftmp[$i] eq 'rdns_id';
  }
  $logdata{group_id} = $parid if $defrec eq 'y';
  $logdata{group_id} = $self->parentID(id => $parid, type => ($revrec eq 'n' ? 'domain' : 'revzone'), revrec => $revrec)
	if $defrec eq 'n';
  $logdata{entry} = "Updated ".($defrec eq 'y' ? 'default record' : 'record')." from\n";
  # Log reverse records "naturally", since they're stored, um, unnaturally.
  if ($revrec eq 'y') {
    $logdata{entry} .= " '$oldrec->{val} $typemap{$oldrec->{type}} $oldrec->{host}";
  } else {
    $logdata{entry} .= " '$oldrec->{host} $typemap{$oldrec->{type}} $oldrec->{val}";
  }
  $logdata{entry} .= " [distance $oldrec->{distance}]" if $typemap{$oldrec->{type}} eq 'MX';
  $logdata{entry} .= " [priority $oldrec->{distance}] [weight $oldrec->{weight}] [port $oldrec->{port}]"
	if $typemap{$oldrec->{type}} eq 'SRV';
  $logdata{entry} .= "', TTL $oldrec->{ttl}";
  $logdata{entry} .= ", location ".$self->getLoc($oldrec->{location})->{description} if $oldrec->{location};
  $logdata{entry} .= ($oldrec->{expires} ? ', expires at ' : ', valid after ').$oldrec->{stamp}
	if $oldrec->{stampactive};
  $logdata{entry} .= "\nto\n";
  # Log reverse records "naturally", since they're stored, um, unnaturally.
  if ($revrec eq 'y') {
    $logdata{entry} .= "'$$val $typemap{$$rectype} $$host";
  } else {
    $logdata{entry} .= "'$$host $typemap{$$rectype} $$val";
  }
  $logdata{entry} .= " [distance $dist]" if $typemap{$$rectype} eq 'MX';
  $logdata{entry} .= " [priority $dist] [weight $weight] [port $port]" if $typemap{$$rectype} eq 'SRV';
  $logdata{entry} .= "', TTL $ttl";
  $logdata{entry} .= ", location ".$self->getLoc($location)->{description} if $location;
  $logdata{entry} .= ($expires ? ', expires at ' : ', valid after ').$stamp if $stamp;

  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  # Fiddle the field list into something suitable for updates
  $fields =~ s/,/=?,/g;
  $fields .= "=?";

  eval {
    $dbh->do("UPDATE "._rectable($defrec,$revrec)." SET $fields WHERE record_id=?", undef, (@vallist, $id) );
    $self->_log(%logdata);
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $logdata{entry} = "Failed updating ".($defrec eq 'y' ? 'default ' : '').
	"record '$oldrec->{host} $typemap{$oldrec->{type}} $oldrec->{val}', TTL $oldrec->{ttl} ($msg)";
      $self->_log(%logdata);
      $dbh->commit;
    }
    return ('FAIL', $msg);
  }

  $resultstr = $logdata{entry};
  return ($retcode, $retmsg);
} # end updateRec()


## DNSDB::downconvert()
# A mostly internal (not exported) semiutilty sub to downconvert from pseudotype <x>
# to a compatible component type.  Only a handful of operations are valid, anything
# else is a null-op.
# Takes the record ID and the new type.  Returns boolean.
sub downconvert {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $recid = shift;
  my $newtype = shift;

  # also, only work on live records;  little to no value trying to do this on default records.
  my $rec = $self->getRecLine('n', 'y', $recid);

  # hm?
  #return 1 if !$rec;

  return 1 if $rec->{type} < 65000;	# Only the reverse-record pseudotypes can be downconverted
  return 1 if $rec->{type} == 65282;	# Nowhere to go

  my $delpar;
  my @sqlargs;
  if ($rec->{type} == 65280) {
    return 1 if $newtype != 1 && $newtype != 12;
    $delpar = ($newtype == 1 ? 'rdns_id' : 'domain_id');
    push @sqlargs, 0, $newtype, $recid;
  } elsif ($rec->{type} == 65281) {
    return 1 if $newtype != 28 && $newtype != 12;
    $delpar = ($newtype == 28 ? 'rdns_id' : 'domain_id');
    push @sqlargs, 0, $newtype, $recid;
  } elsif ($rec->{type} == 65283) {
    return 1 if $newtype != 65282;
    $delpar = 'rdns_id';
  } elsif ($rec->{type} == 65284) {
    return 1 if $newtype != 65282;
    $delpar = 'rdns_id';
  } else {
    # Your llama is on fire.
  }

  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  eval {
    $dbh->do("UPDATE records SET $delpar = ?, type = ? WHERE record_id = ?", undef, @sqlargs);
    $self->_log(domain_id => $rec->{domain_id}, rdns_id => $rec->{rdns_id},
	group_id => $self->parentID(id => $rec->{rdns_id}, type => 'revzone', revrec => 'y'),
	entry => "'$rec->{host} $typemap{$rec->{type}} $rec->{val}' downconverted to ".
		"'$rec->{host} $typemap{$newtype} $rec->{val}'");
    $dbh->commit;
  };
  if ($@) {
    $errstr = $@;
    eval { $dbh->rollback; };
    return 0;
  }
  return 1;
} # end downconvert()


## DNSDB::delRec()
# Delete a record.  
# Takes a default/live flag, forward/reverse flag, and the ID of the record to delete.
sub delRec {
  $errstr = '';
  my $self = shift;
  my $dbh = $self->{dbh};
  my $defrec = shift;
  my $revrec = shift;
  my $id = shift;

  my $oldrec = $self->getRecLine($defrec, $revrec, $id);

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  # Put together the log entry
  my %logdata;
  $logdata{domain_id} = $oldrec->{domain_id};
  $logdata{rdns_id} = $oldrec->{rdns_id};
  $logdata{group_id} = $oldrec->{group_id} if $defrec eq 'y';
  $logdata{group_id} = $self->parentID(id => ($revrec eq 'n' ? $oldrec->{domain_id} : $oldrec->{rdns_id}), 
	type => 'domain', revrec => $revrec)
	if $defrec eq 'n';
  $logdata{entry} = "Deleted ".($defrec eq 'y' ? 'default record ' : 'record ').
	"'$oldrec->{host} $typemap{$oldrec->{type}} $oldrec->{val}";
  $logdata{entry} .= " [distance $oldrec->{distance}]" if $typemap{$oldrec->{type}} eq 'MX';
  $logdata{entry} .= " [priority $oldrec->{distance}] [weight $oldrec->{weight}] [port $oldrec->{port}]"
	if $typemap{$oldrec->{type}} eq 'SRV';
  $logdata{entry} .= "', TTL $oldrec->{ttl}";
  $logdata{entry} .= ", location ".$self->getLoc($oldrec->{location})->{description} if $oldrec->{location};

  eval {
    my $sth = $dbh->do("DELETE FROM "._rectable($defrec,$revrec)." WHERE record_id=?", undef, ($id));
    $self->_log(%logdata);
    $dbh->commit;
  };
  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    if ($self->{log_failures}) {
      $logdata{entry} = "Error deleting ".($defrec eq 'y' ? 'default record' : 'record').
	" '$oldrec->{host} $typemap{$oldrec->{type}} $oldrec->{val}', TTL $oldrec->{ttl} ($msg)";
      $self->_log(%logdata);
      $dbh->commit;
    }
    return ('FAIL', $msg);
  }

  return ('OK',$logdata{entry});
} # end delRec()


## DNSDB::getLogCount()
# Get a count of log entries
# Takes a database handle and a hash containing at least:
# - Entity ID and entity type as the primary log "slice"
sub getLogCount {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  my @filterargs;
##fixme:  which fields do we want to filter on?
# push @filterargs, 

  $errstr = 'Missing primary parent ID and/or type';
  # fail early if we don't have a "prime" ID to look for log entries for
  return if !$args{id};

  # or if the prime id type is missing or invalid
  return if !$args{logtype};
  $args{logtype} = 'revzone' if $args{logtype} eq 'rdns';	# hack pthui
  $args{logtype} = 'domain' if $args{logtype} eq 'dom';		# hack pthui
  return if !grep /^$args{logtype}$/, ('group', 'domain', 'revzone', 'user');

  my $sql = "SELECT count(*) FROM log ".
	"WHERE $id_col{$args{logtype}}=?".
	($args{filter} ? " AND entry ~* ?" : '');
  my ($count) = $dbh->selectrow_array($sql, undef, ($args{id}, @filterargs) );
  $errstr = $dbh->errstr if !$count;
  return $count;
} # end getLogCount()


## DNSDB::getLogEntries()
# Get a list of log entries
# Takes arguments as with getLogCount() above, plus optional:
# - sort field
# - sort order
# - offset for pagination
sub getLogEntries {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  my @filterargs;

  # fail early if we don't have a "prime" ID to look for log entries for
  return if !$args{id};

  # or if the prime id type is missing or invalid
  return if !$args{logtype};
  $args{logtype} = 'revzone' if $args{logtype} eq 'rdns';	# hack pthui
  $args{logtype} = 'domain' if $args{logtype} eq 'dom';		# hack pthui
  return if !grep /^$args{logtype}$/, ('group', 'domain', 'revzone', 'user');

  # Sorting defaults
  $args{sortorder} = 'DESC' if !$args{sortorder} || !grep /^$args{sortorder}$/, ('ASC','DESC');
  $args{sortby} = 'stamp' if !$args{sortby} || $args{sortby} !~ /^[\w_.]+$/;
  $args{offset} = 0 if !$args{offset} || $args{offset} !~ /^(?:all|\d+)$/;

  my %sortmap = (fname => 'name', username => 'email', entry => 'entry', stamp => 'stamp');
  $args{sortby} = $sortmap{$args{sortby}};

  my $sql = "SELECT user_id AS userid, email AS useremail, name AS userfname, entry AS logentry, ".
	"date_trunc('second',stamp) AS logtime ".
	"FROM log ".
	"WHERE $id_col{$args{logtype}}=?".
	($args{filter} ? " AND entry ~* ?" : '').
	" ORDER BY $args{sortby} $args{sortorder}, log_id $args{sortorder}".
	($args{offset} eq 'all' ? '' : " LIMIT $self->{perpage} OFFSET ".$args{offset}*$self->{perpage});
  my $loglist = $dbh->selectall_arrayref($sql, { Slice => {} }, ($args{id}, @filterargs) );
  $errstr = $dbh->errstr if !$loglist;
  return $loglist;
} # end getLogEntries()


## DNSDB::getRevPattern()
# Get the narrowest template pattern applicable to a passed CIDR address (may be a netblock or an IP)
sub getRevPattern {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $cidr = shift;
  my %args = @_;
  $args{group} = 1 if !$args{group};	# just in case
  $args{location} = '' if !$args{location};

  # for speed!  Casting and comparing even ~7K records takes ~2.5s, so narrow it down to one revzone first.
  my ($revid) = $dbh->selectrow_array("SELECT rdns_id FROM revzones WHERE revnet >>= ?",
	undef, ($cidr) );

##fixme?  may need to narrow things down more by octet-chopping and doing text comparisons before casting.
  my ($revpatt) = $dbh->selectrow_array("SELECT host FROM records ".
        "WHERE (type in (12,65280,65281,65282,65283,65284)) AND rdns_id = ? ".
        "AND location = ? AND inetlazy(val) >>= ? ".
        "ORDER BY inetlazy(val) DESC LIMIT 1",
        undef, ($revid, $args{location}, $cidr) );

  return $revpatt;
} # end getRevPattern()


## DNSDB::getRevSet()
# Return the unique per-IP reverse hostnames, if any, for the passed
# CIDR address (may be a netblock or an IP)
sub getRevSet {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $cidr = shift;
  my %args = @_;
  $args{group} = 1 if !$args{group};	# just in case
  $args{location} = '' if !$args{location};

  # for speed!  Casting and comparing even ~7K records takes ~2.5s, so narrow it down to one revzone first.
  my ($revid) = $dbh->selectrow_array("SELECT rdns_id FROM revzones WHERE revnet >>= ?",
	undef, ($cidr) );

  $cidr = new NetAddr::IP $cidr;
  if ($cidr->num > 256) {	# should also catch v6!
    # Even reverse entries for a v4 /24 of IPs is a bit much.  I don't expect
    # there to be a sane reason to retrive more than a /27 at once, really.
    # v6 is going to be hairy no matter how you slice it.
    $errstr = "Reverse hostname detail range too large";
    return;
  }

  my $sth = $dbh->prepare("SELECT val, host FROM records ".
	"WHERE (type in (12,65280,65281,65282,65283,65284)) AND rdns_id = ? AND location = ? AND inetlazy(val) = ?");

  my @ret;
  foreach my $ip (@{$cidr->splitref()}) {
    $sth->execute($revid, $args{location}, $ip);
    my @data = $sth->fetchrow_array();
    my %row;
    if (@data) {
      $row{r_ip} = $data[0];
      $row{iphost} = $data[1];
    } else {
      $row{r_ip} = $ip->addr;
      $row{iphost} = '';
    }
    push @ret, \%row;
  }

  return \@ret;
} # end getRevSet()


## DNSDB::getTypelist()
# Get a list of record types for various UI dropdowns
# Takes database handle, forward/reverse/lookup flag, and optional "tag as selected" indicator (defaults to A)
# Returns an arrayref to list of hashrefs perfect for HTML::Template
sub getTypelist {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $recgroup = shift;
  my $type = shift || $reverse_typemap{A};

  # also accepting $webvar{revrec}!
  $recgroup = 'f' if $recgroup eq 'n';
  $recgroup = 'r' if $recgroup eq 'y';

  my $sql = "SELECT val,name FROM rectypes WHERE ";
  if ($recgroup eq 'r') {
    # reverse zone types
    $sql .= "stdflag=2 OR stdflag=3";
  } elsif ($recgroup eq 'l') {
    # DNS lookup types.  Note we avoid our custom types >= 65280, since those are entirely internal.
    $sql .= "(stdflag=1 OR stdflag=2 OR stdflag=3) AND val < 65280";
  } else {
    # default;  forward zone types.  technically $type eq 'f' but not worth the error message.
    $sql .= "stdflag=1 OR stdflag=2";
    $sql .= " AND val < 65280" if $recgroup eq 'fo';  # An extra flag to trim off the pseudotypes as well.
  }
  $sql .= " ORDER BY listorder";

  my $sth = $dbh->prepare($sql);
  $sth->execute;
  my @typelist;
  # track whether the passed type is in the list at all.  allows you to edit a record
  # that wouldn't otherwise be generally available in that zone (typically, reverse zones)
  # without changing its type (accidentally or otherwise)
  my $selflag = 0;
  while (my ($rval,$rname) = $sth->fetchrow_array()) {
    my %row = ( recval => $rval, recname => $rname );
    if ($rval == $type) {
      $row{tselect} = 1;
      $selflag = 1;
    }
    push @typelist, \%row;
  }

  # add the passed type if it wasn't in the list
  if (!$selflag) {
    my %row = ( recval => $type, recname => $typemap{$type}, tselect => 1 );
    push @typelist, \%row;
  }

  # Add SOA on lookups since it's not listed in other dropdowns.
  if ($recgroup eq 'l') {
    my %row = ( recval => $reverse_typemap{SOA}, recname => 'SOA' );
    $row{tselect} = 1 if $reverse_typemap{SOA} == $type;
    push @typelist, \%row;
  }

  return \@typelist;
} # end getTypelist()


## DNSDB::parentID()
# Get ID of entity that is nearest parent to requested id
# Takes a database handle and a hash of entity ID, entity type, optional parent type flag
# (domain/reverse zone or group), and optional default/live and forward/reverse flags
# Returns the ID or undef on failure
sub parentID {
  my $self = shift;
  my $dbh = $self->{dbh};

  my %args = @_;

  # clean up defrec and revrec.  default to live record, forward zone
  $args{defrec} = 'n' if !$args{defrec};
  $args{revrec} = 'n' if !$args{revrec};

  # clean up the parent-type.  Set it to group if not set
  $args{partype} = 'group' if !$args{partype};

  # allow callers to be lazy with type
  $args{type} = 'revzone' if $args{type} eq 'domain' && $args{revrec} eq 'y';

  if ($par_type{$args{partype}} eq 'domain' || $par_type{$args{partype}} eq 'revzone') {
    # only live records can have a domain/zone parent
    return unless ($args{type} eq 'record' && $args{defrec} eq 'n');
    my $result = $dbh->selectrow_hashref("SELECT ".($args{revrec} eq 'n' ? 'domain_id' : 'rdns_id').
	" FROM records WHERE record_id = ?",
	undef, ($args{id}) ) or return;
    return $result;
  } else {
    # snag some arguments that will either fall through or be overwritten to save some code duplication
    my $tmpid = $args{id};
    my $type = $args{type};
    if ($type eq 'record' && $args{defrec} eq 'n') {
      # Live records go through the records table first.
      ($tmpid) = $dbh->selectrow_array("SELECT ".($args{revrec} eq 'n' ? 'domain_id' : 'rdns_id').
	" FROM records WHERE record_id = ?",
	undef, ($args{id}) ) or return;
      $type = ($args{revrec} eq 'n' ? 'domain' : 'revzone');
    }
    my ($result) = $dbh->selectrow_array("SELECT $par_col{$type} FROM $par_tbl{$type} WHERE $id_col{$type} = ?",
	undef, ($tmpid) );
    return $result;
  }
# should be impossible to get here with even remotely sane arguments
  return;
} # end parentID()


## DNSDB::isParent()
# Returns true if $id1 is a parent of $id2, false otherwise
sub isParent {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $id1 = shift;
  my $type1 = shift; 
  my $id2 = shift;
  my $type2 = shift;
##todo:  immediate, secondary, full (default)

  # Return false on invalid types
  return 0 if !grep /^$type1$/, ('record','defrec','defrevrec','user','domain','revzone','group');
  return 0 if !grep /^$type2$/, ('record','defrec','defrevrec','user','domain','revzone','group');

  # Return false on impossible relations
  return 0 if $type1 eq 'record';	# nothing may be a child of a record
  return 0 if $type1 eq 'defrec';	# nothing may be a child of a record
  return 0 if $type1 eq 'defrevrec';	# nothing may be a child of a record
  return 0 if $type1 eq 'user';		# nothing may be child of a user
  return 0 if $type1 eq 'domain' && $type2 ne 'record';	# domain may not be a parent of anything other than a record
  return 0 if $type1 eq 'revzone' && $type2 ne 'record';# reverse zone may not be a parent of anything other than a record

  # ennnhhhh....  if we're passed an id of 0, it will never be found.  usual
  # case would be the UI creating a new <thing>, and so we don't have an ID for
  # <thing> to look up yet.  in that case the UI should check the parent as well.
  return 0 if $id1 == 0;	# nothing can have a parent id of 0
  return 1 if $id2 == 0;	# anything could have a child id of 0 (or "unknown")

  # group 1 is the ultimate root parent
  return 1 if $type1 eq 'group' && $id1 == 1;

  # groups are always (a) parent of themselves
  return 1 if $type1 eq 'group' && $type2 eq 'group' && $id1 == $id2;

  my $id = $id2;
  my $type = $type2;
  my $foundparent = 0;

  # Records are the only entity with two possible parents.  We need to split the parent checks on
  # domain/rdns.
  if ($type eq 'record') {
    my ($dom,$rdns) = $dbh->selectrow_array("SELECT domain_id,rdns_id FROM records WHERE record_id=?",
	undef, ($id));
    # check immediate parent against request
    return 1 if $type1 eq 'domain' && $id1 == $dom;
    return 1 if $type1 eq 'revzone' && $id1 == $rdns;
    # if request is group, check *both* parents.  Only check if the parent is nonzero though.
    return 1 if $dom && $self->isParent($id1, $type1, $dom, 'domain');
    return 1 if $rdns && $self->isParent($id1, $type1, $rdns, 'revzone');
    # exit here since we've executed the loop below by proxy in the above recursive calls.
    return 0;
  }

# almost the same loop as getParents() above
  my $limiter = 0;
  while (1) {
    my $sql = "SELECT $par_col{$type} FROM $par_tbl{$type} WHERE $id_col{$type} = ?";
    my $result = $dbh->selectrow_hashref($sql,
	undef, ($id) );
    if (!$result) {
      $limiter++;
##fixme:  how often will this happen on a live site?  fail at max limiter <n>?
# 2013/10/22 only seems to happen when you request an entity that doesn't exist.
      warn "no results looking for $sql with id $id (depth $limiter)\n";
      last;
    }
    if ($result && $result->{$par_col{$type}} == $id1) {
      $foundparent = 1;
      last;
    } else {
##fixme: do we care about trying to return a "no such record/domain/user/group" error?
# should be impossible to create an inconsistent DB just with API calls.
      warn $dbh->errstr." $sql, $id" if $dbh->errstr;
    }
    # group 1 is its own parent.  need this here more to break strange loops than for detecting a parent
    last if $result->{$par_col{$type}} == 1;
    $id = $result->{$par_col{$type}};
    $type = $par_type{$type};
  }

  return $foundparent;
} # end isParent()


## DNSDB::zoneStatus()
# Returns and optionally sets a zone's status
# Takes a database handle, domain/revzone ID, forward/reverse flag, and optionally a status argument
# Returns status, or undef on errors.
sub zoneStatus {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $id = shift;
  my $revrec = shift;
  my $newstatus = shift || 'mu';

  return undef if $id !~ /^\d+$/;

  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  if ($newstatus ne 'mu') {
    # ooo, fun!  let's see what we were passed for status
    eval {
      $newstatus = 0 if $newstatus eq 'domoff';
      $newstatus = 1 if $newstatus eq 'domon';
      $dbh->do("UPDATE ".($revrec eq 'n' ? 'domains' : 'revzones')." SET status=? WHERE ".
	($revrec eq 'n' ? 'domain_id' : 'rdns_id')."=?", undef, ($newstatus,$id) );

##fixme  switch to more consise "Enabled <domain"/"Disabled <domain>" as with users?
      $resultstr = "Changed ".($revrec eq 'n' ? $self->domainName($id) : $self->revName($id)).
	" state to ".($newstatus ? 'active' : 'inactive');

      my %loghash;
      $loghash{domain_id} = $id if $revrec eq 'n';
      $loghash{rdns_id} = $id if $revrec eq 'y';
      $loghash{group_id} = $self->parentID(id => $id, type => ($revrec eq 'n' ? 'domain' : 'revzone'), revrec => $revrec);
      $loghash{entry} = $resultstr;
      $self->_log(%loghash);

      $dbh->commit;
    };
    if ($@) {
      my $msg = $@;
      eval { $dbh->rollback; };
      $resultstr = '';
      $errstr = $msg;
      return;
    }
  }

  my ($status) = $dbh->selectrow_array("SELECT status FROM ".
	($revrec eq 'n' ? "domains WHERE domain_id=?" : "revzones WHERE rdns_id=?"),
	undef, ($id) );
  return $status;
} # end zoneStatus()


## DNSDB::getZonesByCIDR()
# Get a list of zone names and IDs that records for a passed CIDR block are within.
sub getZonesByCIDR {
  my $self = shift;
  my $dbh = $self->{dbh};
  my %args = @_;

  my $result = $dbh->selectall_arrayref("SELECT rdns_id,revnet FROM revzones WHERE revnet >>= ? OR revnet <<= ?",
	{ Slice => {} }, ($args{cidr}, $args{cidr}) );
  return $result;
} # end getZonesByCIDR()


## DNSDB::importAXFR
# Import a domain via AXFR
# Takes AXFR host, domain to transfer, group to put the domain in,
# and an optional hash containing:
#  status - active/inactive state flag (defaults to active)
#  rwsoa - overwrite-SOA flag (defaults to off)
#  rwns - overwrite-NS flag (defaults to off, doesn't affect subdomain NS records)
#  merge - flag to automerge A or AAAA records with matching PTR records
# Returns a status code (OK, WARN, or FAIL) and message - message should be blank
# if status is OK, but WARN includes conditions that are not fatal but should
# really be reported.
sub importAXFR {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $ifrom_in = shift;
  my $zone = shift;
  my $group = shift;

  my %args = @_;

##fixme:  add mode to delete&replace, merge+overwrite, merge new?

  $args{status} = (defined($args{status}) ? $args{status} : 0);
  $args{status} = 1 if $args{status} eq 'on'; 

  my $nrecs = 0;
  my $soaflag = 0;
  my $nsflag = 0;
  my $warnmsg = '';
  my $ifrom;

  my $rev = 'n';
  my $code = 'OK';
  my $msg = 'foobar?';

  # choke on possible bad setting in ifrom
  # IPv4 and v6, and valid hostnames!
  ($ifrom) = ($ifrom_in =~ /^([0-9a-f\:.]+|[0-9a-z_.-]+)$/i);
  return ('FAIL', "Bad AXFR source host $ifrom")
	unless ($ifrom) = ($ifrom_in =~ /^([0-9a-f\:.]+|[0-9a-z_.-]+)$/i);

  my $errmsg;

  my $zone_id;
  my $domain_id = 0;
  my $rdns_id = 0;
  my $cidr;

# magic happens!  detect if we're importing a domain or a reverse zone
# while we're at it, figure out what the CIDR netblock is (if we got a .arpa)
# or what the formal .arpa zone is (if we got a CIDR netblock)
# Handles sub-octet v4 zones in the format specified in the Cricket Book, 2nd Ed, p217-218

  if ($zone =~ m{(?:\.arpa\.?|/\d+|^[\d.]+|^[a-fA-F0-9:]+)$}) {
    # we seem to have a reverse zone
    $rev = 'y';

    if ($zone =~ /\.arpa\.?$/) {
      # we have a formal reverse zone.  call _zone2cidr and get the CIDR block.
      ($code,$msg) = _zone2cidr($zone);
      return ($code, $msg) if $code eq 'FAIL';
      $cidr = $msg;
    } elsif ($zone =~ m|^[\d.]+/\d+$|) {
      # v4 revzone, CIDR netblock
      $cidr = NetAddr::IP->new($zone) or return ('FAIL',"$zone is not a valid CIDR block");
      $zone = _ZONE($cidr, 'ZONE.in-addr.arpa', 'r', '.');
    } elsif ($zone =~ /^[\d.]+$/) {
      # v4 revzone, leading-octet format
      my $mask = 32;
      while ($zone !~ /^\d+\.\d+\.\d+\.\d+$/) {
        $zone .= '.0';
        $mask -= 8;
      }
      $zone .= "/$mask";
      $cidr = NetAddr::IP->new($zone) or return ('FAIL',"$zone is not a valid CIDR block");
      $zone = _ZONE($cidr, 'ZONE.in-addr.arpa', 'r', '.');
    } elsif ($zone =~ m|^[a-fA-F\d:]+/\d+$|) {
      # v6 revzone, CIDR netblock
      $cidr = NetAddr::IP->new($zone) or return ('FAIL',"$zone is not a valid CIDR block");
      return ('FAIL', "$zone is not a nibble-aligned block") if $cidr->masklen % 4 != 0;
      $zone = _ZONE($cidr, 'ZONE.ip6.arpa', 'r', '.');
    } elsif ($zone =~ /^[a-fA-F\d:]+$/) {
      # v6 revzone, leading-group format
      $zone =~ s/::$//;
      my $mask = 128;
      while ($zone !~ /^(?:[a-fA-F\d]{1,4}:){7}[a-fA-F\d]$/) {
        $zone .= ":0";
        $mask -= 16;
      }
      $zone .= "/$mask";
      $cidr = NetAddr::IP->new($zone) or return ('FAIL',"$zone is not a valid CIDR block");
      $zone = _ZONE($cidr, 'ZONE.ip6.arpa', 'r', '.');
    } else {
      # there is. no. else!
      return ('FAIL', "Unknown zone name format '$zone'");
    }

    # several places this can be triggered from;  better to do it once.
    $warnmsg .= "Apparent sub-/64 IPv6 reverse zone\n" if $cidr->masklen > 64;

    # quick check to start to see if we've already got one

    ($zone_id) = $dbh->selectrow_array("SELECT rdns_id FROM revzones WHERE revnet=?",
	undef, ("$cidr"));
    $rdns_id = $zone_id;
  } else {
    # default to domain
    ($zone_id) = $dbh->selectrow_array("SELECT domain_id FROM domains WHERE lower(domain) = lower(?)",
	undef, ($zone));
    $domain_id = $zone_id;
  }

  return ('FAIL', ($rev eq 'n' ? 'Domain' : 'Reverse zone')." already exists") if $zone_id;

  # little local utility sub to swap $val and $host for revzone records.
  sub _revswap {
    my $rechost = shift;
    my $recdata = shift;

    if ($rechost =~ /\.in-addr\.arpa\.?$/) {
      $rechost =~ s/\.in-addr\.arpa\.?$//;
      $rechost = join '.', reverse split /\./, $rechost;
    } else {
      $rechost =~ s/\.ip6\.arpa\.?$//;
      my @nibs = reverse split /\./, $rechost;
      $rechost = '';
      my $nc;
      foreach (@nibs) {
#        # fail on multicharacter nibbles;  it's syntactically valid but no standard lookup
#        # will ever reach it, because it doesn't directly represent a real IP address.
#        return ('FAIL', "Invalid reverse v6 entry") if $_ !~ /^.$/;
        $rechost.= $_;
        $rechost .= ":" if ++$nc % 4 == 0 && $nc < 32;
      }
      $rechost .= ":" if $nc < 32 && $rechost !~ /\*$/;	# close netblock records?
##fixme:  there's a case that ends up with a partial entry here:
# ip:add:re:ss::
# can't reproduce after letting it sit overnight after discovery.  :(
#print "$rechost\n";
      # canonicalize with NetAddr::IP
      $rechost = NetAddr::IP->new($rechost)->addr unless $rechost =~ /\*$/;
    }
    return ($recdata,$rechost)
  }


  # Allow transactions, and raise an exception on errors so we can catch it later.
  # Use local to make sure these get "reset" properly on exiting this block
  local $dbh->{AutoCommit} = 0;
  local $dbh->{RaiseError} = 1;

  my $sth;
  eval {

    if ($rev eq 'n') {
##fixme:  serial
      $dbh->do("INSERT INTO domains (domain,group_id,status) VALUES (?,?,?)", undef,
        ($zone, $group, $args{status}) );
      # get domain id so we can do the records
      ($zone_id) = $dbh->selectrow_array("SELECT currval('domains_domain_id_seq')");
      $domain_id = $zone_id;
      $self->_log(group_id => $group, domain_id => $domain_id,
		entry => "[Added ".($args{status} ? 'active' : 'inactive')." domain $zone via AXFR]");
    } else {
##fixme:  serial
      $dbh->do("INSERT INTO revzones (revnet,group_id,status) VALUES (?,?,?)", undef,
        ($cidr,$group,$args{status}) );
      # get revzone id so we can do the records
      ($zone_id) = $dbh->selectrow_array("SELECT currval('revzones_rdns_id_seq')");
      $rdns_id = $zone_id;
      $self->_log(group_id => $group, rdns_id => $rdns_id,
		entry => "[Added ".($args{status} ? 'active' : 'inactive')." reverse zone $cidr via AXFR]");
    }

## bizarre DBI<->Net::DNS interaction bug:
## sometimes a zone will cause an immediate commit-and-exit (sort of) of the while()
## fixed, apparently I was doing *something* odd, but not certain what it was that
## caused a commit instead of barfing

    my $res = Net::DNS::Resolver->new;
    $res->nameservers($ifrom);
    $res->axfr_start($zone)
	or die "Couldn't begin AXFR\n";

    $sth = $dbh->prepare("INSERT INTO records (domain_id,rdns_id,host,type,val,distance,weight,port,ttl)".
	" VALUES (?,?,?,?,?,?,?,?,?)");

    # Stash info about sub-octet v4 revzones here so we don't have
    # to store the CNAMEs used to delegate a suboctet zone
    # $suboct{zone}{ns}[] -> array of nameservers
    # $suboct{zone}{cname}[] -> array of extant CNAMEs (Just In Case someone did something bizarre)
## commented pending actual use of this data.  for now, we'll just
## auto-(re)create the CNAMEs in revzones on export
#    my %suboct;

    while (my $rr = $res->axfr_next()) {

      # Discard out-of-zone records.  After trying for a while to replicate this with
      # *nix-based DNS servers, it appears that only MS DNS is prone to including these
      # in the AXFR data in the first place, and possibly only older versions at that...
      # so it can't be reasonably tested.  Yay Microsoft.
      if ($rr->name !~ /$zone$/i) {
        $warnmsg .= "Discarding out-of-zone record ".$rr->string."\n";
      }

      my $val;
      my $distance = 0;
      my $weight = 0;
      my $port = 0;
      my $logfrag = '';

      # Collect some record parts
      my $type = $rr->type;
      my $host = $rr->name;
      my $ttl = ($args{newttl} ? $args{newttl} : $rr->ttl);	# allow force-override TTLs

      # Info flags for SOA and NS records
      $soaflag = 1 if $type eq 'SOA';
      $nsflag = 1 if $type eq 'NS';

# "Primary" types:
# A, NS, CNAME, SOA, PTR(warn in forward), MX, TXT, AAAA, SRV, A6(ob), SPF
# maybe KEY

# BIND supports:
# [standard]
# A AAAA CNAME MX NS PTR SOA TXT
# [variously experimental, obsolete, or obscure]
# HINFO MB(ex) MD(ob) MF(ob) MG(ex) MINFO(ex) MR(ex) NULL WKS AFSDB(ex) ISDN(ex) RP(ex) RT(ex) X25(ex) PX
# ... if one can ever find the right magic to format them correctly

# Net::DNS supports:
# RRSIG SIG NSAP NS NIMLOC NAPTR MX MR MINFO MG MB LOC ISDN IPSECKEY HINFO
# EID DNAME CNAME CERT APL AFSDB AAAA A DS NXT NSEC3PARAM NSEC3 NSEC KEY
# DNSKEY DLV X25 TXT TSIG TKEY SSHFP SRV SPF SOA RT RP PX PTR NULL APL::AplItem

# nasty big ugly case-like thing here, since we have to do *some* different
# processing depending on the record.  le sigh.

# do the initial processing as if the record was in a forward zone.  If we're
# doing a revzone, we can flip $host and $val as needed, once, after this
# monster if-elsif-...-elsif-else.  This actually simplifies things a lot.

##fixme:  what record types other than TXT can/will have >255-byte payloads?

      if ($type eq 'A') {
	$val = $rr->address;
      } elsif ($type eq 'NS') {
# hmm.  should we warn here if subdomain NS'es are left alone?  OTOH, those should rarely be rewritten anyway.
        next if ($args{rwns} && ($host eq $zone));
	$val = $rr->nsdname;
        $warnmsg .= "Suspect record '".$rr->string."' may not be imported correctly: NS records may not be bare IP addresses\n"
          if $val =~ /^(?:(?:\d+\.){3}\d+|[a-fA-F0-9:]+)$/;
	$nsflag = 1;
      } elsif ($type eq 'CNAME') {
	$val = $rr->cname;
        $warnmsg .= "Suspect record '".$rr->string."' may not be imported correctly: CNAME records may not be bare IP addresses\n"
          if $val =~ /^(?:(?:\d+\.){3}\d+|[a-fA-F0-9:]+)$/;
      } elsif ($type eq 'SOA') {
	next if $args{rwsoa};
	$host = $rr->rname.":".$rr->mname;
	$val = $rr->refresh.":".$rr->retry.":".$rr->expire.":".$rr->minimum;
	$soaflag = 1;
      } elsif ($type eq 'PTR') {
	$val = $rr->ptrdname;
      } elsif ($type eq 'MX') {
	$val = $rr->exchange;
	$distance = $rr->preference;
      } elsif ($type eq 'TXT') {
##fixme:  Net::DNS docs say this should be deprecated for rdatastr() or char_str_list(),
## but don't really seem enthusiastic about it.
#print "should use rdatastr:\n\t".$rr->rdatastr."\n  or char_str_list:\n\t".join(' ',$rr->char_str_list())."\n";
# rdatastr returns a BIND-targetted logical string, including opening and closing quotes
# char_str_list returns a list of the individual string fragments in the record
# txtdata returns the more useful all-in-one form (since we want to push such protocol
# details as far down the stack as we can)
# NB:  this may turn out to be more troublesome if we ever have need of >512-byte TXT records.
	$val = $rr->txtdata;
      } elsif ($type eq 'SPF') {
##fixme: and the same caveat here, since it is apparently a clone of ::TXT
	$val = $rr->txtdata;
      } elsif ($type eq 'AAAA') {
	$val = $rr->address;
      } elsif ($type eq 'SRV') {
	$val = $rr->target;
	$distance = $rr->priority;
	$weight = $rr->weight;
	$port = $rr->port;
        $warnmsg .= "Suspect record '".$rr->string."' may not be imported correctly: SRV records may not be bare IP addresses\n"
          if $val =~ /^(?:(?:\d+\.){3}\d+|[a-fA-F0-9:]+)$/;
      } elsif ($type eq 'KEY') {
	# we don't actually know what to do with these...
	$val = $rr->flags.":".$rr->protocol.":".$rr->algorithm.":".$rr->key.":".$rr->keytag.":".$rr->privatekeyname;
      } else {
	$val = $rr->rdatastr;
	# Finding a different record type is not fatal.... just problematic.
	# We may not be able to export it correctly.
	$warnmsg .= "Unusual record ".$rr->name." ($type) found\n";
      }

      if ($rev eq 'y' && $type ne 'SOA') {
        # up to this point we haven't meddled with the record's hostname part or rdata part.
        # for reverse records, (except SOA) we must swap the two.
        $host = $val;
        $val = $rr->name;
        my ($tmpcode,$tmpmsg) = _zone2cidr($val);
        if ($tmpcode eq 'FAIL') {
          # $val did not have a valid IP value.  It's syntactically valid but WTF?
          $warnmsg .= "Suspect record '".$rr->string."' may not be imported correctly: $tmpmsg\n";
        } else {
          # $val has a valid IP value.  See if we can store it as that IP value.
          # Note we're enumerating do-nothing cases for clarity.
##enhance:  this is where we will implement the more subtle variations on #53
          if ($type ne 'PTR' && $type ne 'NS' && $type ne 'CNAME' && $type ne 'TXT') {
            # case: the record is "weird" - ie, not a PTR, NS, CNAME, or TXT
            # $warnmsg .= "Discarding suspect record '".$rr->string."'\n" if $self->{strict} eq 'full';
          } elsif ($type eq 'PTR' && $tmpmsg->masklen != 32 && $tmpmsg->masklen != 128) {
            # case: PTR with netblock value, not IP value
            # eg, "@ PTR foo" in zone f.e.e.b.d.a.e.d.ip6.arpa should not be
            # stored/displayed as dead:beef::/32 PTR foo

## hrm.  WTF is this case for, anyway?  Needs testing to check the logic.
#          } elsif ( ($type eq 'PTR' || $type eq 'NS' || $type eq 'CNAME' || $type eq 'TXT') &&
#                    ($tmpmsg->masklen != $cidr->masklen) 
#                  ) {
#            # leave $val as-is if the record is "normal" (a PTR, NS, CNAME, or TXT),
#            # and the mask does not match the zone
#$warnmsg .= "WTF case: $host $type $val\n";
#            # $warnmsg .= "Discarding suspect record '".$rr->string."'\n" if $self->{strict} eq 'full';

          } else {
            $val = $tmpmsg;
            $val =~ s/\/(?:32|128)$//;  # automagically converts $val back to a string before s///
            #$val =~ s/:0$//g;
          }
        }
        # magic?  convert * records to PTR template (not sure this actually makes sense)
        #if ($val =~ /^\*/) {
        #  $val =~ s/\*\.//;
        #  ($tmpcode,$tmpmsg) = _zone2cidr($val);
        #  if ($tmpcode eq 'FAIL') {
        #    $val = "*.$val";
        #    $warnmsg .= "Suspect record '".$rr->string."' may not be converted to PTR template correctly: $tmpmsg\n";
        #  } else {
        #    $type = 'PTR template';
        #    $val = $tmpmsg; if $tmp
        #    $val =~ s/\/(?:32|128)$//;  # automagically converts $val back to a string before s///
        #  }
        #}
      } # non-SOA revrec $host/$val inversion and munging

      my $logentry = "[AXFR ".($rev eq 'n' ? $zone : $cidr)."] ";

      if ($args{merge}) {
	if ($rev eq 'n') {
	  # importing a domain;  we have A and AAAA records that could be merged with matching PTR records
	  my $etype;
	  my ($erdns,$erid,$ettl) = $dbh->selectrow_array("SELECT rdns_id,record_id,ttl FROM records ".
		"WHERE host=? AND val=? AND type=12",
		undef, ($host, $val) );
	  if ($erid) {
	    if ($type eq 'A') {	# PTR -> A+PTR
	      $etype = 65280;
	      $logentry .= "Merged A record with existing PTR record '$host A+PTR $val', TTL $ettl";
	    }
	    if ($type eq 'AAAA') {	# PTR -> AAAA+PTR
	      $etype = 65281;
	      $logentry .= "Merged AAAA record with existing PTR record '$host AAAA+PTR $val', TTL $ettl";
	    }
	    $ettl = ($ettl < $ttl ? $ettl : $ttl);    # use lower TTL
	    $dbh->do("UPDATE records SET domain_id=?,ttl=?,type=? WHERE record_id=?", undef,
		($domain_id, $ettl, $etype, $erid));
	    $nrecs++;
	    $self->_log(group_id => $group, domain_id => $domain_id, rdns_id => $erdns, entry => $logentry);
	    next;	# while axfr_next
	  }
	} # $rev eq 'n'
	else {
	  # importing a revzone, we have PTR records that could be merged with matching A/AAAA records
	  my ($domid,$erid,$ettl,$etype) = $dbh->selectrow_array("SELECT domain_id,record_id,ttl,type FROM records ".
		"WHERE host=? AND val=? AND (type=1 OR type=28)",
		undef, ($host, $val) );
	  if ($erid) {
	    if ($etype == 1) {	# A -> A+PTR
	      $etype = 65280;
	      $logentry .= "Merged PTR record with existing matching A record '$host A+PTR $val', TTL $ettl";
	    }
	    if ($etype == 28) {	# AAAA -> AAAA+PTR
	      $etype = 65281;
	      $logentry .= "Merged PTR record with existing matching AAAA record '$host AAAA+PTR $val', TTL $ettl";
	    }
	    $ettl = ($ettl < $ttl ? $ettl : $ttl);    # use lower TTL
	    $dbh->do("UPDATE records SET rdns_id=?,ttl=?,type=? WHERE record_id=?", undef,
		($rdns_id, $ettl, $etype, $erid));
	    $nrecs++;
	    $self->_log(group_id => $group, domain_id => $domid, rdns_id => $rdns_id, entry => $logentry);
	    next;	# while axfr_next
	  }
	} # $rev eq 'y'
      } # if $args{merge}

      # Insert the new record
      $sth->execute($domain_id, $rdns_id, $host, $reverse_typemap{$type}, $val,
	$distance, $weight, $port, $ttl);

      $nrecs++;

      if ($type eq 'SOA') {
	# also !$args{rwsoa}, but if that's set, it should be impossible to get here.
	my @tmp1 = split /:/, $host;
	my @tmp2 = split /:/, $val;
	$logentry .= "Added SOA record [contact $tmp1[0]] [master $tmp1[1]] ".
		"[refresh $tmp2[0]] [retry $tmp2[1]] [expire $tmp2[2]] [minttl $tmp2[3]], TTL $ttl";
      } elsif ($logfrag) {
	# special case for log entries we need to meddle with a little.
	$logentry .= $logfrag;
      } else {
	$logentry .= "Added record '".($rev eq 'y' ? $val : $host)." $type";
	$logentry .= " [distance $distance]" if $type eq 'MX';
	$logentry .= " [priority $distance] [weight $weight] [port $port]" if $type eq 'SRV';
	$logentry .= " ".($rev eq 'y' ? $host : $val)."', TTL $ttl";
      }
      $self->_log(group_id => $group, domain_id => $domain_id, rdns_id => $rdns_id, entry => $logentry);

    } # while axfr_next

# Detect and handle delegated subzones
# Placeholder for when we decide what to actually do with this, see previous comments in NS and CNAME handling.
#foreach (keys %suboct) {
#  print "found ".($suboct{$_}{ns} ? @{$suboct{$_}{ns}} : '0')." NS records and ".
#	($suboct{$_}{cname} ? @{$suboct{$_}{cname}} : '0')." CNAMEs for $_\n";
#}

    # Overwrite SOA record
    if ($args{rwsoa}) {
      $soaflag = 1;
      my $sthgetsoa = $dbh->prepare("SELECT host,val,ttl FROM "._rectable('y', $rev)." WHERE group_id=? AND type=?");
      my $sthputsoa = $dbh->prepare("INSERT INTO records (".
	($rev eq 'n' ? 'domain_id' : 'rdns_id').",host,type,val,ttl) VALUES (?,?,?,?,?)");
      $sthgetsoa->execute($group,$reverse_typemap{SOA});
      while (my ($host,$val,$ttl) = $sthgetsoa->fetchrow_array()) {
	if ($rev eq 'n') {
	  $host =~ s/DOMAIN/$zone/g;
	  $val =~ s/DOMAIN/$zone/g;	# arguably useless
	} else {
	  $host =~ s/ADMINDOMAIN/$self->{domain}/g;
	}
	$sthputsoa->execute($zone_id,$host,$reverse_typemap{SOA},$val,$ttl);
      }
    }

    # Add standard NS records.  The old one(s) should have been skipped by this point.
    if ($args{rwns}) {
      $nsflag = 1;
      my $sthgetns = $dbh->prepare("SELECT host,val,ttl FROM "._rectable('y',$rev)." WHERE group_id=? AND type=?");
      my $sthputns = $dbh->prepare("INSERT INTO records (".
	($rev eq 'n' ? 'domain_id' : 'rdns_id').",host,type,val,ttl) VALUES (?,?,?,?,?)");
      $sthgetns->execute($group,$reverse_typemap{NS});
      while (my ($host,$val,$ttl) = $sthgetns->fetchrow_array()) {
	if ($rev eq 'n') {
	  $host =~ s/DOMAIN/$zone/g;
	  $val =~ s/DOMAIN/$zone/g;	#hmm.
	} else {
	  $host =~ s/ADMINDOMAIN/$self->{domain}/g;	#hmm.
	  $val =~ s/ZONE/$cidr/g;
	}
	$sthputns->execute($zone_id,$host,$reverse_typemap{NS},$val,$ttl);
      }
    }

    die "No records found;  either $ifrom is not authoritative or doesn't allow transfers\n" if !$nrecs;
    die "Bad zone:  No SOA record!\n" if !$soaflag;
    die "Bad zone:  No NS records!\n" if !$nsflag;

    $dbh->commit;

  };

  if ($@) {
    my $msg = $@;
    eval { $dbh->rollback; };
    return ('FAIL',$msg." $warnmsg");
  } else {
    return ('WARN', $warnmsg) if $warnmsg;
    return ('OK',"Imported OK");
  }

  # it should be impossible to get here.
  return ('WARN',"OOOK!");
} # end importAXFR()


## DNSDB::importBIND()
sub importBIND {
} # end importBIND()


## DNSDB::import_tinydns()
sub import_tinydns {
} # end import_tinydns()


## DNSDB::export()
# Export the DNS database, or a part of it
# Takes a string indicating the export type, plus optional arguments depending on type
# Writes zone data to targets as appropriate for type
sub export {
  my $self = shift;
  my $target = shift;

  if ($target eq 'tiny') {
    eval {
      $self->__export_tiny(@_);
    };
    if ($@) {
      $errstr = $@;
      return undef;
    }
  }
# elsif ($target eq 'foo') {
#   __export_foo(@_);
#}
# etc

  return 1;
} # end export()


## DNSDB::__export_tiny
# Internal sub to implement tinyDNS (compatible) export
# Takes filehandle to write export to, optional argument(s)
# to determine which data gets exported
sub __export_tiny {
  my $self = shift;
  my $dbh = $self->{dbh};
  my $datafile = shift;
  my $zonefilehandle = $datafile;	# makes cache/no-cache a little simpler

##fixme: slurp up further options to specify particular zone(s) to export

##fixme: fail if $datafile isn't an open, writable file

  # Error check - does the cache dir exist, if we're using one?
  if ($self->{usecache}) {
    die "Cache directory does not exist\n" if !-e $self->{exportcache};
    die "$self->{exportcache} is not a directory\n" if !-d $self->{exportcache};
    die "$self->{exportcache} must be both readable and writable\n"
        if !-r $self->{exportcache} || !-w $self->{exportcache};
  }

  # easy case - export all evarything
  # not-so-easy case - export item(s) specified
  # todo:  figure out what kind of list we use to export items

# raw packet in unknown format:  first byte indicates length
# of remaining data, allows up to 255 raw bytes

  # note: the only I/O failures we seem to be able to actually catch
  # here are "closed filehandle" errors.  we're probably not writing
  # enough data at this point to properly trigger an "out of space"
  # error.  :/
  eval {
    use warnings FATAL => ('io');
    # Locations/views - worth including in the caching setup?
    my $lochash = $dbh->selectall_hashref("SELECT location,iplist FROM locations", 'location');
    foreach my $location (keys %$lochash) {
      foreach my $ipprefix (split /[,\s]+/, $lochash->{$location}{iplist}) {
        $ipprefix =~ s/\s+//g;
        $ipprefix = new NetAddr::IP $ipprefix;
##fixme:  how to handle IPv6?
next if $ipprefix->{isv6};
        # have to account for /nn CIDR entries.  tinydns only speaks octet-sliced prefix.
        if ($ipprefix->masklen <= 8) {
          foreach ($ipprefix->split(8)) {
            my $tmp = $_->addr;
            $tmp =~ s/\.\d+\.\d+\.\d+$//;
            print $datafile "%$location:$tmp\n";
          }
        } elsif ($ipprefix->masklen <= 16) {
          foreach ($ipprefix->split(16)) {
            my $tmp = $_->addr;
            $tmp =~ s/\.\d+\.\d+$//;
            print $datafile "%$location:$tmp\n";
          }
        } elsif ($ipprefix->masklen <= 24) {
          foreach ($ipprefix->split(24)) {
            my $tmp = $_->addr;
            $tmp =~ s/\.\d+$//;
            print $datafile "%$location:$tmp\n";
          }
        } else {
          foreach ($ipprefix->split(32)) {
            print $datafile "%$location:".$_->addr."\n";
          }
        }
      }
      print $datafile "%$location\n" if !$lochash->{$location}{iplist};
    }
  };
  if ($@) {
    die "Error writing locations to master file: $@, $!\n";
  }

  # tracking hash so we don't double-export A+PTR or AAAA+PTR records.
  my %recflags;

# For reasons unknown, we can't sanely UNION these statements.  Feh.
# Supposedly it should work though (note last 3 lines):
## PG manual
#UNION Clause
#
#The UNION clause has this general form:
#
#    select_statement UNION [ ALL ] select_statement
#
#select_statement is any SELECT statement without an ORDER BY, LIMIT, FOR UPDATE, or FOR SHARE clause. (ORDER BY
#and LIMIT can be attached to a subexpression if it is enclosed in parentheses. Without parentheses, these
#clauses will be taken to apply to the result of the UNION, not to its right-hand input expression.)
  my $soasth = $dbh->prepare("SELECT host,type,val,distance,weight,port,ttl,record_id,location ".
	"FROM records WHERE rdns_id=? AND type=6");
  my $recsth = $dbh->prepare("SELECT host,type,val,distance,weight,port,ttl,record_id,location,extract(epoch from stamp),expires,stampactive ".
	"FROM records WHERE rdns_id=? AND NOT type=6 ".
	"ORDER BY masklen(inetlazy(val)) DESC, inetlazy(val)");
  my $revsth = $dbh->prepare("SELECT rdns_id,revnet,status,changed FROM revzones WHERE status=1 ".
	"ORDER BY masklen(revnet) DESC, rdns_id");
  my $zonesth = $dbh->prepare("UPDATE revzones SET changed='n' WHERE rdns_id=?");
  $revsth->execute();
  while (my ($revid,$revzone,$revstat,$changed) = $revsth->fetchrow_array) {
##fixme: need to find a way to block opening symlinked files without introducing a race.
#       O_NOFOLLOW
#              If  pathname  is a symbolic link, then the open fails.  This is a FreeBSD extension, which was
#              added to Linux in version 2.1.126.  Symbolic links in earlier components of the pathname  will
#              still be followed.
# but that doesn't help other platforms.  :/
    my $tmpzone = NetAddr::IP->new($revzone);
##fixme:  locations/views?  subnet mask?  need to avoid possible collisions with zone/superzone
##        (eg /20 vs /24, starting on .0.0)
    my $cz = $tmpzone->network->addr."-".$tmpzone->masklen;
    my $cachefile = "$self->{exportcache}/$cz";
    my $tmpcache = "$self->{exportcache}/tmp.$cz.$$";
    eval {

      # write fresh records if:
      #  - we are not using the cache
      #  - force_refresh is set
      #  - the zone has changed
      #  - the cache file does not exist
      #  - the cache file is empty
      if (!$self->{usecache} || $self->{force_refresh} || $changed || !-e $cachefile || -z $cachefile) {
        if ($self->{usecache}) {
          open ZONECACHE, ">$tmpcache" or die "Error creating temporary file $tmpcache: $!\n";
          $zonefilehandle = *ZONECACHE;
        }

        # need to fetch this separately since the rest of the records all (should) have real IPs in val
        $soasth->execute($revid);
        my (@zsoa) = $soasth->fetchrow_array();
        $self->_printrec_tiny($zonefilehandle, $zsoa[7], 'y',\%recflags,$revzone,
          $zsoa[0],$zsoa[1],$zsoa[2],$zsoa[3],$zsoa[4],$zsoa[5],$zsoa[6],$zsoa[8],'');

        $recsth->execute($revid);
        my $fullzone = _ZONE($tmpzone, 'ZONE', 'r', '.').($tmpzone->{isv6} ? '.ip6.arpa' : '.in-addr.arpa');

        while (my ($host, $type, $val, $dist, $weight, $port, $ttl, $recid, $loc, $stamp, $expires, $stampactive)
		= $recsth->fetchrow_array) {
          next if $recflags{$recid};

          # Check for out-of-zone data
          if ($val =~ /\.arpa$/) {
            # val is non-IP
            if ($val !~ /$fullzone$/) {
              warn "Not exporting out-of-zone record $val $typemap{$type} $host, $ttl (zone $tmpzone)\n";
              next;
            }
          } else {
            my $ipval = new NetAddr::IP $val;
            if (!$tmpzone->contains($ipval)) {
              warn "Not exporting out-of-zone record $val $typemap{$type} $host, $ttl (zone $tmpzone)\n";
              next;
            }
          } # is $val a raw .arpa name?

	  # Spaces are evil.
	  $val =~ s/^\s+//;
	  $val =~ s/\s+$//;
	  if ($typemap{$type} ne 'TXT') {
	    # Leading or trailng spaces could be legit in TXT records.
	    $host =~ s/^\s+//;
	    $host =~ s/\s+$//;
	  }

          $self->_printrec_tiny($zonefilehandle, $recid, 'y', \%recflags, $revzone,
            $host, $type, $val, $dist, $weight, $port, $ttl, $loc, $stamp, $expires, $stampactive);

          $recflags{$recid} = 1;

        } # while ($recsth)

        if ($self->{usecache}) {
          close ZONECACHE; # force the file to be written
	  # catch obvious write errors that leave an empty temp file
          if (-s $tmpcache) {
            rename $tmpcache, $cachefile
              or die "Error overwriting cache file $cachefile with temporary file: $!\n";
          }
        }

      } # if $changed or cache filesize is 0

    };
    if ($@) {
      die "error writing ".($self->{usecache} ? 'new data for ' : '')."$revzone: $@\n";
      # error!  something borked, and we should be able to fall back on the old cache file
      # report the error, somehow.
    } else {
      # mark zone as unmodified.  Only do this if no errors, that way
      # export failures should recover a little more automatically.
      $zonesth->execute($revid);
    }

    if ($self->{usecache}) {
      # We've already made as sure as we can that a cached zone file is "good",
      # although possibly stale/obsolete due to errors creating a new one.
      eval {
        open CACHE, "<$cachefile" or die $!;
        print $datafile $_ or die "error copying cached $revzone to master file: $!" while <CACHE>;
        close CACHE;
      };
      die $@ if $@;
    }

  } # while ($revsth)

  $soasth = $dbh->prepare("SELECT host,type,val,distance,weight,port,ttl,record_id,location ".
	"FROM records WHERE domain_id=? AND type=6");
  $recsth = $dbh->prepare("SELECT host,type,val,distance,weight,port,ttl,record_id,location,extract(epoch from stamp),expires,stampactive ".
	"FROM records WHERE domain_id=? AND NOT type=6");	# Just exclude all types relating to rDNS
#	"FROM records WHERE domain_id=? AND type < 65280");	# Just exclude all types relating to rDNS
  my $domsth = $dbh->prepare("SELECT domain_id,domain,status,changed FROM domains WHERE status=1 ORDER BY domain_id");
  $zonesth = $dbh->prepare("UPDATE domains SET changed='n' WHERE domain_id=?");
  $domsth->execute();
  while (my ($domid,$dom,$domstat,$changed) = $domsth->fetchrow_array) {
##fixme: need to find a way to block opening symlinked files without introducing a race.
#       O_NOFOLLOW
#              If  pathname  is a symbolic link, then the open fails.  This is a FreeBSD extension, which was
#              added to Linux in version 2.1.126.  Symbolic links in earlier components of the pathname  will
#              still be followed.
# but that doesn't help other platforms.  :/
    my $cachefile = "$self->{exportcache}/$dom";
    my $tmpcache = "$self->{exportcache}/tmp.$dom.$$";
    eval {

      # write fresh records if:
      #  - we are not using the cache
      #  - force_refresh is set
      #  - the zone has changed
      #  - the cache file does not exist
      #  - the cache file is empty
      if (!$self->{usecache} || $self->{force_refresh} || $changed || !-e $cachefile || -z $cachefile) {
        if ($self->{usecache}) {
          open ZONECACHE, ">$tmpcache" or die "Error creating temporary file $tmpcache: $!\n";
          $zonefilehandle = *ZONECACHE;
        }

        # need to fetch this separately so the SOA comes first in the flatfile....
        # Just In Case we need/want to reimport from the flatfile later on.
        $soasth->execute($domid);
        my (@zsoa) = $soasth->fetchrow_array();
        $self->_printrec_tiny($zonefilehandle, $zsoa[7], 'n',\%recflags,$dom,
          $zsoa[0],$zsoa[1],$zsoa[2],$zsoa[3],$zsoa[4],$zsoa[5],$zsoa[6],$zsoa[8],'');

        $recsth->execute($domid);
        while (my ($host,$type,$val,$dist,$weight,$port,$ttl,$recid,$loc,$stamp,$expires,$stampactive) = $recsth->fetchrow_array) {
	  next if $recflags{$recid};

          # Check for out-of-zone data
          $host = $dom if $host eq '@';
          if ($host !~ /$dom$/i) {
            warn "Not exporting out-of-zone record $host $type $val, $ttl (zone $dom)\n";
            next;
          }

	  # Spaces are evil.
	  $host =~ s/^\s+//;
	  $host =~ s/\s+$//;
	  if ($typemap{$type} ne 'TXT') {
	    # Leading or trailng spaces could be legit in TXT records.
	    $val =~ s/^\s+//;
	    $val =~ s/\s+$//;
	  }

	  $self->_printrec_tiny($zonefilehandle, $recid, 'n', \%recflags,
		$dom, $host, $type, $val, $dist, $weight, $port, $ttl, $loc, $stamp, $expires, $stampactive);

	  $recflags{$recid} = 1;

        } # while ($recsth)


        if ($self->{usecache}) {
          close ZONECACHE; # force the file to be written
	  # catch obvious write errors that leave an empty temp file
          if (-s $tmpcache) {
            rename $tmpcache, $cachefile
              or die "Error overwriting cache file $cachefile with temporary file: $!\n";
          }
        }

      } # if $changed or cache filesize is 0

    };
    if ($@) {
      die "error writing ".($self->{usecache} ? 'new data for ' : '')."$dom: $@\n";
      # error!  something borked, and we should be able to fall back on the old cache file
      # report the error, somehow.
    } else {
      # mark domain as unmodified.  Only do this if no errors, that way
      # export failures should recover a little more automatically.
      $zonesth->execute($domid);
    }

    if ($self->{usecache}) {
      # We've already made as sure as we can that a cached zone file is "good",
      # although possibly stale/obsolete due to errors creating a new one.
      eval {
        open CACHE, "<$cachefile" or die $!;
        print $datafile $_ or die "error copying cached $dom to master file: $!" while <CACHE>;
        close CACHE;
      };
      die $@ if $@;
    }

  } # while ($domsth)

  return 1;
} # end __export_tiny()


# Utility sub for __export_tiny above
sub _printrec_tiny {
  my $self = shift;
  my ($datafile, $recid, $revrec, $recflags, $zone, $host, $type, $val, $dist, $weight, $port, $ttl,
	$loc, $stamp, $expires, $stampactive) = @_;

  $loc = '' if !$loc;	# de-nullify - just in case
##fixme:  handle case of record-with-location-that-doesn't-exist better.
# note this currently fails safe (tested) - records with a location that
# doesn't exist will not be sent to any client
#	$loc = '' if !$lochash->{$loc};


## Records that are valid only before or after a set time

# record due to expire sometime is the complex case.  we don't want to just
# rely on tinydns' auto-adjusting TTLs, because the default TTL in that case
# is one day instead of the SOA minttl as BIND might do.

# consider the case where a record is set to expire a week ahead, but the next
# day later you want to change it NOW (or as NOWish as you get with your DNS
# management practice).  but now you're stuck, because someone, somewhere, 
# has just done a lookup before your latest change was published, and they'll
# be caching that old, broken record for 1 day instead of your zone default
# TTL.

# $stamp-$ttl is the *latest* we can publish the record with the defined TTL
# to still have the expiry happen as scheduled, but we need to find some
# *earlier* point.  We can maybe guess, and 2x TTL is probably reasonable,
# but we need info on the export frequency.

# export the normal, non-expiring record up until $stamp-<guesstimate>, then
# switch to exporting a record with the TAI64 stamp and a 0 TTL so tinydns
# takes over TTL management.

  if ($stampactive) {
    if ($expires) {
      # record expires at $stamp;  decide if we need to keep the TTL and ignore
      # the stamp for a time or if we need to change the TTL to 0 and convert
      # $stamp to TAI64 so tinydns can use $stamp to autoadjust the TTL on the fly.
# extra hack, optimally needs more knowledge of data export frequency
# smack the idiot customer who insists on 0 TTLs;  they can suck up and
# deal with a 10-minute TTL.  especially on scheduled changes.  note this
# should be (export freq * 2), but we don't know the actual export frequency.
$ttl = 300 if $ttl == 0;	#hack phtui
      my $ahead = (86400 < $ttl*2 ? 86400 : $ttl*2);
      if ((time() + $ahead) < $stamp) {
        # more than 2x TTL OR more than one day (whichever is less) from expiry time;  publish normal record
        $stamp = '';
      } else {
        # less than 2x TTL from expiry time, let tinydns take over TTL management and publish the TAI64 stamp.
        $ttl = 0;
        $stamp = unixtai64($stamp);
        $stamp =~ s/\@//;
      }
    } else {
      # record is "active after";  convert epoch from database to TAI64, publish, and collect $200.
      $stamp = unixtai64($stamp);
      $stamp =~ s/\@//;
    }
  } else {
    # flag for active timestamp is false;  don't actually put a timestamp in the output
    $stamp = '';
  }

  # support tinydns' auto-TTL
  $ttl = '' if $ttl == -1;
# these are WAY FREAKING HIGH - higher even than most TLD registry TTLs!
# NS          259200  => 3d
# all others   86400  => 1d

  if ($revrec eq 'y') {
    $val = $zone if $val eq '@';
  } else {
    $host = $zone if $host eq '@';
  }

  ## Convert a bare number into an octal-coded pair of octets.
  # Take optional arg to indicate a decimal or hex input.  Defaults to hex.
  sub octalize {
    my $tmp = shift;
    my $srctype = shift || 'h';	# default assumes hex string
    $tmp = sprintf "%0.4x", hex($tmp) if $srctype eq 'h';	# 0-pad hex to 4 digits
    $tmp = sprintf "%0.4x", $tmp if $srctype eq 'd';	# 0-pad decimal to 4 hex digits
    my @o = ($tmp =~ /^(..)(..)$/);	# split into octets
    return sprintf "\\%0.3o\\%0.3o", hex($o[0]), hex($o[1]);
  }

  # Utility sub-sub for reverse records;  with "any-record-in-any-zone"
  # we may need to do extra processing on $val to make it publishable.
  sub __revswap {
    my $host = shift;
    my $val = shift;
    return ($val, $host) if $val =~ /\.arpa/;
    $val = new NetAddr::IP $val;
    my $newval = _ZONE($val, 'ZONE', 'r', '.').($val->{isv6} ? '.ip6.arpa' : '.in-addr.arpa');
    return ($newval, $host);
  }

## WARNING:  This works to export even the whole Internet's worth of IP space...
##  if you have the disk/RAM to handle the dataset, and you call this sub based on /16-sized chunks
##  A /16 took ~3 seconds with a handful of separate records;  adding a /8 pushed export time out to ~13m:40s 
##  0/0 is estimated to take ~54 hours and ~256G of disk
##  RAM usage depends on how many non-template entries you have in the set.
##  This should probably be done on record addition rather than export;  large blocks may need to be done in a
##  forked process
  sub __publish_subnet {
    my $self = shift;	# *sigh*  need to pass in the DNSDB object so we can read a couple of options
    my $sub = shift;
    my $recflags = shift;
    my $hpat = shift;
    my $fh = shift;
    my $ttl = shift;
    my $stamp = shift;
    my $loc = shift;
    my $zone = new NetAddr::IP shift;
    my $ptronly = shift || 0;

    # do this conversion once, not (number-of-ips-in-subnet) times
    my $arpabase = _ZONE($zone, 'ZONE.in-addr.arpa', 'r', '.');

    my $iplist = $sub->splitref(32);
    my $ipindex = -1;
    foreach (@$iplist) {
      my $ip = $_->addr;
      $ipindex++;
      # make as if we split the non-octet-aligned block into octet-aligned blocks as with SOA
      my $lastoct = (split /\./, $ip)[3];
      next if $$recflags{$ip}; # && $self->{skip_bcast_255}
      $$recflags{$ip}++;
      next if $hpat eq '%blank%';	# Allows blanking a subnet so no records are published.
      my $rec = $hpat;	# start fresh with the template for each IP
##fixme:  there really isn't a good way to handle sub-/24 zones here.  This way at least
# seems less bad than some alternatives.
      $self->_template4_expand(\$rec, $ip, \$sub, $ipindex);
      # _template4_expand may blank $rec;  if so, don't publish a record
      next if !$rec;
      if ($ptronly || $zone->masklen > 24) {
        print $fh "^$lastoct.$arpabase:$rec:$ttl:$stamp:$loc\n" or die $!;
        if (!$ptronly) {
          # print a separate A record.  Arguably we could use an = record here instead.
          print $fh "+$rec:$ip:$ttl:$stamp:$loc\n" or die $!;
        }
      } else {
        print $fh "=$rec:$ip:$ttl:$stamp:$loc\n" or die $!;
      }
    }
  } # __publish_subnet

## And now the meat.

##fixme?  append . to all host/val hostnames
#print "debug: rawdata: $host $typemap{$type} $val\n";

  if ($typemap{$type} eq 'SOA') {
    # host contains pri-ns:responsible
    # val is abused to contain refresh:retry:expire:minttl
##fixme:  "manual" serial vs tinydns-autoserial
    # let's be explicit about abusing $host and $val
    my ($email, $primary) = (split /:/, $host)[0,1];
    my ($refresh, $retry, $expire, $min_ttl) = (split /:/, $val)[0,1,2,3];
    if ($revrec eq 'y') {
##fixme:  have to publish SOA records for each v4 /24 in sub-/16, and each /16 in sub-/8
# what about v6?
# -> only need SOA for local chunks offset from reverse delegation boundaries, so v6 is fine
# anyone who says they need sub-nibble v6 delegations, at this time, needs their head examined.
      $zone = NetAddr::IP->new($zone);
      # handle split-n-multiply SOA for off-octet (8 < mask < 16) or (16 < mask < 24) v4 zones
      if (!$zone->{isv6} && ($zone->masklen < 24) && ($zone->masklen % 8 != 0)) {
        foreach my $szone ($zone->split($zone->masklen + (8 - $zone->masklen % 8))) {
          $szone = _ZONE($szone, 'ZONE.in-addr.arpa', 'r', '.');
          print $datafile "Z$szone:$primary:$email"."::$refresh:$retry:$expire:$min_ttl:$ttl:$stamp:$loc\n"
            or die $!;
        }
        return; # skips "default" bits just below
      }
      $zone = _ZONE($zone, 'ZONE', 'r', '.').($zone->{isv6} ? '.ip6.arpa' : '.in-addr.arpa');
    }
    print $datafile "Z$zone:$primary:$email"."::$refresh:$retry:$expire:$min_ttl:$ttl:$stamp:$loc\n"
      or die $!;

  } elsif ($typemap{$type} eq 'A') {

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';
    print $datafile "+$host:$val:$ttl:$stamp:$loc\n" or die $!;

  } elsif ($typemap{$type} eq 'NS') {

    if ($revrec eq 'y') {
      $val = NetAddr::IP->new($val);
      # handle split-n-multiply SOA for off-octet (8 < mask < 16) or (16 < mask < 24) v4 zones
      if (!$val->{isv6} && ($val->masklen < 24) && ($val->masklen % 8 != 0)) {
        foreach my $szone ($val->split($val->masklen + (8 - $val->masklen % 8))) {
          my $szone2 = _ZONE($szone, 'ZONE.in-addr.arpa', 'r', '.');
          next if $$recflags{$szone2} && $$recflags{$szone2} > $val->masklen;
          print $datafile "\&$szone2"."::$host:$ttl:$stamp:$loc\n" or die $!;
          $$recflags{$szone2} = $val->masklen;
        }
      } elsif ($val->{isv6} && ($val->masklen < 64) && ($val->masklen % 4 !=0)) {
        foreach my $szone ($val->split($val->masklen + (4 - $val->masklen % 4))) {
          my $szone2 = _ZONE($szone, 'ZONE.ip6.arpa', 'r', '.');
          next if $$recflags{$szone2} && $$recflags{$szone2} > $val->masklen;
          print $datafile "\&$szone2"."::$host:$ttl:$stamp:$loc\n" or die $!;
          $$recflags{$szone2} = $val->masklen;
        }
      } else {
        my $val2 = _ZONE($val, 'ZONE', 'r', '.').($val->{isv6} ? '.ip6.arpa' : '.in-addr.arpa');
        print $datafile "\&$val2"."::$host:$ttl:$stamp:$loc\n" or die $!;
        $$recflags{$val2} = $val->masklen;
      }
    } else {
      print $datafile "\&$host"."::$val:$ttl:$stamp:$loc\n" or die $!;
    }

  } elsif ($typemap{$type} eq 'AAAA') {

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';
    my $altgrp = 0;
    my @altconv;
    # Split in to up to 8 groups of hex digits (allows for IPv6 :: 0-collapsing)
    foreach (split /:/, $val) {
      if (/^$/) {
        # flag blank entry;  this is a series of 0's of (currently) unknown length
        $altconv[$altgrp++] = 's';
      } else {
        # call sub to convert 1-4 hex digits to 2 string-rep octal bytes
        $altconv[$altgrp++] = octalize($_)
      }
    }
    my $prefix = ":$host:28:";
    foreach my $octet (@altconv) {
      # if not 's', output
      $prefix .= $octet unless $octet =~ /^s$/;
      # if 's', output (9-array length)x literal '\000\000'
      $prefix .= '\000\000'x(9-$altgrp) if $octet =~ /^s$/;
    }
    print $datafile "$prefix:$ttl:$stamp:$loc\n" or die $!;

  } elsif ($typemap{$type} eq 'MX') {

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';
    print $datafile "\@$host"."::$val:$dist:$ttl:$stamp:$loc\n" or die $!;

  } elsif ($typemap{$type} eq 'TXT') {

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';
# le sigh.  Some idiot DNS implementations don't seem to like tinydns autosplitting
# long TXT records at 127 characters instead of 255.  Hand-crafting a record seems
# to paper over the remote stupid.  We will NOT try to split on whitespace;  the
# contents of a TXT record are opaque and clients who can't deal are even more broken
# than the ones that don't like them split at 127 characters...  because BIND tries
# to "intelligently" split TXT data, and abso-by-damn-lutely generates chunks <255
# characters, and anything that can't interpret BIND's DNS responses has no business
# trying to interpret DNS data at all.

    if ($self->{autotxt}) {
      # let tinydns deal with splitting the record.  note tinydns autosplits at 127
      # characters, not 255.  Because Reasons.
      $val =~ s/:/\\072/g;	# may need to replace other symbols
      print $datafile "'$host:$val:$ttl:$stamp:$loc\n" or die $!;
    } else {
      print $datafile ":$host:16:";
      my @txtsegs = $val =~ /.{1,255}/g;
      foreach (@txtsegs) {
        my $len = length($_);
        s/:/\\072/g;
        printf $datafile "\\%0.3o%s", $len, $_;
      }
      print $datafile ":$ttl:$stamp:$loc\n";
    }

# by-hand TXT
#:deepnet.cx:16:2v\075spf1\040a\040a\072bacon.deepnet.cx\040a\072home.deepnet.cx\040-all:3600
#@       IN      TXT     "v=spf1 a a:bacon.deepnet.cx a:home.deepnet.cx -all"
#'deepnet.cx:v=spf1 a a\072bacon.deepnet.cx a\072home.deepnet.cx -all:3600

#txttest IN      TXT     "v=foo bar:bob kn;ob' \" !@#$%^&*()-=_+[]{}<>?"
#:txttest.deepnet.cx:16:\054v\075foo\040bar\072bob\040kn\073ob\047\040\042\040\041\100\043\044\045\136\046\052\050\051-\075\137\053\133\135\173\175\074\076\077:3600

# very long TXT record as brought in by axfr-get
# note tinydns does not support >512-byte RR data, need axfr-dns (for TCP support) for that
# also note, tinydns does not seem to support <512, >256-byte RRdata from axfr-get either.  :/
#:longtxt.deepnet.cx:16:
#\170this is a very long txt record.  it is really long.  long. very long.  really very long. this is a very long txt record.
#\263  it is really long.  long. very long.  really very long. this is a very long txt record.  it is really long.  long. very long.  really very long. this is a very long txt record. 
#\351 it is really long.  long. very long.  really very long.this is a very long txt record.  it is really long.  long. very long.  really very long. this is a very long txt record.  it is really long.  long. very long.  really very long.
#:3600

  } elsif ($typemap{$type} eq 'CNAME') {

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';
    print $datafile "C$host:$val:$ttl:$stamp:$loc\n" or die $!;

  } elsif ($typemap{$type} eq 'SRV') {

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';

    # data is two-byte values for priority, weight, port, in that order,
    # followed by length/string data

    my $prefix = ":$host:33:".octalize($dist,'d').octalize($weight,'d').octalize($port,'d');

    $val .= '.' if $val !~ /\.$/;
    foreach (split /\./, $val) {
      $prefix .= sprintf "\\%0.3o%s", length($_), $_ or die $!;
    }
    print $datafile "$prefix\\000:$ttl:$stamp:$loc\n" or die $!;

  } elsif ($typemap{$type} eq 'RP') {

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';
    # RP consists of two mostly free-form strings.
    # The first is supposed to be an email address with @ replaced by . (as with the SOA contact)
    # The second is the "hostname" of a TXT record with more info.
    my $prefix = ":$host:17:";
    my ($who,$what) = split /\s/, $val;
    foreach (split /\./, $who) {
      $prefix .= sprintf "\\%0.3o%s", length($_), $_;
    }
    $prefix .= '\000';
    foreach (split /\./, $what) {
      $prefix .= sprintf "\\%0.3o%s", length($_), $_;
    }
    print $datafile "$prefix\\000:$ttl:$stamp:$loc\n" or die $!;

  } elsif ($typemap{$type} eq 'PTR') {

    $$recflags{$val}++;
    if ($revrec eq 'y') {

      if ($val =~ /\.arpa$/) {
        # someone put in the formal .arpa name.  humor them.
        print $datafile "^$val:$host:$ttl:$stamp:$loc\n" or die $!;
      } else {
        $zone = NetAddr::IP->new($zone);
        if (!$zone->{isv6} && $zone->masklen > 24) {
          # sub-octet v4 zone
          ($val) = ($val =~ /\.(\d+)$/);
          print $datafile "^$val."._ZONE($zone, 'ZONE', 'r', '.').'.in-addr.arpa'.
            ":$host:$ttl:$stamp:$loc\n" or die $!;
        } else {
          # not going to care about strange results if $val is not an IP value and is resolveable in DNS
          $val = NetAddr::IP->new($val);
          print $datafile "^".
            _ZONE($val, 'ZONE', 'r', '.').($val->{isv6} ? '.ip6.arpa' : '.in-addr.arpa').
            ":$host:$ttl:$stamp:$loc\n" or die $!;
        }
      } # non-".arpa" $val

    } else {
      # PTRs in forward zones are less bizarre and insane than some other record types
      # in reverse zones...  OTOH we can't validate them any which way, so we cross our
      # fingers and close our eyes and make it Someone Else's Problem.
      print $datafile "^$host:$val:$ttl:$stamp:$loc\n" or die $!;
    }

  } elsif ($type == 65280) { # A+PTR

    $$recflags{$val}++;
    print $datafile "=$host:$val:$ttl:$stamp:$loc\n" or die $!;

  } elsif ($type == 65281) { # AAAA+PTR

    $$recflags{$val}++;
    # treat these as two separate records.  since tinydns doesn't have
    # a native combined type, we have to create them separately anyway.
    # print both;  a dangling record is harmless, and impossible via web
    # UI anyway
    $self->_printrec_tiny($datafile,$recid,'n',$recflags, $self->domainName($self->_hostparent($host)),
	$host, 28, $val, $dist, $weight, $port, $ttl, $loc, $stamp);
    $self->_printrec_tiny($datafile, $recid, 'y', $recflags, $zone,
	$host, 12, $val, $dist, $weight, $port, $ttl, $loc, $stamp);

##fixme: add a config flag to indicate use of the patch from http://www.fefe.de/dns/
# type 6 is for AAAA+PTR, type 3 is for AAAA

  } elsif ($type == 65282) { # PTR template

    # only useful for v4 with standard DNS software, since this expands all
    # IPs in $zone (or possibly $val?) with autogenerated records
    $val = NetAddr::IP->new($val);
    return if $val->{isv6};

    if ($val->masklen <= 16) {
      foreach my $sub ($val->split(16)) {
        $self->__publish_subnet($sub, $recflags, $host, $datafile, $ttl, $stamp, $loc, $zone, 1);
      }
    } else {
      $self->__publish_subnet($val, $recflags, $host, $datafile, $ttl, $stamp, $loc, $zone, 1);
    }

  } elsif ($type == 65283) { # A+PTR template

    $val = NetAddr::IP->new($val);
    # Just In Case.  An A+PTR should be impossible to add to a v6 revzone via API.
    return if $val->{isv6};

    if ($val->masklen < 16) {
      foreach my $sub ($val->split(16)) {
        $self->__publish_subnet($sub, $recflags, $host, $datafile, $ttl, $stamp, $loc, $zone, 0);
      }
    } else {
      $self->__publish_subnet($val, $recflags, $host, $datafile, $ttl, $stamp, $loc, $zone, 0);
    }

  } elsif ($type == 65284) { # AAAA+PTR template
    # Stub for completeness.  Could be exported to DNS software that supports
    # some degree of internal automagic in generic-record-creation
    # (eg http://search.cpan.org/dist/AllKnowingDNS/ )

  } elsif ($type == 65285) { # Delegation
    # This is intended for reverse zones, but may prove useful in forward zones.

    # All delegations need to create one or more NS records.  The NS record handler knows what to do.
    $self->_printrec_tiny($datafile,$recid,$revrec,$recflags,$zone,$host,$reverse_typemap{'NS'},
      $val,$dist,$weight,$port,$ttl,$loc,$stamp);
    if ($revrec eq 'y') {
      # In the case of a sub-/24 v4 reverse delegation, we need to generate CNAMEs
      # to redirect all of the individual IP lookups as well.
      # OR
      # create NS records for each IP
      # Not sure how this would actually resolve if a /24 or larger was delegated
      # one way, and a sub-/24 in that >=/24 was delegated elsewhere...
      my $dblock = NetAddr::IP->new($val);
      if (!$dblock->{isv6} && $dblock->masklen > 24) {
        my @subs = $dblock->split;
        foreach (@subs) {
          next if $$recflags{"$_"};
          my ($oct) = ($_->addr =~ /(\d+)$/);
          print $datafile "C"._ZONE($_, 'ZONE.in-addr.arpa', 'r', '.').":$oct.".
            _ZONE($dblock, 'ZONE.in-addr.arpa', 'r', '.').":$ttl:$stamp:$loc\n" or die $!;
            $$recflags{"$_"}++;
        }
      }
    }

##
## Uncommon types.  These will need better UI support Any Day Sometime Maybe(TM).
##

  } elsif ($type == 44) { # SSHFP

    ($host,$val) = __revswap($host,$val) if $revrec eq 'y';

    my ($algo,$fpt,$fp) = split /\s+/, $val;

    my $rec = sprintf ":$host:44:\\%0.3o\\%0.3o", $algo, $fpt;
    while (my ($byte) = ($fp =~ /^(..)/) ) {
      $rec .= sprintf "\\%0.3o", hex($byte);
      $fp =~ s/^..//;
    }
    print $datafile "$rec:$ttl:$stamp:$loc\n" or die $!;

  } else {
    # raw record.  we don't know what's in here, so we ASS-U-ME the user has
    # put it in correctly, since either the user is messing directly with the
    # database, or the record was imported via AXFR
    # <split by char>
    # convert anything not a-zA-Z0-9.- to octal coding

##fixme: add flag to export "unknown" record types - note we'll probably end up
# mangling them since they were written to the DB from Net::DNS::RR::<type>->rdatastr.
    #print $datafile ":$host:$type:$val:$ttl:$stamp:$loc\n";

  } # record type if-else

} # end _printrec_tiny()


## DNSDB::mailNotify()
# Sends notification mail to recipients regarding a DNSDB operation
sub mailNotify {
  my $self = shift;
  my $dbh = $self->{dbh};
  my ($subj,$message) = @_;

  return if $self->{mailhost} eq 'smtp.example.com';   # do nothing if still using default SMTP host.

  my $mailer = Net::SMTP->new($self->{mailhost}, Hello => "dnsadmin.$self->{domain}");

  my $mailsender = ($self->{mailsender} ? $self->{mailsender} : $self->{mailnotify});

  $mailer->mail($mailsender);
  $mailer->to($self->{mailnotify});
  $mailer->data("From: \"$self->{mailname}\" <$mailsender>\n",
	"To: <$self->{mailnotify}>\n",
	"Date: ".strftime("%a, %d %b %Y %H:%M:%S %z",localtime)."\n",
	"Subject: $subj\n",
	"X-Mailer: DNSAdmin v".$DNSDB::VERSION." Notify\n",
	"Organization: $self->{orgname}\n",
	"\n$message\n");
  $mailer->quit;
}

# shut Perl up
1;
