#!/usr/bin/perl -w -T
# Main web UI script for DeepNet DNS Administrator
##
# $Id: dns.cgi 920 2025-08-14 17:35:51Z kdeugau $
# Copyright 2008-2022 Kris Deugau <kdeugau@deepnet.cx>
# 
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version. 
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##

use strict;
use warnings;

use CGI::Carp qw (fatalsToBrowser);
use CGI::Simple;
use HTML::Template;
use CGI::Session '-ip_match';
use Net::DNS;
use DBI;

use Data::Dumper;

#sub is_tainted {
#  # from perldoc perlsec
#  return ! eval { eval("#" . substr(join("", @_), 0, 0)); 1 };
#}

# Taint-safe (ish) voodoo to push "the directory the script is in" into @INC.
# See https://secure.deepnet.cx/trac/dnsadmin/ticket/80 for more gory details on how we got here.
use File::Spec ();
use File::Basename ();
my $path;
BEGIN {
    $path = File::Basename::dirname(File::Spec->rel2abs($0));
    if ($path =~ /(.*)/) {
        $path = $1;
    }
}
use lib $path;

use DNSDB;

my @debugbits;  # temp, to be spit out near the end of processing
my $debugenv = 0;

# Let's do these templates right...
my $templatedir = "templates";

# Set up the CGI object...
my $q = new CGI::Simple;
# ... and get query-string params as well as POST params if necessary
$q->parse_query_string;

# This is probably excessive fiddling, but it puts the parameters somewhere my fingers know about...
my %webvar = $q->Vars;

# shut up some warnings, in case we arrive somewhere we forgot to set this
$webvar{defrec} = 'n' if !$webvar{defrec};	# non-default records
$webvar{revrec} = 'n' if !$webvar{revrec};	# non-reverse (domain) records

# create a DNSDB object.  this loads some local system defaults and connects to the DB
# with the credentials configured
##fixme:  pass params for loadConfig, and use them there, to allow one codebase to support multiple sites
my $dnsdb = new DNSDB;

my $header = HTML::Template->new(filename => "$templatedir/header.tmpl");
my $footer = HTML::Template->new(filename => "$templatedir/footer.tmpl");
$footer->param(version => $DNSDB::VERSION);

##fixme:  slim chance this could be triggered on errors other than DB failure?
if (!$dnsdb) {
  print "Content-type: text/html\n\n";
  print $header->output;
  my $errpage = HTML::Template->new(filename => "$templatedir/dberr.tmpl");
  $errpage->param(errmsg => $DNSDB::errstr);
  print $errpage->output;
  print $footer->output;
  exit;
}

$header->param(orgname => $dnsdb->{orgname}) if $dnsdb->{orgname} ne 'Example Corp';

my $logingroup;
my $curgroup;
my @viewablegroups;

# retrieve the session ID from our cookie, if possible
my $sid = $q->cookie('dnsadmin_session');

# see if the session loads
my $session = CGI::Session->load("driver:File", $sid, {Directory => $dnsdb->{sessiondir}})
	or die CGI::Session->errstr();

if (!$sid || $session->is_expired || !$session->param('uid') || !$dnsdb->userStatus($session->param('uid')) ) {
  $webvar{page} = 'login';
} else {
  # we have a session to load from, maybe
  $logingroup = ($session->param('logingroup') ? $session->param('logingroup') : 1);
  $curgroup = ($session->param('curgroup') ? $session->param('curgroup') : $logingroup);
  # security check - does the user have permission to view this entity?
  # this is a prep step used "many" places
  $dnsdb->getChildren($logingroup, \@viewablegroups, 'all');
  push @viewablegroups, $logingroup;
##fixme: make sessions persist through closing the site?
# this even bridges browser close too.  hmm...
  $webvar{page} = 'domlist' if !$webvar{page};
}

# set $webvar{page} before we try to use it.
$webvar{page} = 'login' if !$webvar{page};

## per-page startwith, filter, searchsubs

##fixme:  complain-munge-and-continue with non-"[a-z0-9-.]" filter and startwith
$webvar{startwith} =~ s/^(0-9|[a-z]).*/$1/ if $webvar{startwith};
# not much call for chars not allowed in domain names
# allow <>= so searches can use the Postgres CIDR operators
# allow , for things like DMARC records
$webvar{filter} =~ s{[^a-zA-Z0-9_.,:\@%<>=/-]}{}g if $webvar{filter};
## only set 'y' if box is checked, no other values legal
## however, see https://secure.deepnet.cx/trac/dnsadmin/ticket/31
# first, drop obvious fakes
delete $webvar{searchsubs} if $webvar{searchsubs} && $webvar{searchsubs} !~ /^[ny]/;
# strip the known "turn me off!" bit.
$webvar{searchsubs} =~ s/^n\s?// if $webvar{searchsubs};
# strip non-y/n - note this legitimately allows {searchsubs} to go empty
$webvar{searchsubs} =~ s/[^yn]//g if $webvar{searchsubs};

# pagination
my $perpage = 15;  # Just In Case
$perpage = $dnsdb->{perpage} if $dnsdb->{perpage};
my $offset = ($webvar{offset} ? $webvar{offset} : 0);

## set up "URL to self" (whereami edition)
# @#$%@%@#% XHTML - & in a URL must be escaped.  >:(
my $uri_self = $ENV{REQUEST_URI};
$uri_self = "/dns.cgi" if !$uri_self || $uri_self eq '/';
$uri_self =~ s/\&([a-z])/\&amp\;$1/g;

# le sigh.  and we need to strip any previous action
$uri_self =~ s/\&amp;action=[^&]+//g;

# much magic happens.  if startwith or a search string change (to, from, or
# across, in the request vs whatever's in the session) then the offset should
# be reset to 0 so that the first/prev/next/last widget populates correctly,
# and so that the list of whatever we're looking at actually shows things
# (since we may have started on page 42 of 300 with a LOOOOONG list, but we
# now only need 3 pages for the filtered list).
# while we're at it, plonk these into the session for safekeeping.
if (defined($webvar{startwith})) {
  if ($webvar{startwith} ne $session->param($webvar{page}.'startwith')) {
    $uri_self =~ s/\&amp;offset=[^&]//;
    $offset = 0;
  }
  $session->param($webvar{page}.'startwith', $webvar{startwith});
}
if (defined($webvar{filter})) {
  $session->param($webvar{page}.'filter', '') if !$session->param($webvar{page}.'filter');
  if ($webvar{filter} ne $session->param($webvar{page}.'filter')) {
    $uri_self =~ s/\&amp;offset=[^&]//;
    $offset = 0;
  }
  $session->param($webvar{page}.'filter', $webvar{filter})
}
$session->param($webvar{page}.'searchsubs', $webvar{searchsubs}) if defined($webvar{searchsubs});

# and now that the search/filter criteria for this page are set, put them in some globals for actual use.
my $startwith = $session->param($webvar{page}.'startwith');
my $filter = $session->param($webvar{page}.'filter');
my $searchsubs = $session->param($webvar{page}.'searchsubs');

# ... and assemble the args
my @filterargs;
push @filterargs, "^[$startwith]" if $startwith;
push @filterargs, $filter if $filter;

# and search filter options.  these get stored in the session, but discarded
# as soon as you switch to a different page.
##fixme:  think about retaining these on a per-page basis, as well as offset;  same as the sort-order bits
no warnings qw(uninitialized);
$uri_self =~ s/\&amp;startwith=[a-z09-]*(\&)?/$1/g;
$uri_self =~ s/\&amp;searchsubs=[a-z09-]*(\&)?/$1/g;
$uri_self =~ s/\&amp;filter=[a-z09-]*(\&)?/$1/g;
use warnings qw(uninitialized);

# Fix up $uri_self so we don't lose the session/page
$uri_self .= "?page=$webvar{page}" if $uri_self =~ m{/dns.cgi$};
$uri_self = "$ENV{SCRIPT_NAME}?page=$webvar{page}$1" if $uri_self =~ m{/dns.cgi\&(.+)$};

## end uri_self monkeying

# NB:  these must match the field name and SQL ascend/descend syntax respectively
# sortby is reset to a suitable "default", then re-reset to whatever the user has
# clicked on last in the record=listing subs, but best to put a default here.
my $sortby = "domain";
my $sortorder = "ASC";

# Create the page template object.  Display a reasonable error page and whine if the template doesn't exist.
my $page;
eval {
  # sigh.  can't set loop_context_vars or global_vars once instantiated.
  $page = HTML::Template->new(filename => "$templatedir/$webvar{page}.tmpl",
	loop_context_vars => 1, global_vars => 1);
};
if ($@) {
  my $msg = $@;
  $page = HTML::Template->new(filename => "$templatedir/badpage.tmpl");
  if (-e "$templatedir/$webvar{page}.tmpl") {
    $page->param(badtemplate => $q->escapeHTML($msg));
  } else {
    warn "Bad page $webvar{page} requested";
    $page->param(badpage => $q->escapeHTML($webvar{page}));
  }
  $webvar{page} = 'badpage';
}

$session->expire($dnsdb->{timeout});
my $sesscookie = $q->cookie( -name => 'dnsadmin_session',
        -value => $sid,
        -expires => "+".$dnsdb->{timeout},
        -secure => 0,
## fixme:  need to extract root path for cookie, so as to limit cookie to dnsadmin instance
#        -path => $url
        );

# handle can-happen-on-(almost)-any-page actions
if ($webvar{action}) {

  if ($webvar{action} eq 'login') {
    # Snag ACL/permissions here too

    my $userdata = $dnsdb->login($webvar{username}, $webvar{password});

    if ($userdata) {

      # (re)create the session
      $session = new CGI::Session("driver:File", $sid, {Directory => $dnsdb->{sessiondir}})
        or die CGI::Session->errstr();
      $sid = $session->id();

      $sesscookie = $q->cookie( -name => 'dnsadmin_session',
        -value => $sid,
        -expires => "+".$dnsdb->{timeout},
        -secure => 0,
## fixme:  need to extract root path for cookie, so as to limit cookie to dnsadmin instance
#        -path => $url
        );

      # set session bits
      $session->expire($dnsdb->{timeout});
      $session->param('logingroup',$userdata->{group_id});
      $session->param('curgroup',$userdata->{group_id});
      $session->param('uid',$userdata->{user_id});
      $session->param('username',$webvar{username});
      $curgroup = $userdata->{group_id};

# for reference.  seems we don't need to set these on login any more.
#  $session->param('domlistsortby','domain');
#  $session->param('domlistorder','ASC');
#  $session->param('revzonessortby','revnet');
#  $session->param('revzonesorder','ASC');
#  $session->param('useradminsortby','user');
#  $session->param('useradminorder','ASC');
#  $session->param('grpmansortby','group');
#  $session->param('grpmanorder','ASC');
#  $session->param('reclistsortby','host');
#  $session->param('reclistorder','ASC');
#  $session->param('loclistsortby','description');
#  $session->param('loclistorder','ASC');
#  $session->param('logsortby','stamp');
#  $session->param('logorder','DESC');

      ## "recover my link" - tack on request bits and use requested page instead of hardcoding domlist
      # this could possibly be compacted by munging changepage a little so we don't have to deconstruct
      # and reconstruct the URI argument list.
      my %target = (page => "domlist");
      if ($webvar{target} && $webvar{target} =~ /\?/ && $webvar{target} !~ /page=login/) {
        my $tmp = (split /\?/, $webvar{target})[1];
        $tmp =~ s/^\&//;
        my @targs = split /\&/, $tmp;
        foreach (@targs) {
          my ($k,$v) = split /=/;
          $target{$k} = $v if $k;
          # if we're going through a "session expired" login, we may have a different
          # "current group" than the login group.
          $session->param('curgroup', $v) if $k eq 'curgroup';
##fixme:  page=record goes "FOOM", sometimes - cause/fix?
        }
      }
      changepage(%target);

    } else {
      $webvar{loginfailed} = 1;
    } # user data fetch check

  } elsif ($webvar{action} eq 'logout') {
    # delete the session
    $session->delete();
    $session->flush();

    my $sesscookie = $q->cookie( -name => 'dnsadmin_session',
      -value => $sid,
      -expires => "-1",
      -secure => 0,
## fixme:  need to extract root path for cookie, so as to limit cookie to dnsadmin instance
#      -path => $url
      );

    my $newurl = "http://$ENV{HTTP_HOST}$ENV{SCRIPT_NAME}";
    $newurl =~ s|/[^/]+$|/|;
    print $q->redirect( -uri => $newurl, -cookie => $sesscookie);
    exit;

  } elsif ($webvar{action} eq 'chgroup' && $webvar{page} ne 'login') {
    # fiddle session-stored group data
    # magic incantation to... uhhh...

    # ... and the "change group" bits...
    $uri_self =~ s/\&amp;group=[^&]*//g;

    # security check - does the user have permission to view this entity?
    my $errmsg;
    if (!(grep /^$webvar{group}$/, @viewablegroups)) {
      # hmm.  Reset the current group to the login group?  Yes.  Prevents confusing behaviour elsewhere.
      $session->param('curgroup',$logingroup);
      $webvar{group} = $logingroup;
      $curgroup = $logingroup;
      $errmsg = "You are not permitted to view or make changes in the requested group";
      $page->param(errmsg => $errmsg);
    }

    $session->param('curgroup', $webvar{group});
    $curgroup = ($webvar{group} ? $webvar{group} : $session->param('curgroup'));

    # I hate special cases.
##fixme: probably need to handle webvar{revrec}=='y' too
    if ($webvar{page} eq 'reclist' && $webvar{defrec} eq 'y') {
      my %args = (page => $webvar{page}, id => $curgroup, defrec => $webvar{defrec}, revrec => $webvar{revrec});
      $args{errmsg} = $errmsg if $errmsg;
      changepage(%args);
    }
    # add offset back *into* $uri_self if we're also currently looking at a live record list.
    if ($webvar{page} eq 'reclist' && $webvar{defrec} eq 'n') {
      $uri_self .= "\&amp;offset=$offset";
    }
  } # done action=chgroup
} # handle global webvar{action}s


# finally check if the user was disabled.  we could just leave this for logout/session expiry,
# but if they keep the session active they'll continue to have access long after being disabled.  :/
# Treat it as a session expiry.
if ($session->param('uid') && !$dnsdb->userStatus($session->param('uid')) ) {
  $sid = '';
  $session->delete;	# force expiry of the session Right Away
  $session->flush;	# make sure it hits storage
  changepage(page=> "login", sessexpired => 1);
}

# Misc Things To Do on most pages
my %permissions;
$dnsdb->getPermissions('user', $session->param('uid'), \%permissions);
$dnsdb->initActionLog($session->param('uid'));

##
## Per-page processing
##

if ($webvar{page} eq 'login') {

  my $target = $ENV{REQUEST_URI};
  $target =~ s/\&/\&amp;/g;
  $page->param(target => $target); # needs to be trimmed a little, maybe?

  $page->param(sessexpired => 1) if (!$sid && $target !~ m|/$|);

  if ($webvar{loginfailed}) {
    $page->param(loginfailed => 1);
    $webvar{target} =~ s/\&/\&amp;/g;	# XHTML we do (not) love you so
    $page->param(target => $webvar{target}) if $webvar{target};
  }
#  if $webvar{sessexpired};	 # or this with below?
  if ($session->is_expired) {
    $page->param(sessexpired => 1);
    $session->delete();   # Just to make sure
    $session->flush();
  }
  $page->param(version => $DNSDB::VERSION);
  $page->param(script_self => ($ENV{SCRIPT_NAME} =~ m|/([^/]+)$|)[0]);

} elsif ($webvar{page} eq 'domlist' or $webvar{page} eq 'index') {

  $page->param(domlist => 1);

# hmm.  seeing problems in some possibly-not-so-corner cases.
# this currently only handles "domain on", "domain off"
  if (defined($webvar{zonestatus})) {
    # security check - does the user have permission to access this entity?
    my $flag = 0;
    foreach (@viewablegroups) {
      $flag = 1 if $dnsdb->isParent($_, 'group', $webvar{id}, 'domain');
    }
    if ($flag && ($permissions{admin} || $permissions{domain_edit})) {
      my $stat = $dnsdb->zoneStatus($webvar{id}, 'n', $webvar{zonestatus});
      $page->param(resultmsg => $DNSDB::resultstr);
    } else {
      $page->param(errmsg => "You are not permitted to view or change the requested domain");
    }
    $uri_self =~ s/\&amp;zonestatus=[^&]*//g;	# clean up URL for stuffing into templates
  }

  show_msgs();

  $page->param(curpage => $webvar{page});

  listdomains();

} elsif ($webvar{page} eq 'newdomain') {

  changepage(page => "domlist", errmsg => "You are not permitted to add domains")
	unless ($permissions{admin} || $permissions{domain_create});

  $webvar{group} = $curgroup if !$webvar{group};
  fill_grouplist("grouplist", $webvar{group});
  fill_loclist($curgroup, $webvar{defloc} ? $webvar{defloc} : '');

  if ($session->param('add_failed')) {
    $session->clear('add_failed');
    $page->param(add_failed => 1);
    $page->param(errmsg => $session->param('errmsg'));
    $session->clear('errmsg');
    $page->param(domain => $webvar{domain});
    $page->param(addinactive => $webvar{makeactive} eq 'n');
  }

} elsif ($webvar{page} eq 'adddomain') {

  changepage(page => "domlist", errmsg => "You are not permitted to add domains")
	unless ($permissions{admin} || $permissions{domain_create});

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{group}, type => 'group')) {
    $session->param('add_failed', 1);
##fixme:  domain a security risk for XSS?
    changepage(page => "newdomain", domain => $webvar{domain},
	errmsg => "You do not have permission to add a domain to the requested group");
  }

  $webvar{makeactive} = 0 if !defined($webvar{makeactive});

  my ($code,$msg) = $dnsdb->addDomain($webvar{domain}, $webvar{group}, ($webvar{makeactive} eq 'on' ? 1 : 0),
        $webvar{defloc});

  if ($code eq 'OK') {
    $webvar{domain} = lc($webvar{domain}) if $dnsdb->{lowercase};
    $dnsdb->mailNotify("New ".($webvar{makeactive} eq 'on' ? 'Active' : 'Inactive')." Domain Created",
	($webvar{makeactive} eq 'on' ? 'Active' : 'Inactive').qq( domain "$webvar{domain}" added by ).
	$session->param("username"));
    changepage(page => "reclist", id => $msg);
  } else {
    $session->param('add_failed', 1);
##fixme:  domain a security risk for XSS?
    changepage(page => "newdomain", errmsg => $msg, domain => $webvar{domain},
	group => $webvar{group}, makeactive => ($webvar{makeactive} ? 'y' : 'n'), defloc => $webvar{defloc});
  }

} elsif ($webvar{page} eq 'deldom') {

  changepage(page => "domlist", errmsg => "You are not permitted to delete domains")
	unless ($permissions{admin} || $permissions{domain_delete});

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{id}, type => 'domain')) {
    changepage(page => "domlist", errmsg => "You do not have permission to delete the requested domain");
  }

  $page->param(id => $webvar{id});

  # first pass = confirm y/n (sorta)
  if (!defined($webvar{del})) {

    $page->param(del_getconf => 1);
    $page->param(domain => $dnsdb->domainName($webvar{id}));

  } elsif ($webvar{del} eq 'ok') {
    my $pargroup = $dnsdb->parentID(id => $webvar{id}, type => 'domain', revrec => $webvar{revrec});
    my ($code,$msg) = $dnsdb->delZone($webvar{id}, $webvar{revrec});
    if ($code eq 'OK') {
      changepage(page => "domlist", resultmsg => $msg);
    } else {
      changepage(page => "domlist", errmsg => $msg);
    }

  } else {
    # cancelled.  whee!
    changepage(page => "domlist");
  }

} elsif ($webvar{page} eq 'revzones') {

  $webvar{revrec} = 'y';

  if (defined($webvar{zonestatus})) {
    # security check - does the user have permission to access this entity?
    my $flag = 0;
    foreach (@viewablegroups) {
      $flag = 1 if $dnsdb->isParent($_, 'group', $webvar{id}, 'revzone');
    }
    if ($flag && ($permissions{admin} || $permissions{domain_edit})) {
      my $stat = $dnsdb->zoneStatus($webvar{id}, 'y', $webvar{zonestatus});
      $page->param(resultmsg => $DNSDB::resultstr);
    } else {
      $page->param(errmsg => "You are not permitted to view or change the requested reverse zone");
    }
    $uri_self =~ s/\&amp;zonestatus=[^&]*//g;	# clean up URL for stuffing into templates
  }

  show_msgs();

  $page->param(curpage => $webvar{page});
  listzones();

} elsif ($webvar{page} eq 'newrevzone') {

## scope/access check - use domain settings?  invent new (bleh)
  changepage(page => "revzones", errmsg => "You are not permitted to add reverse zones")
       unless ($permissions{admin} || $permissions{domain_create});

  fill_grouplist("grouplist");
  fill_loclist($curgroup, $webvar{defloc} ? $webvar{defloc} : '');

  # prepopulate revpatt with the matching default record
# $dnsdb->getRecByName(revrec => $webvar{revrec}, defrec => $webvar{defrec}, host => 'string');

  if ($session->param('add_failed')) {
    $session->clear('add_failed');
    $page->param(errmsg => $session->param('errmsg'));
    $session->clear('errmsg');
    $page->param(revzone => $webvar{revzone});
    $page->param(revpatt => $webvar{revpatt});
  }

} elsif ($webvar{page} eq 'addrevzone') {

  changepage(page => "revzones", errmsg => "You are not permitted to add reverse zones")
       unless ($permissions{admin} || $permissions{domain_create});

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{group}, type => 'group')) {
    changepage(page => "newrevzone", add_failed => 1, revzone => $webvar{revzone}, revpatt => $webvar{revpatt},
       errmsg => "You do not have permission to add a reverse zone to the requested group");
  }

  my ($code,$msg) = $dnsdb->addRDNS($webvar{revzone}, $webvar{revpatt}, $webvar{group},
	($webvar{makeactive} eq 'on' ? 1 : 0), $webvar{location});

  if ($code eq 'OK') {
    changepage(page => "reclist", id => $msg, revrec => 'y');
  } elsif ($code eq 'WARN') {
    changepage(page => "reclist", id => $msg, revrec => 'y', warnmsg => $DNSDB::resultstr);
  } else {
    $session->param('add_failed', 1);
    changepage(page => "newrevzone", revzone => $webvar{revzone}, revpatt => $webvar{revpatt}, errmsg => $msg);
  }

} elsif ($webvar{page} eq 'delrevzone') {

  changepage(page => "revzones", errmsg => "You are not permitted to delete reverse zones")
	unless ($permissions{admin} || $permissions{domain_delete});

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{id}, type => 'revzone')) {
    changepage(page => "revzones", errmsg => "You do not have permission to delete the requested reverse zone");
  }

  $page->param(id => $webvar{id});

  # first pass = confirm y/n (sorta)
  if (!defined($webvar{del})) {

    $page->param(del_getconf => 1);
    $page->param(revzone => $dnsdb->revName($webvar{id}));

  } elsif ($webvar{del} eq 'ok') {
    my $pargroup = $dnsdb->parentID(id => $webvar{id}, type => 'revzone', revrec => $webvar{revrec});
    my $zone = $dnsdb->revName($webvar{id});
    my ($code,$msg) = $dnsdb->delZone($webvar{id}, 'y');
    if ($code eq 'OK') {
      changepage(page => "revzones", resultmsg => $msg);
    } else {
      changepage(page => "revzones", errmsg => $msg);
    }

  } else {
    # cancelled.  whee!
    changepage(page => "revzones");
  }

} elsif ($webvar{page} eq 'secondaryzones') {

#  $page->param(domlist => 1);

# hmm.  seeing problems in some possibly-not-so-corner cases.
# this currently only handles "domain on", "domain off"
  if (defined($webvar{zonestatus})) {
    # security check - does the user have permission to access this entity?
    my $flag = 0;
    foreach (@viewablegroups) {
      $flag = 1 if $dnsdb->isParent($_, 'group', $webvar{id}, 'secondary');
    }
    if ($flag && ($permissions{admin} || $permissions{domain_edit})) {
      my $stat = $dnsdb->secondaryzoneStatus($webvar{id}, 'n', $webvar{zonestatus});
      $page->param(resultmsg => $DNSDB::resultstr);
    } else {
      $page->param(errmsg => "You are not permitted to view or change the requested domain");
    }
    $uri_self =~ s/\&amp;zonestatus=[^&]*//g;	# clean up URL for stuffing into templates
  }

  show_msgs();

  $page->param(curpage => $webvar{page});

  listsecondaryzones();

} elsif ($webvar{page} eq 'editsecondary') {

  # security check - does the user have permission to view this entity?
  # id is zone id
  if (!check_scope(id => $webvar{id}, type => 'secondaryzone')) {
    changepage(page => 'secondaryzones',
	errmsg => "You do not have permission to edit the requested secondary zone");
  }

  changepage(page => "secondaryzones", errmsg => "You are not permitted to edit secondary zones", id => $webvar{id})
	unless ($permissions{admin} || $permissions{domain_edit});

  $page->param(id => $webvar{id});
  $page->param(parent => $dnsdb->secondaryName($webvar{id}));
  my $zoneinfo = $dnsdb->getSecondaryDetails($webvar{id});
  $page->param(prins => $zoneinfo->{primaryserver});
  fill_loclist($curgroup, $zoneinfo->{default_location});

} elsif ($webvar{page} eq 'updatesecondary') {

  # security check - does the user have permission to view this entity?
  # pass 1, record ID
  # id is zone id
  if (!check_scope(id => $webvar{id}, type => 'secondaryzone')) {
    changepage(page => 'secondaryzones',
	errmsg => "You do not have permission to edit the requested secondary zone");
  }

  changepage(page => "secondaryzones", errmsg => "You are not permitted to edit secondary zones", id => $webvar{id})
	unless ($permissions{admin} || $permissions{domain_edit});

  my ($code, $msg) = $dnsdb->updateSecondaryDetails(
	id => $webvar{id}, primary => $webvar{prins}, location => $webvar{defloc});
  if ($code eq 'OK') {
    my $zone = $dnsdb->secondaryName($webvar{id});
    changepage(page => "secondaryzones", resultmsg => "Secondary zone $zone updated");
  } else {
    $page->param(update_failed => 1);
    $page->param(msg => $msg);

    $page->param(id => $webvar{id});
    $page->param(prins => $webvar{prins});
    fill_loclist($curgroup, $webvar{defloc});
  }

} elsif ($webvar{page} eq 'reclist') {

  # security check - does the user have permission to view this entity?
  if (!check_scope(id => $webvar{id}, type =>
	($webvar{defrec} eq 'y' ? 'group' : ($webvar{revrec} eq 'y' ? 'revzone' : 'domain')))) {
    $page->param(errmsg => "You are not permitted to view or change the requested ".
	($webvar{defrec} eq 'y' ? "group's default records" :
		($webvar{revrec} eq 'y' ? "reverse zone's records" : "domain's records")));
    $page->param(perm_err => 1);	# this causes the template to skip the record listing output.
    goto DONERECLIST;	# and now we skip filling in the content which is not printed due to perm_err above
  }

# hmm.  where do we send them?
  if ($webvar{defrec} eq 'y' && !$permissions{admin}) {
    $page->param(errmsg => "You are not permitted to edit default records");
    $page->param(perm_err => 1);
  } else {

    $page->param(mayeditsoa => $permissions{admin} || $permissions{domain_edit});
##fixme:  ACL needs pondering.  Does "edit domain" interact with record add/remove/etc?
# Note this seems to be answered "no" in Vega.
# ACLs
    $page->param(record_create	=> ($permissions{admin} || $permissions{record_create}) );
# we don't have any general edit links on the page;  they're all embedded in the TMPL_LOOP
#    $page->param(record_edit	=> ($permissions{admin} || $permissions{record_edit}) );
    $page->param(record_delete	=> ($permissions{admin} || $permissions{record_delete}) );

  # Handle record list for both default records (per-group) and live domain records

    $page->param(defrec => $webvar{defrec});
    $page->param(revrec => $webvar{revrec});
    $page->param(id => $webvar{id});
    $page->param(curpage => $webvar{page});

    my $count = $dnsdb->getRecCount(defrec => $webvar{defrec}, revrec => $webvar{revrec},
	id => $webvar{id}, filter => $filter);

    $sortby = 'host';
# sort/order
    $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
    $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};

    $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
    $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

# set up the headers
    my @cols;
    my %colheads;
    if ($webvar{revrec} eq 'n') {
      @cols = ('host', 'type', 'val', 'distance', 'weight', 'port', 'ttl');
      %colheads = (host => 'Name', type => 'Type', val => 'Address',
	distance => 'Distance', weight => 'Weight', port => 'Port', ttl => 'TTL');
    } else {
      @cols = ('val', 'type', 'host', 'ttl');
      %colheads = (val => 'IP Address', type => 'Type', host => 'Hostname', ttl => 'TTL');
    }
    my %custom = (id => $webvar{id}, defrec => $webvar{defrec}, revrec => $webvar{revrec});
    fill_colheads($sortby, $sortorder, \@cols, \%colheads, \%custom);

# fill the page-count and first-previous-next-last-all details
    fill_pgcount($count,"records",
	($webvar{defrec} eq 'y' ? "group ".$dnsdb->groupName($webvar{id}) : 
		($webvar{revrec} eq 'y' ? $dnsdb->revName($webvar{id}) : $dnsdb->domainName($webvar{id}))
	));
    fill_fpnla($count);  # should put some params on this sub...

    $page->param(defrec => $webvar{defrec});
    showzone($webvar{defrec}, $webvar{revrec}, $webvar{id});
    if ($webvar{defrec} eq 'n') {
      if ($webvar{revrec} eq 'n') {
	$page->param(logdom => 1);
      } else {
	$page->param(logrdns => 1);
      }
    }

    show_msgs();

  } # close "you can't edit default records" check

  # Yes, this is a GOTO target.  PTBHTTT.
  DONERECLIST: ;

} elsif ($webvar{page} eq 'record') {

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{id}, type =>
	($webvar{defrec} eq 'y' ? ($webvar{revrec} eq 'y' ? 'defrevrec' : 'defrec') : 'record'))) {
    $page->param(perm_err => "You are not permitted to edit the requested record");
    goto DONEREC;
  }
  # round 2, check the parent.
  if (!check_scope(id => $webvar{parentid}, type =>
	($webvar{defrec} eq 'y' ? 'group' : ($webvar{revrec} eq 'y' ? 'revzone' : 'domain')))) {
    my $msg = ($webvar{defrec} eq 'y' ?
	"You are not permitted to add or edit default records in the requested group" :
	"You are not permitted to add or edit records in the requested domain/zone");
    $page->param(perm_err => $msg);
    goto DONEREC;
  }

  $page->param(defrec => $webvar{defrec});
  $page->param(revrec => $webvar{revrec});
  $page->param(fwdzone => $webvar{revrec} eq 'n');

  if ($webvar{recact} eq 'new') {

    changepage(page => "reclist", errmsg => "You are not permitted to add records", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{record_create});

    $page->param(todo => "Add record");
    $page->param(recact => "add");
    $page->param(parentid => $webvar{parentid});

    fill_recdata();

    if ($webvar{defrec} eq 'n') {
      my $defloc = $dnsdb->getZoneLocation($webvar{revrec}, $webvar{parentid});
      fill_loclist($curgroup, $defloc);
    }

  } elsif ($webvar{recact} eq 'add') {

    changepage(page => "reclist", errmsg => "You are not permitted to add records", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{record_create});

    # location check - if user does not have record_locchg, set $webvar{location} to default location for zone
    my $parloc = $dnsdb->getZoneLocation($webvar{revrec}, $webvar{parentid});
    $webvar{location} = $parloc unless ($permissions{admin} || $permissions{record_locchg});

    my @recargs = ($webvar{defrec}, $webvar{revrec}, $webvar{parentid},
	\$webvar{name}, \$webvar{type}, \$webvar{address}, $webvar{ttl}, $webvar{location},
	$webvar{expires}, $webvar{stamp});
    if ($webvar{type} == $reverse_typemap{MX} or $webvar{type} == $reverse_typemap{SRV}) {
      push @recargs, $webvar{distance};
      if ($webvar{type} == $reverse_typemap{SRV}) {
        push @recargs, $webvar{weight};
        push @recargs, $webvar{port};
      }
    }

    my ($code,$msg) = $dnsdb->addRec(@recargs);

    if ($code eq 'OK' || $code eq 'WARN') {
      my %pageparams = (page => "reclist", id => $webvar{parentid},
	defrec => $webvar{defrec}, revrec => $webvar{revrec});
      $pageparams{warnmsg} = $msg."<br />\n".$DNSDB::resultstr if $code eq 'WARN';
      $pageparams{resultmsg} = $DNSDB::resultstr if $code eq 'OK';
      changepage(%pageparams);
    } else {
      $page->param(failed	=> 1);
      $page->param(errmsg	=> $msg);
      $page->param(wastrying	=> "adding");
      $page->param(todo		=> "Add record");
      $page->param(recact	=> "add");
      $page->param(parentid	=> $webvar{parentid});
      $page->param(id		=> $webvar{id});
      fill_recdata();	# populate the form... er, mostly.
      if ($webvar{defrec} eq 'n') {
	fill_loclist($curgroup, $webvar{location});
      }
    }

  } elsif ($webvar{recact} eq 'edit') {

    changepage(page => "reclist", errmsg => "You are not permitted to edit records", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{record_edit});

    $page->param(todo		=> "Update record");
    $page->param(recact		=> "update");
    $page->param(parentid	=> $webvar{parentid});
    $page->param(id		=> $webvar{id});
    my $recdata = $dnsdb->getRecLine($webvar{defrec}, $webvar{revrec}, $webvar{id});
    $page->param(name		=> $recdata->{host});
    $page->param(address	=> $recdata->{val});
    $page->param(distance	=> $recdata->{distance});
    $page->param(weight		=> $recdata->{weight});
    $page->param(port		=> $recdata->{port});
    $page->param(ttl		=> $recdata->{ttl});
    $page->param(typelist	=> $dnsdb->getTypelist($webvar{revrec}, $recdata->{type}));
    if ($recdata->{stampactive}) {
      $page->param(stamp => $recdata->{stamp});
      $page->param(stamp_until => $recdata->{expires});
    }
    if ($webvar{defrec} eq 'n') {
      fill_loclist($curgroup, $recdata->{location});
    }

  } elsif ($webvar{recact} eq 'update') {

    changepage(page => "reclist", errmsg => "You are not permitted to edit records", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{record_edit});

    # retain old location if user doesn't have permission to fiddle locations
    my $oldrec = $dnsdb->getRecLine($webvar{defrec}, $webvar{revrec}, $webvar{id});
    $webvar{location} = $oldrec->{location} unless ($permissions{admin} || $permissions{record_locchg});

    my ($code,$msg) = $dnsdb->updateRec($webvar{defrec}, $webvar{revrec}, $webvar{id}, $webvar{parentid},
	\$webvar{name}, \$webvar{type}, \$webvar{address}, $webvar{ttl}, $webvar{location},
	$webvar{expires}, $webvar{stamp},
	$webvar{distance}, $webvar{weight}, $webvar{port});

    if ($code eq 'OK' || $code eq 'WARN') {
      my %pageparams = (page => "reclist", id => $webvar{parentid},
	defrec => $webvar{defrec}, revrec => $webvar{revrec});
      $pageparams{warnmsg} = $msg."<br />\n".$DNSDB::resultstr if $code eq 'WARN';
      $pageparams{resultmsg} = $DNSDB::resultstr if $code eq 'OK';
      changepage(%pageparams);
    } else {
      $page->param(failed	=> 1);
      $page->param(errmsg	=> $msg);
      $page->param(wastrying	=> "updating");
      $page->param(todo		=> "Update record");
      $page->param(recact	=> "update");
      $page->param(parentid	=> $webvar{parentid});
      $page->param(id		=> $webvar{id});
      fill_recdata();
    }
  }

  if ($webvar{defrec} eq 'y') {
    $page->param(dohere => "default records in group ".$dnsdb->groupName($webvar{parentid}));
  } else {
    $page->param(dohere => $dnsdb->domainName($webvar{parentid})) if $webvar{revrec} eq 'n';
    $page->param(dohere => $dnsdb->revName($webvar{parentid})) if $webvar{revrec} eq 'y';
  }

  # Yes, this is a GOTO target.  PTBHTTT.
  DONEREC: ;

} elsif ($webvar{page} eq 'delrec') {

  # This is a complete separate segment since it uses a different template from add/edit records above

  changepage(page => "reclist", errmsg => "You are not permitted to delete records", id => $webvar{parentid},
		defrec => $webvar{defrec}, revrec => $webvar{revrec})
	unless ($permissions{admin} || $permissions{record_delete});

  if (!check_scope(id => $webvar{id}, type =>
	($webvar{defrec} eq 'y' ? ($webvar{revrec} eq 'y' ? 'defrevrec' : 'defrec') : 'record'))) {
    # redirect to domlist because we don't have permission for the entity requested
    changepage(page => 'domlist', revrec => $webvar{revrec},
	errmsg => "You do not have permission to delete records in the requested ".
	($webvar{defrec} eq 'y' ? 'group' : 'domain'));
  }

  $page->param(id => $webvar{id});
  $page->param(defrec => $webvar{defrec});
  $page->param(revrec => $webvar{revrec});
  $page->param(parentid => $webvar{parentid});
  # first pass = confirm y/n (sorta)
  if (!defined($webvar{del})) {
    $page->param(del_getconf => 1);
    my $rec = $dnsdb->getRecLine($webvar{defrec}, $webvar{revrec}, $webvar{id});
    $page->param(host => $rec->{host});
    $page->param(ftype => $typemap{$rec->{type}});
    $page->param(recval => $rec->{val});
  } elsif ($webvar{del} eq 'ok') {
    my ($code,$msg) = $dnsdb->delRec($webvar{defrec}, $webvar{revrec}, $webvar{id});
    if ($code eq 'OK') {
      changepage(page => "reclist", id => $webvar{parentid}, defrec => $webvar{defrec},
		revrec => $webvar{revrec}, resultmsg => $msg);
    } else {
## need to find failure mode
      changepage(page => "reclist", id => $webvar{parentid}, defrec => $webvar{defrec},
		revrec => $webvar{revrec}, errmsg => $msg);
    }
  } else {
    changepage(page => "reclist", id => $webvar{parentid}, defrec => $webvar{defrec}, revrec => $webvar{revrec});
  }

} elsif ($webvar{page} eq 'editsoa') {

  # security check - does the user have permission to view this entity?
  # id is domain/revzone/group id
  if (!check_scope(id => $webvar{id}, type =>
	($webvar{defrec} eq 'y' ? 'group' : ($webvar{revrec} eq 'y' ? 'revzone' : 'domain')))) {
    changepage(page => 'domlist', errmsg => "You do not have permission to edit the ".
	($webvar{defrec} eq 'y' ? 'default ' : '')."SOA record for the requested ".
	($webvar{defrec} eq 'y' ? 'group' : 'domain'));
  }

  if ($webvar{defrec} eq 'y') {
    changepage(page => "domlist", errmsg => "You are not permitted to edit default records")
	unless $permissions{admin};
  } else {
    changepage(page => "reclist", errmsg => "You are not permitted to edit domain SOA records", id => $webvar{id})
	unless ($permissions{admin} || $permissions{domain_edit});
  }

  fillsoa($webvar{defrec},$webvar{revrec},$webvar{id});

} elsif ($webvar{page} eq 'updatesoa') {

  # security check - does the user have permission to view this entity?
  # pass 1, record ID
  if (!check_scope(id => $webvar{recid}, type =>
	($webvar{defrec} eq 'y' ? ($webvar{revrec} eq 'y' ? 'defrevrec' : 'defrec') : 'record'))) {
##fixme:  should we redirect to the requested record list page instead of the domain list?
    changepage(page => 'domlist', errmsg => "You do not have permission to edit the requested SOA record");
  }
  # pass 2, parent (group or domain) ID
  if (!check_scope(id => $webvar{id}, type =>
	($webvar{defrec} eq 'y' ? 'group' : ($webvar{revrec} eq 'y' ? 'revzone' : 'domain')))) {
    changepage(page => ($webvar{revrec} eq 'y' ? 'revzones' : 'domlist'),
	errmsg => "You do not have permission to edit the ".
	($webvar{defrec} eq 'y' ? 'default ' : '')."SOA record for the requested ".
	($webvar{defrec} eq 'y' ? 'group' : ($webvar{revrec} eq 'y' ? 'reverse zone' : 'domain')) );
  }

  changepage(page => "reclist", errmsg => "You are not permitted to edit domain SOA records", id => $webvar{id})
	unless ($permissions{admin} || $permissions{domain_edit});

  my ($code, $msg) = $dnsdb->updateSOA($webvar{defrec}, $webvar{revrec},
	(contact => $webvar{contact}, prins => $webvar{prins}, refresh => $webvar{refresh},
	retry => $webvar{retry}, expire => $webvar{expire}, minttl => $webvar{minttl},
	ttl => $webvar{ttl}, id => $webvar{id}) );
  if ($code eq 'OK') {
    changepage(page => "reclist", id => $webvar{id}, defrec => $webvar{defrec}, revrec => $webvar{revrec},
	resultmsg => "SOA record updated");
  } else {
    $page->param(update_failed => 1);
    $page->param(msg => $msg);
    fillsoa($webvar{defrec}, $webvar{revrec}, $webvar{id}, 'w');
  }

} elsif ($webvar{page} eq 'grpman') {

  listgroups();

# Permissions!
  $page->param(addgrp => $permissions{admin} || $permissions{group_create});
  $page->param(edgrp => $permissions{admin} || $permissions{group_edit});
  $page->param(delgrp => $permissions{admin} || $permissions{group_delete});

  show_msgs();
  $page->param(curpage => $webvar{page});

} elsif ($webvar{page} eq 'newgrp') {

  changepage(page => "grpman", errmsg => "You are not permitted to add groups")
	unless ($permissions{admin} || $permissions{group_create});

  # do.. uhh.. stuff.. if we have no webvar{grpaction}
  if ($webvar{grpaction} && $webvar{grpaction} eq 'add') {

    # security check - does the user have permission to access this entity?
    if (!check_scope(id => $webvar{pargroup}, type => 'group')) {
      changepage(page => "grpman", errmsg => "You are not permitted to add a group to the requested parent group");
    }

    my %newperms;
    my $alterperms = 0;
    foreach (@permtypes) {
      $newperms{$_} = 0;
      if ($permissions{admin} || $permissions{$_}) { 
	$newperms{$_} = (defined($webvar{$_}) && $webvar{$_} eq 'on' ? 1 : 0);
      } else { 
	$alterperms = 1; 
      }
    }
    # "Chained" permissions.  Some permissions imply others;  make sure they get set.
    foreach (keys %permchains) {
      if ($newperms{$_} && !$newperms{$permchains{$_}}) {
	$newperms{$permchains{$_}} = 1;
      }
    }
    # force inheritance of parent group's default records with inherit flag,
    # otherwise we end up with the hardcoded defaults from DNSDB.pm.  See
    # https://secure.deepnet.cx/trac/dnsadmin/ticket/8 for the UI enhancement
    # that will make this variable.
    my ($code,$msg) = $dnsdb->addGroup($webvar{newgroup}, $webvar{pargroup}, \%newperms, 1);
    if ($code eq 'OK') {
      if ($alterperms) {
	changepage(page => "grpman", warnmsg => 
		"You can only grant permissions you hold.  New group $webvar{newgroup} added with reduced access.");
      } else {
	changepage(page => "grpman", resultmsg => "Added group $webvar{newgroup}");
      }
    } # fallthrough else
    # no point in doing extra work
    fill_permissions($page, \%newperms);
    $page->param(add_failed => 1);
    $page->param(errmsg => $msg);
    $page->param(newgroup => $webvar{newgroup});
    fill_grouplist('pargroup',$webvar{pargroup});
  } else {
    fill_grouplist('pargroup',$curgroup);
    # fill default permissions with immediate parent's current ones
    my %parperms;
    $dnsdb->getPermissions('group', $curgroup, \%parperms);
    fill_permissions($page, \%parperms);
  }

} elsif ($webvar{page} eq 'delgrp') {

  changepage(page => "grpman", errmsg => "You are not permitted to delete groups", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{group_delete});

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{id}, type => 'group')) {
    changepage(page => "grpman", errmsg => "You are not permitted to delete the requested group");
  }

  $page->param(id => $webvar{id});
  # first pass = confirm y/n (sorta)
  if (!defined($webvar{del})) {
    $page->param(del_getconf => 1);

##fixme
# do a check for "group has stuff in it", and splatter a big warning
# up along with an unchecked-by-default check box to YES DAMMIT DELETE THE WHOLE THING

  } elsif ($webvar{del} eq 'ok') {
    my ($code,$msg) = $dnsdb->delGroup($webvar{id});
    if ($code eq 'OK') {
##fixme: need to clean up log when deleting a major container
      changepage(page => "grpman", resultmsg => $msg);
    } else {
# need to find failure mode
      changepage(page => "grpman", errmsg => $msg);
    }
  } else {
    # cancelled.  whee!
    changepage(page => "grpman");
  }
  $page->param(delgroupname => $dnsdb->groupName($webvar{id}));

} elsif ($webvar{page} eq 'edgroup') {

  changepage(page => "grpman", errmsg => "You are not permitted to edit groups")
	unless ($permissions{admin} || $permissions{group_edit});

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{gid}, type => 'group')) {
    changepage(page => "grpman", errmsg => "You are not permitted to edit the requested group");
  }

  if ($webvar{grpaction} && $webvar{grpaction} eq 'updperms') {
    # extra safety check;  make sure user can't construct a URL to bypass ACLs
    my %curperms;
    $dnsdb->getPermissions('group', $webvar{gid}, \%curperms);
    my %chperms;
    my $alterperms = 0;
    foreach (@permtypes) {
      $webvar{$_} = 0 if !defined($webvar{$_});
      $webvar{$_} = 1 if $webvar{$_} eq 'on';
      if ($permissions{admin} || $permissions{$_}) {
        $chperms{$_} = $webvar{$_} if $curperms{$_} ne $webvar{$_};
      } else {
        $alterperms = 1;
        $chperms{$_} = 0;
      }
    }
    # "Chained" permissions.  Some permissions imply others;  make sure they get set.
    foreach (keys %permchains) {
      if ($chperms{$_} && !$chperms{$permchains{$_}}) {
	$chperms{$permchains{$_}} = 1;
      }
    }
    my ($code,$msg) = $dnsdb->changePermissions('group', $webvar{gid}, \%chperms);
    if ($code eq 'OK') {
      if ($alterperms) {
	changepage(page => "grpman", warnmsg =>
		"You can only grant permissions you hold.  Default permissions in group ".
		$dnsdb->groupName($webvar{gid})." updated with reduced access");
      } else {
	changepage(page => "grpman", resultmsg => $msg);
      }
    } # fallthrough else
    # no point in doing extra work
    fill_permissions($page, \%chperms);
    $page->param(errmsg => $msg);
  }
  $page->param(gid => $webvar{gid});
  $page->param(grpmeddle => $dnsdb->groupName($webvar{gid}));
  my %grpperms;
  $dnsdb->getPermissions('group', $webvar{gid}, \%grpperms);
  fill_permissions($page, \%grpperms);

} elsif ($webvar{page} eq 'bulkdomain' || $webvar{page} eq 'bulkrev') {
  # Bulk operations on domains.  Note all but group move are available on the domain list.

  changepage(page => "domlist", errmsg => "You are not permitted to make bulk zone changes")
	unless ($permissions{admin} || $permissions{domain_edit} || $permissions{domain_create} || $permissions{domain_delete});

  fill_grouplist("grouplist");

  $page->param(fwdzone => $webvar{page} eq 'bulkdomain');

  my $count = $dnsdb->getZoneCount(revrec => ($webvar{page} eq 'bulkdomain' ? 'n' : 'y'),
	curgroup => $curgroup);

  $page->param(curpage => $webvar{page});
  fill_pgcount($count, 'domains', $dnsdb->groupName($curgroup));
  fill_fpnla($count);
  $page->param(perpage => $perpage);

  my $domlist = $dnsdb->getZoneList(revrec => ($webvar{page} eq 'bulkdomain' ? 'n' : 'y'),
	curgroup => $curgroup, offset => $offset);
  my $rownum = 0;
  foreach my $dom (@{$domlist}) {
    delete $dom->{status};
    delete $dom->{group};
    $dom->{newrow} = (++$rownum) % 5 == 0 && $rownum != $perpage;
  }

  $page->param(domtable => $domlist);
  # ACLs
  $page->param(maymove => ($permissions{admin} || ($permissions{domain_edit} && $permissions{domain_create} && $permissions{domain_delete})));
  $page->param(maystatus => $permissions{admin} || $permissions{domain_edit});
  $page->param(maydelete => $permissions{admin} || $permissions{domain_delete});

#} elsif ($webvar{page} eq 'confirmbulkdom' || $webvar{page} eq 'confirmbulkrev') {
} elsif ($webvar{page} eq 'confirmbulk') {

  changepage(page => "domlist", errmsg => "You are not permitted to make bulk zone changes")
	unless ($permissions{admin} || $permissions{domain_edit} || $permissions{domain_create} || $permissions{domain_delete});

  $page->param(bulkaction => $webvar{bulkaction});
  $page->param(destgroup => $webvar{destgroup});
  my @zlist;
  my $rownum = 0;

##fixme: this could probably be made more efficient, since this looks up 2 zone names for
# each comparison during sort rather than slurping them in bulk once before doing the sort
  # sort zones by zone name, not ID
  sub zsort {
    my $tmpa = ($a =~ /^dom/ ? $dnsdb->domainName($webvar{$a}) : $dnsdb->revName($webvar{$a}) );
    my $tmpb = ($b =~ /^dom/ ? $dnsdb->domainName($webvar{$b}) : $dnsdb->revName($webvar{$b}) );
    return $tmpa cmp $tmpb;
  }
  # eugh.  can't see a handy way to sort this mess by zone name the way it is on the submitting page.  :(
  foreach my $input (sort zsort grep(/^(?:dom|rev)_/, keys %webvar) ) {
    next unless $input =~ /^(dom|rev)_\d+$/;
    my $fr = $1;
    my %row = (zoneid => $webvar{$input},
        zone => ($fr eq 'dom' ? $dnsdb->domainName($webvar{$input}) : $dnsdb->revName($webvar{$input}) ),
        zvarname => $input,
	newrow => ( (++$rownum) % 5 == 0 && $rownum != $perpage),
	);
    push @zlist, \%row;
  }
  $page->param(domtable => \@zlist);

} elsif ($webvar{page} eq 'bulkchange') {

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{destgroup}, type => 'group')) {
    $page->param(errmsg => "You are not permitted to make bulk changes in the requested group");
    goto DONEBULK;
  }

  # skip the changes if user did not confirm
  my $wasrev = grep /^rev_/, keys %webvar;
  changepage(page => ($wasrev ? "bulkrev" : "bulkdomain")) unless $webvar{okdel} eq 'y';

  changepage(page => "domlist", errmsg => "You are not permitted to make bulk zone changes")
	unless ($permissions{admin} || $permissions{domain_edit} || $permissions{domain_create} || $permissions{domain_delete});

  # per-action scope checks
  if ($webvar{bulkaction} eq 'move') {
    changepage(page => "domlist", errmsg => "You are not permitted to bulk-move zones")
	unless ($permissions{admin} || ($permissions{domain_edit} && $permissions{domain_create} && $permissions{domain_delete}));
    my $newgname = $dnsdb->groupName($webvar{destgroup});
    $page->param(action => "Move to group $newgname");
  } elsif ($webvar{bulkaction} eq 'deactivate' || $webvar{bulkaction} eq 'activate') {
    changepage(page => "domlist", errmsg => "You are not permitted to bulk-$webvar{bulkaction} zones")
	unless ($permissions{admin} || $permissions{domain_edit});
    $page->param(action => "$webvar{bulkaction} zones");
  } elsif ($webvar{bulkaction} eq 'delete') {
    changepage(page => "domlist", errmsg => "You are not permitted to bulk-delete zones")
	unless ($permissions{admin} || $permissions{domain_delete});
    $page->param(action => "$webvar{bulkaction} zones");
  } else {
    # unknown action, bypass actually doing anything.  it should not be possible in
    # normal operations, and anyone who meddles with the URL gets what they deserve.
    goto DONEBULK;
  } # move/(de)activate/delete if()

  my @bulkresults;
  # nngh.  due to alpha-sorting on the previous page, we can't use domid-numeric
  # order here, and since we don't have the domain names until we go around this
  # loop, we can't alpha-sort them here.  :(
  foreach my $input (keys %webvar) {
    my %row;
    next unless $input =~ /^(dom|rev)_\d+$/;
    my $fr = $1;
    # second security check - does the user have permission to meddle with this domain?
    if (!check_scope(id => $webvar{$input}, type => ($fr eq 'dom' ? 'domain' : 'revzone'))) {
      $row{domerr} = "You are not permitted to make changes to the requested zone";
      $row{domain} = $webvar{$input};
      push @bulkresults, \%row;
      next;
    }
    $row{domain} = ($fr eq 'dom' ? $dnsdb->domainName($webvar{$input}) : $dnsdb->revName($webvar{$input}));

    # Do the $webvar{bulkaction}
    my ($code, $msg);
    ($code, $msg) = $dnsdb->changeGroup(($fr eq 'dom' ? 'domain' : 'revzone'), $webvar{$input}, $webvar{destgroup})
	if $webvar{bulkaction} eq 'move';
    if ($webvar{bulkaction} eq 'deactivate' || $webvar{bulkaction} eq 'activate') {
      my $stat = $dnsdb->zoneStatus($webvar{$input}, ($fr eq 'dom' ? 'n' : 'y'),
	($webvar{bulkaction} eq 'activate' ? 'domon' : 'domoff'));
      $code = (defined($stat) ? 'OK' : 'FAIL');
      $msg = (defined($stat) ? $DNSDB::resultstr : $DNSDB::errstr);
    }
    ($code, $msg) = $dnsdb->delZone($webvar{$input}, ($fr eq 'dom' ? 'n' : 'y'))
	if $webvar{bulkaction} eq 'delete';

    # Set the result output from the action
    if ($code eq 'OK') {
      $row{domok} = $msg;
    } elsif ($code eq 'WARN') {
      $row{domwarn} = $msg;
    } else {
      $row{domerr} = $msg;
    }
    push @bulkresults, \%row;

  } # foreach (keys %webvar)
  $page->param(bulkresults => \@bulkresults);

  # Yes, this is a GOTO target.  PTHBTTT.
  DONEBULK: ;

} elsif ($webvar{page} eq 'useradmin') {

  if (defined($webvar{userstatus})) {
    # security check - does the user have permission to access this entity?
    my $flag = 0;
    foreach (@viewablegroups) {
      $flag = 1 if $dnsdb->isParent($_, 'group', $webvar{id}, 'user');
    }
    if ($flag && ($permissions{admin} || $permissions{user_edit} ||
	($permissions{self_edit} && $webvar{id} == $session->param('uid')) )) {
      my $stat = $dnsdb->userStatus($webvar{id}, $webvar{userstatus});
      # kick user out if user disabled self
      # arguably there should be a more specific error message for this case
      changepage(page=> 'login', sessexpired => 1) if $webvar{id} == $session->param('uid');
      $page->param(resultmsg => $DNSDB::resultstr);
    } else {
      $page->param(errmsg => "You are not permitted to view or change the requested user");
    }
    $uri_self =~ s/\&amp;userstatus=[^&]*//g;   # clean up URL for stuffing into templates
  }

  list_users();

# Permissions!
  $page->param(adduser => $permissions{admin} || $permissions{user_create});
# should we block viewing other users?  Vega blocks "editing"...
#  NB:  no "edit self" link as with groups here.  maybe there should be?
#  $page->param(eduser => $permissions{admin} || $permissions{user_edit});
  $page->param(deluser => $permissions{admin} || $permissions{user_delete});

  show_msgs();
  $page->param(curpage => $webvar{page});

} elsif ($webvar{page} eq 'user') {

  # All user add/edit actions fall through the same page, since there aren't
  # really any hard differences between the templates

  #fill_actypelist($webvar{accttype});
  fill_clonemelist();
  my %grpperms;
  $dnsdb->getPermissions('group', $curgroup, \%grpperms);

  my $grppermlist = new HTML::Template(filename => "$templatedir/permlist.tmpl");
  my %noaccess;
  fill_permissions($grppermlist, \%grpperms, \%noaccess);
  $grppermlist->param(info => 1);
  $page->param(grpperms => $grppermlist->output);

  $page->param(is_admin => $permissions{admin});

  $webvar{useraction} = '' if !$webvar{useraction};

  if ($webvar{useraction} eq 'add' or $webvar{useraction} eq 'update') {

    $page->param(add => 1) if $webvar{useraction} eq 'add';

    # can't re-use $code and $msg for update if we want to be able to identify separate failure states
    my ($code,$code2,$msg,$msg2) = ('OK','OK','OK','OK');

    my $alterperms = 0;	# flag iff we need to force custom permissions due to user's current access limits

    my %newperms;	# we're going to prefill the existing permissions, so we can change them.
    $dnsdb->getPermissions('user', $webvar{uid}, \%newperms);

    if ($webvar{pass1} ne $webvar{pass2}) {
      $code = 'FAIL';
      $msg = "Passwords don't match";
    } else {

      my $permstring = 'i';	# start with "inherit"

      # Remap passed checkbox states from webvar to integer/boolean values in %newperms
      foreach (@permtypes) {
	$newperms{$_} = (defined($webvar{$_}) && $webvar{$_} eq 'on' ? 1 : 0);
      }

      # Check for chained permissions.  Some permissions imply others;  make sure they get set.
      foreach (keys %permchains) {
	if ($newperms{$_} && !$newperms{$permchains{$_}}) {
	  $newperms{$permchains{$_}} = 1;
	}
      }

      # check for possible priviledge escalations
      if (!$permissions{admin}) {
	if ($webvar{perms_type} eq 'inherit') {
	  # Group permissions are only relevant if inheriting
	  my %grpperms;
	  $dnsdb->getPermissions('group', $curgroup, \%grpperms);
	  my $ret = $dnsdb->comparePermissions(\%permissions, \%grpperms);
	  if ($ret eq '<' || $ret eq '!') {
	    # User's permissions are not a superset or equivalent to group.  Can't inherit
	    # (and include access user doesn't currently have), so we force custom.
	    $webvar{perms_type} = 'custom';
	    $alterperms = 1;
	  }
	}
	my $ret = $dnsdb->comparePermissions(\%newperms, \%permissions);
	if ($ret eq '>' || $ret eq '!') {
	  # User's new permissions are not a subset or equivalent to previous.  Can't add
	  # permissions user doesn't currently have, so we force custom.
	  $webvar{perms_type} = 'custom';
	  $alterperms = 1;
	}
      }

##fixme:
# could possibly factor building the meat of the permstring out of this if/elsif set, so
# as to avoid running around @permtypes quite so many times
      if ($webvar{perms_type} eq 'custom') {
	$permstring = 'C:';
	foreach (@permtypes) {
	  if ($permissions{admin} || $permissions{$_}) {
	    $permstring .= ",$_" if defined($webvar{$_}) && $webvar{$_} eq 'on';
	  } else {
	    $newperms{$_} = 0;	# remove permissions user doesn't currently have
	  }
	}
	$page->param(perm_custom => 1);
      } elsif ($permissions{admin} && $webvar{perms_type} eq 'clone') {
	$permstring = "c:$webvar{clonesrc}";
	$dnsdb->getPermissions('user', $webvar{clonesrc}, \%newperms);
	$page->param(perm_clone => 1);
      }
      # Recheck chained permissions, in the supposedly impossible case that the removals
      # above mangled one of them.  This *should* be impossible via normal web UI operations.
      foreach (keys %permchains) {
	if ($newperms{$_} && !$newperms{$permchains{$_}}) {
	  $newperms{$permchains{$_}} = 1;
	  $permstring .= ",$permchains{$_}";
	}
      }
      if ($webvar{useraction} eq 'add') {
	changepage(page => "useradmin", errmsg => "You do not have permission to add new users")
		unless $permissions{admin} || $permissions{user_create};
	# no scope check;  user is created in the current group
        ($code,$msg) = $dnsdb->addUser($webvar{uname}, $curgroup, $webvar{pass1},
		($webvar{makeactive} eq 'on' ? 1 : 0), $webvar{accttype}, $permstring,
		$webvar{fname}, $webvar{lname}, $webvar{phone});
      } else {
	changepage(page => "useradmin", errmsg => "You do not have permission to edit users")
		unless $permissions{admin} || $permissions{user_edit} || 
			($permissions{self_edit} && $session->param('uid') == $webvar{uid});
	# security check - does the user have permission to access this entity?
	if (!check_scope(id => $webvar{user}, type => 'user')) {
	  changepage(page => "useradmin", errmsg => "You do not have permission to edit the requested user");
	}
# User update is icky.  I'd really like to do this in one atomic operation,
# but that gets hairy by either duplicating a **lot** of code in DNSDB.pm
# or self-torture trying to not commit the transaction until we're really done.
	# Allowing for changing group, but not coding web support just yet.
	($code,$msg) = $dnsdb->updateUser($webvar{uid}, $webvar{uname}, $webvar{gid}, $webvar{pass1},
		($webvar{makeactive} eq 'on' ? 1 : 0), $webvar{accttype},
		$webvar{fname}, $webvar{lname}, $webvar{phone});
	if ($code eq 'OK') {
	  $newperms{admin} = 1 if $permissions{admin} && $webvar{accttype} eq 'S';
	  ($code2,$msg2) = $dnsdb->changePermissions('user', $webvar{uid}, \%newperms, ($permstring eq 'i'));
	}
      }
    }

    if ($code eq 'OK' && $code2 eq 'OK') {
      my %pageparams = (page => "useradmin");
      if ($alterperms) {
	$pageparams{warnmsg} = "You can only grant permissions you hold.\nUser ".
		($webvar{useraction} eq 'add' ? "$webvar{uname} added" : "info updated for $webvar{uname}").
		".\nPermissions ".($webvar{useraction} eq 'add' ? 'added' : 'updated')." with reduced access.";
      } else {
	$pageparams{resultmsg} = "$msg".($webvar{useraction} eq 'add' ? '' : "\n$msg2");
      }
      changepage(%pageparams);

    # add/update failed:
    } else {
      $page->param(add_failed => 1);
      $page->param(action => $webvar{useraction});
      $page->param(set_permgroup => 1);
      if ($webvar{perms_type} eq 'inherit') {	# set permission class radio
	$page->param(perm_inherit => 1);
      } elsif ($webvar{perms_type} eq 'clone') {
	$page->param(perm_clone => 1);
      } else {
	$page->param(perm_custom => 1);
      }
      $page->param(uname => $webvar{uname});
      $page->param(fname => $webvar{fname});
      $page->param(lname => $webvar{lname});
      $page->param(pass1 => $webvar{pass1});
      $page->param(pass2 => $webvar{pass2});
      $page->param(errmsg => "User info updated but permissions update failed: $msg2") if $code eq 'OK';
      $page->param(errmsg => $msg) if $code ne 'OK';
      fill_permissions($page, \%newperms);
      fill_actypelist($webvar{accttype});
      fill_clonemelist();
    }

  } elsif ($webvar{useraction} eq 'edit') {

    changepage(page => "useradmin", errmsg => "You do not have permission to edit users")
	unless $permissions{admin} || $permissions{user_edit} ||
		($permissions{self_edit} && $session->param('uid') == $webvar{user});

    # security check - does the user have permission to access this entity?
    if (!check_scope(id => $webvar{user}, type => 'user')) {
      changepage(page => "useradmin", errmsg => "You do not have permission to edit the requested user");
    }

    $page->param(set_permgroup => 1);
    $page->param(action => 'update');
    $page->param(uid => $webvar{user});
    fill_clonemelist();

    my $userinfo = $dnsdb->getUserData($webvar{user});
    fill_actypelist($userinfo->{type});
    # not using this yet, but adding it now means we can *much* more easily do so later.
    $page->param(gid => $userinfo->{group_id});

    my %curperms;
    $dnsdb->getPermissions('user', $webvar{user}, \%curperms);
    fill_permissions($page, \%curperms);

    $page->param(uname => $userinfo->{username});
    $page->param(fname => $userinfo->{firstname});
    $page->param(lname => $userinfo->{lastname});
    $page->param(set_permgroup => 1);
    if ($userinfo->{inherit_perm}) {
      $page->param(perm_inherit => 1);
    } else {
      $page->param(perm_custom => 1);
    }
  } else {
    changepage(page => "useradmin", errmsg => "You are not allowed to add new users")
	unless $permissions{admin} || $permissions{user_create};
    # default is "new"
    $page->param(add => 1);
    $page->param(action => 'add');
    fill_permissions($page, \%grpperms);
    fill_actypelist();
  }

} elsif ($webvar{page} eq 'deluser') {

  changepage(page=> "useradmin", errmsg => "You are not allowed to delete users")
	unless $permissions{admin} || $permissions{user_delete};

  # security check - does the user have permission to access this entity?
  if (!check_scope(id => $webvar{id}, type => 'user')) {
    changepage(page => "useradmin", errmsg => "You are not permitted to delete the requested user");
  }

  $page->param(id => $webvar{id});
  # first pass = confirm y/n (sorta)
  if (!defined($webvar{del})) {
    $page->param(del_getconf => 1);
    $page->param(user => $dnsdb->userFullName($webvar{id}));
  } elsif ($webvar{del} eq 'ok') {
    my ($code,$msg) = $dnsdb->delUser($webvar{id});
    if ($code eq 'OK') {
      # success.  go back to the user list, do not pass "GO"
      changepage(page => "useradmin", resultmsg => $msg);
    } else {
      changepage(page => "useradmin", errmsg => $msg);
    }
  } else {
    # cancelled.  whee!
    changepage(page => "useradmin");
  }

} elsif ($webvar{page} eq 'loclist') {

  changepage(page => "domlist", errmsg => "You are not allowed access to this function")
	unless $permissions{admin} || $permissions{location_view};

  # security check - does the user have permission to access this entity?
#  if (!check_scope(id => $webvar{id}, type => 'loc')) {
#    changepage(page => "loclist", errmsg => "You are not permitted to <foo> the requested location/view");
#  }
  list_locations();
  show_msgs();

# Permissions!
  $page->param(addloc => $permissions{admin} || $permissions{location_create});
  $page->param(delloc => $permissions{admin} || $permissions{location_delete});

} elsif ($webvar{page} eq 'location') {

  changepage(page => "domlist", errmsg => "You are not allowed access to this function")
	unless $permissions{admin} || $permissions{location_view};

  # security check - does the user have permission to access this entity?
#  if (!check_scope(id => $webvar{id}, type => 'loc')) {
#    changepage(page => "loclist", errmsg => "You are not permitted to <foo> the requested location/view");
#  }

  $webvar{locact} = '' if !$webvar{locact};

  if ($webvar{locact} eq 'add') {
    changepage(page => "loclist", errmsg => "You are not permitted to add locations/views", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{location_create});

    my ($code,$msg) = $dnsdb->addLoc(group => $curgroup, desc => $webvar{locname},
	comments => $webvar{comments}, iplist => $webvar{iplist});

    if ($code eq 'OK' || $code eq 'WARN') {
      my %pageparams = (page => "loclist", id => $webvar{parentid},
	defrec => $webvar{defrec}, revrec => $webvar{revrec});
      $pageparams{warnmsg} = $msg."<br />\n".$DNSDB::resultstr if $code eq 'WARN';
      $pageparams{resultmsg} = $DNSDB::resultstr if $code eq 'OK';
      changepage(%pageparams);
    } else {
      $page->param(failed	=> 1);
      $page->param(errmsg	=> $msg);
      $page->param(wastrying	=> "adding");
      $page->param(todo		=> "Add location/view");
      $page->param(locact	=> "add");
      $page->param(id		=> $webvar{id});
      $page->param(locname	=> $webvar{locname});
      $page->param(comments	=> $webvar{comments});
      $page->param(iplist	=> $webvar{iplist});
    }

  } elsif ($webvar{locact} eq 'edit') {
    changepage(page => "loclist", errmsg => "You are not permitted to edit locations/views", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{location_edit});

    my $loc = $dnsdb->getLoc($webvar{loc});
    $page->param(wastrying	=> "editing");
    $page->param(todo		=> "Edit location/view");
    $page->param(locact		=> "update");
    $page->param(id		=> $webvar{loc});
    $page->param(locname	=> $loc->{description});
    $page->param(comments	=> $loc->{comments});
    $page->param(iplist		=> $loc->{iplist});

  } elsif ($webvar{locact} eq 'update') {
    changepage(page => "loclist", errmsg => "You are not permitted to edit locations/views", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{location_edit});

    my ($code,$msg) = $dnsdb->updateLoc($webvar{id}, $curgroup, $webvar{locname}, $webvar{comments}, $webvar{iplist});

    if ($code eq 'OK') {
      changepage(page => "loclist", resultmsg => $msg);
    } else {
      $page->param(failed	=> 1);
      $page->param(errmsg	=> $msg);
      $page->param(wastrying	=> "editing");
      $page->param(todo		=> "Edit location/view");
      $page->param(locact	=> "update");
      $page->param(id		=> $webvar{loc});
      $page->param(locname	=> $webvar{locname});
      $page->param(comments	=> $webvar{comments});
      $page->param(iplist	=> $webvar{iplist});
    }
  } else {
    changepage(page => "loclist", errmsg => "You are not permitted to add locations/views", id => $webvar{parentid})
	unless ($permissions{admin} || $permissions{location_create});

    $page->param(todo => "Add location/view");
    $page->param(locact => "add");
    $page->param(locname => ($webvar{locname} ? $webvar{locname} : ''));
    $page->param(iplist => ($webvar{iplist} ? $webvar{iplist} : ''));

    show_msgs();
  }

} elsif ($webvar{page} eq 'delloc') {

  changepage(page=> "loclist", errmsg => "You are not allowed to delete locations")
	unless $permissions{admin} || $permissions{location_delete};

  # security check - does the user have permission to access this entity?
#  if (!check_scope(id => $webvar{id}, type => 'loc')) {
#    changepage(page => "loclist", errmsg => "You are not permitted to <foo> the requested location/view");
#  }

  $page->param(locid => $webvar{locid});
  my $locdata = $dnsdb->getLoc($webvar{locid});
  $locdata->{description} = $webvar{locid} if !$locdata->{description};
  # first pass = confirm y/n (sorta)
  if (!defined($webvar{del})) {
    $page->param(del_getconf => 1);
    $page->param(location => $locdata->{description});
  } elsif ($webvar{del} eq 'ok') {
    my ($code,$msg) = $dnsdb->delLoc($webvar{locid});
    if ($code eq 'OK') {
      # success.  go back to the user list, do not pass "GO"
      changepage(page => "loclist", resultmsg => $msg);
    } else {
      changepage(page => "loclist", errmsg => $msg);
    }
  } else {
    # cancelled.  whee!
    changepage(page => "loclist");
  }

} elsif ($webvar{page} eq 'dnsq') {

  if ($webvar{qfor}) {
    $webvar{qfor} =~ s/^\s*//;
    $webvar{qfor} =~ s/\s*$//;
    $page->param(qfor => $webvar{qfor});
  }
  if ($webvar{resolver}) {
    $webvar{resolver} =~ s/^\s*//;
    $webvar{resolver} =~ s/\s*$//;
    $page->param(resolver => $webvar{resolver});
  }
  $page->param(typelist => $dnsdb->getTypelist('l', ($webvar{type} ? $webvar{type} : undef)));
  $page->param(nrecurse => $webvar{nrecurse}) if $webvar{nrecurse};

  if ($webvar{qfor}) {
    my $resolv = Net::DNS::Resolver->new;
    $resolv->tcp_timeout(5);	# make me adjustable!
    $resolv->udp_timeout(5);	# make me adjustable!
    $resolv->recurse(0) if $webvar{nrecurse};
    $resolv->nameservers($webvar{resolver}) if $webvar{resolver};
    my $query = $resolv->query($webvar{qfor}, $typemap{$webvar{type}});
    if ($query) {

      $page->param(showresults => 1);

      my @answer;
      foreach my $rr ($query->answer) {
#	next unless $rr->type eq "A" or $rr->type eq 'NS';
	my %row;
	my ($host,$ttl,$class,$type,$data) =
		($rr->string =~ /^([0-9a-zA-Z_.-]+)\s+(\d+)\s+([A-Za-z]+)\s+([A-Za-z]+)\s+(.+)$/s);
	$row{host} = $host;
	$row{ftype} = $type;
	$row{rdata} = ($type eq 'SOA' ? "<pre>$data</pre>" : $data);
        push @answer, \%row;
      }
      $page->param(answer => \@answer);

      my @additional;
      foreach my $rr ($query->additional) {
#	next unless $rr->type eq "A" or $rr->type eq 'NS';
	my %row;
	my ($host,$ttl,$class,$type,$data) =
		($rr->string =~ /^([0-9a-zA-Z_.-]+)\s+(\d+)\s+([A-Za-z]+)\s+([A-Za-z]+)\s+(.+)$/);
	$row{host} = $host;
	$row{ftype} = $type;
	$row{rdata} = $data;
        push @additional, \%row;
      }
      $page->param(additional => \@additional);

      my @authority;
      foreach my $rr ($query->authority) {
#	next unless $rr->type eq "A" or $rr->type eq 'NS';
	my %row;
	my ($host,$ttl,$class,$type,$data) =
		($rr->string =~ /^([0-9a-zA-Z_.-]+)\s+(\d+)\s+([A-Za-z]+)\s+([A-Za-z]+)\s+(.+)$/);
	$row{host} = $host;
	$row{ftype} = $type;
	$row{rdata} = $data;
        push @authority, \%row;
      }
      $page->param(authority => \@authority);

      $page->param(usedresolver => $resolv->answerfrom);
      $page->param(frtype => $typemap{$webvar{type}});

    } else {
      $page->param(errmsg => $resolv->errorstring);
    }
  }
  ## done DNS query

} elsif ($webvar{page} eq 'axfr') {

  changepage(page => "domlist", errmsg => "You are not permitted to import domains")
	unless ($permissions{admin} || $permissions{domain_create});

  # don't need this while we've got the dropdown in the menu.  hmm.
  fill_grouplist("grouplist");

  $page->param(ifrom => $webvar{ifrom}) if $webvar{ifrom};
  $page->param(rwsoa => $webvar{rwsoa}) if $webvar{rwsoa};
  $page->param(rwns => $webvar{rwns}) if $webvar{rwns};
  $page->param(forcettl => $webvar{forcettl}) if $webvar{forcettl};
  $page->param(newttl => $webvar{newttl}) if $webvar{newttl};
  # This next one is arguably better on by default, but Breaking Things Is Bad, Mmmkay?
  $page->param(mergematching => $webvar{mergematching}) if $webvar{mergematching};
  $page->param(dominactive => 1) if (!$webvar{domactive} && $webvar{doit});	# eww.
  $page->param(importdoms => $webvar{importdoms}) if $webvar{importdoms};

  # shut up warning about uninitialized variable
  $webvar{doit} = '' if !defined($webvar{doit});

  if ($webvar{doit} eq 'y' && !$webvar{ifrom}) {
    $page->param(errmsg => "Need to set host to import from");
  } elsif ($webvar{doit} eq 'y' && !$webvar{importdoms}) {
    $page->param(errmsg => "Need domains to import");
  } elsif ($webvar{doit} eq 'y') {

    # security check - does the user have permission to access this entity?
    if (!check_scope(id => $webvar{group}, type => 'group')) {
      $page->param(errmsg => "You are not permitted to import domains into the requested group");
      goto DONEAXFR;
    }

    # Bizarre Things Happen when you AXFR a null-named zone.
    $webvar{importdoms} =~ s/^\s+//;
    my @domlist = split /\s+/, $webvar{importdoms};
    my @results;
    foreach my $domain (@domlist) {
##fixme: Net::DNS has made changes somewhere between 0.66something (~~ Debian wheezy) and
# 0.81 (~~ Debian jessie) that cause taint failures when providing a hostname as a nameserver
# for AXFR.  A proper fix may boil down to "split AXFR into its own script".  Feh.
# For now, we'll just convert the requested AXFR host to an IP, and pass that down the chain instead.
      my $nsip = gethostbyname($webvar{ifrom});
      use Socket;
      $nsip = inet_ntoa($nsip);
      # strangely enough we don't seem to need to detaint:
      #($nsip) = ($nsip =~ /^([a-fA-F0-9:.]+)$/);
      my %row;
      my ($code,$msg) = $dnsdb->importAXFR($nsip, $domain, $webvar{group},
	status => $webvar{domactive}, rwsoa => $webvar{rwsoa}, rwns => $webvar{rwns},
	newttl => ($webvar{forcettl} ? $webvar{newttl} : 0),
	merge => $webvar{mergematching});
      $row{domok} = $msg if $code eq 'OK';
      if ($code eq 'WARN') {
	$msg =~ s|\n|<br />|g;
	$row{domwarn} = $msg;
      }
      if ($code eq 'FAIL') {
	$msg =~ s|\n|<br />\n|g;
	$row{domerr} = $msg;
      }
      $msg = "<br />\n".$msg if $msg =~ m|<br />|;
      $row{domain} = $domain;
      push @results, \%row;
    }
    $page->param(axfrresults => \@results);
  }

  # Yes, this is a GOTO target.  PTBHTTT.
  DONEAXFR: ;

} elsif ($webvar{page} eq 'whoisq') {

  if ($webvar{qfor}) {
    use Net::Whois::Raw;
    use Text::Wrap;

# caching useful?
#$Net::Whois::Raw::CACHE_DIR = "/var/spool/pwhois/";
#$Net::Whois::Raw::CACHE_TIME = 60;

    my ($dominfo, $whois_server) = whois($webvar{qfor});
##fixme:  if we're given an IP, try rwhois as well as whois so we get the real final data

    # le sigh.  idjits spit out data without linefeeds...
    $Text::Wrap::columns = 88;

# &%$@%@# high-bit crap.  We should probably find a way to properly recode these
# instead of one-by-one.  Note CGI::Simple's escapeHTML() doesn't do more than
# the bare minimum.  :/
# Mainly an XHTML validation thing.
    $dominfo = $q->escapeHTML($dominfo);
    $dominfo =~ s/\xa9/\&copy;/g;
    $dominfo =~ s/\xae/\&reg;/g;

    $page->param(qfor => $webvar{qfor});
    $page->param(dominfo => wrap('','',$dominfo));
    $page->param(whois_server => $whois_server);
  } else {
    $page->param(errmsg => "Missing host or domain to query in WHOIS") if $webvar{askaway};
  }

} elsif ($webvar{page} eq 'log') {

  my $id = $curgroup;  # we do this because the group log may be called from (almost) any page,
                       # but the others are much more limited.  this is probably non-optimal.

  if ($webvar{ltype} && $webvar{ltype} eq 'user') {
##fixme:  where should we call this from?
    $id = $webvar{id};
##fixme:  don't include username on out-of-scope users
    $page->param(logfor => 'user '.($id ? $dnsdb->userFullName($id) : $webvar{fname}));
  } elsif ($webvar{ltype} && $webvar{ltype} eq 'dom') {
    $id = $webvar{id};
    if (!check_scope(id => $id, type => 'domain')) {
      $page->param(errmsg => "You are not permitted to view log entries for the requested domain");
      goto DONELOG;
    }
    $page->param(logfor => 'domain '.$dnsdb->domainName($id));
  } elsif ($webvar{ltype} && $webvar{ltype} eq 'rdns') {
    $id = $webvar{id};
    if (!check_scope(id => $id, type => 'revzone')) {
      $page->param(errmsg => "You are not permitted to view log entries for the requested reverse zone");
      goto DONELOG;
    }
    $page->param(logfor => 'reverse zone '.$dnsdb->revName($id));
  } else {
    # Default to listing curgroup log
    $page->param(logfor => 'group '.$dnsdb->groupName($id));
    # note that scope limitations are applied via the change-group check;
    # group log is always for the "current" group
  }
  $webvar{ltype} = 'group' if !$webvar{ltype};

  # Done here since we want to allow more arbitrary blobs in the log filter
  if (defined($webvar{logfilter})) {
    $session->param('logfilter', '') if !$session->param('logfilter');
    if ($webvar{logfilter} ne $session->param('logfilter')) {
      $uri_self =~ s/\&amp;offset=[^&]//;
      $offset = 0;
    }
    $session->param('logfilter', $webvar{logfilter})
  }
  my $logfilter = $session->param('logfilter');
  $filter = $logfilter;
  $page->param(logfilter => $logfilter);

  my $lcount = $dnsdb->getLogCount(id => $id, group => $logingroup, fname => $webvar{fname},
    logtype => $webvar{ltype}, filter => $logfilter);
  if (!$lcount) {
    $page->param(errmsg => $dnsdb->errstr);
    $lcount = 0;
  }

  $page->param(id => $id);
  $page->param(ltype => $webvar{ltype});

  fill_fpnla($lcount);
  fill_pgcount($lcount, "log entries", '');
  $page->param(curpage => $webvar{page}.($webvar{ltype} ? "&amp;ltype=$webvar{ltype}" : ''));

  $sortby = 'stamp';
  $sortorder = 'DESC';	# newest-first;  although filtering is probably going to be more useful than sorting
# sort/order
  $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
  $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};

  $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
  $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

  # Set up the column headings with the sort info
  my @cols = ('fname','domain','revzone','entry','stamp');
  my %colnames = (fname => 'Name', domain => 'Forward zone', revzone => 'Reverse zone',
      entry => 'Log Entry', stamp => 'Date/Time');
  fill_colheads($sortby, $sortorder, \@cols, \%colnames);

##fixme:  increase per-page limit or use separate limit for log?  some ops give *lots* of entries...
  my $logentries = $dnsdb->getLogEntries(id => $id, group => $logingroup, fname => $webvar{fname},
        logtype => $webvar{ltype},
	offset => $webvar{offset}, sortby => $sortby, sortorder => $sortorder, filter => $logfilter);
  if (!$logentries) {
    $page->param(errmsg => $dnsdb->errstr);
  } else {
    # undef $logentries is inexplicably puking instead of showing "no entries found",
    # like all the rest of the methods that return a list the same way.  idunno...
    $page->param(logentries => $logentries);
  }

##fixme:
# - on log record creation, bundle "parented" log actions (eg, "AXFR record blah for domain foo",
#   or "Add record bar for new domain baz") into one entry (eg, "AXFR domain foo", "Add domain baz")?
#   need a way to expand this into the complete list, and to exclude "child" entries

  # scope check fail target
  DONELOG: ;

} elsif ($webvar{page} eq 'recsearch') {

  # we do this for the domain and record list filter/search - it should be extremely rare to
  # need to search on characters outside this set until we get into IDNs
  # note this is a little larger due to template records
  # allow <>= so searches can use the Postgres CIDR operators
  # allow , for things like DMARC records
  $webvar{searchfor} =~ s{[^a-zA-Z0-9_.,:\@%<>=/-]}{}g if $webvar{searchfor};

  # save the search in the session, same as the "filter" in various other lists...
  if (defined($webvar{searchfor})) {
    if ($session->param('recsearch') && $webvar{searchfor} ne $session->param('recsearch')) {
      $uri_self =~ s/\&amp;offset=[^&]//;
      $offset = 0;
    }
    $session->param(recsearch => $webvar{searchfor});
  }
  my $searchfor = $session->param('recsearch');

  $sortby = 'host';
  $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
  $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};
  $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
  $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

  # some magic to label and linkify the column headers for sorting
  my @cols = ('domain','revzone','host','type','val');
  my %colheads = (domain => "Domain (Group)", revzone => "Reverse zone (Group)", host => "Host",
	type => "Type", val => "IP/value");
  # only users allowed to see location/view data get this column
  if ($permissions{admin} || $permissions{location_view}) {
    $colheads{location} = "Location";
    push @cols, 'location';
  }
  fill_colheads($sortby, $sortorder, \@cols, \%colheads);

  # pgcount.tmpl
  my $count = $dnsdb->recSearchCount(searchfor => $searchfor, group => $logingroup);
  fill_pgcount($count, "records");
  fill_fpnla($count);

  # and a bit for fpnla.tmpl
  $page->param(curpage => $webvar{page});

  $page->param(searchfor => $searchfor);
  my $recset = $dnsdb->recSearch(searchfor => $searchfor, group => $logingroup, offset => $webvar{offset},
    sortby => $sortby, sortorder => $sortorder);
  $page->param(searchresults => $recset);

} # end $webvar{page} dance


# start output here so we can redirect pages.
print $q->header( -cookie => $sesscookie);
print $header->output;

##common bits
# mostly things in the menu
if ($webvar{page} ne 'login' && $webvar{page} ne 'badpage') {
  $page->param(username => $session->param("username"));

  $page->param(group => $curgroup);
  $page->param(groupname => $dnsdb->groupName($curgroup));
  $page->param(logingrp => $dnsdb->groupName($logingroup));
  $page->param(logingrp_num => $logingroup);

##fixme
  $page->param(mayrdns => 1);
  $page->param(maysecondary => 1);

  $page->param(mayloc => ($permissions{admin} || $permissions{location_view}));

  $page->param(maydefrec => $permissions{admin});
  $page->param(mayimport => $permissions{admin} || $permissions{domain_create});
  $page->param(maybulk => $permissions{admin} || $permissions{domain_edit} || $permissions{domain_create} || $permissions{domain_delete});

  $page->param(chggrps => ($permissions{admin} || $permissions{group_create} || $permissions{group_edit} || $permissions{group_delete}));

  # group tree.  should go elsewhere, probably
  my $tmpgrplist = fill_grptree($logingroup,$curgroup);
  $page->param(grptree => $tmpgrplist);
  $page->param(subs => ($tmpgrplist ? 1 : 0));	# probably not useful to pass gobs of data in for a boolean
  $page->param(inlogingrp => $curgroup == $logingroup);

# fill in the URL-to-self for the group tree and search-by-letter
  $page->param(whereami => $uri_self);
# fill in general URL-to-self
  $page->param(script_self => "$ENV{SCRIPT_NAME}?");
# fill in the generalized path-to-instance
  $page->param(webpath => ($ENV{SCRIPT_NAME} =~ m|(/.+)/[^/]+$|)[0]);
}

if (@debugbits) {
  print "<pre>\n";
  foreach (@debugbits) { print; }
  print "</pre>\n";
}

# spit it out
print $page->output;

if ($debugenv) {
  print "<div id=\"debug\">webvar keys: <pre>\n";
  foreach my $key (keys %webvar) {
    print "key: $key\tval: $webvar{$key}\n";
  }
  print "</pre>\nsession:\n<pre>\n";
  my $sesdata = $session->dataref();
  foreach my $key (keys %$sesdata) {
    print "key: $key\tval: ".$sesdata->{$key}."\n";
  }
  print "</pre>\nENV:\n<pre>\n";
  foreach my $key (keys %ENV) {
    print "key: $key\tval: $ENV{$key}\n";
  }
  print "</pre></div>\n";
}

print $footer->output;

# as per the docs, Just In Case
$session->flush();

exit 0;


sub fill_grptree {
  my $root = shift;
  my $cur = shift;
  my $indent = shift || '    ';

  my @childlist;

  # some magic to control bad offsets on group change
  my $grp_uri_self = $uri_self;
  $grp_uri_self =~ s/\&amp;offset=[^&]+// unless ($webvar{page} eq 'reclist' && $webvar{defrec} eq 'n');

  my $grptree = HTML::Template->new(filename => 'templates/grptree.tmpl');
  $dnsdb->getChildren($root, \@childlist, 'immediate');
  return if $#childlist == -1;
  my @grouplist;
  foreach (@childlist) {
    my %row;
    $row{grpname} = $dnsdb->groupName($_);
    $row{grpnum} = $_;
    $row{whereami} = $grp_uri_self;
    $row{curgrp} = ($_ == $cur);
    $row{expanded} = $dnsdb->isParent($_, 'group', $cur, 'group');
    $row{expanded} = 1 if $_ == $cur;
    $row{subs} = fill_grptree($_,$cur,$indent.'    ');
    $row{indent} = $indent;
    push @grouplist, \%row;
  }
  $grptree->param(indent => $indent);
  $grptree->param(treelvl => \@grouplist);
  return $grptree->output;
}

sub changepage {
  my %params = @_;	# think this works the way I want...

  # cross-site scripting fixup.  instead of passing error messages by URL/form
  # variable, put them in the session where the nasty user can't meddle.
  # these are done here since it's far simpler to pass them in from wherever
  # than set them locally everywhere.
  foreach my $sessme ('resultmsg','warnmsg','errmsg') {
    if (my $tmp = $params{$sessme}) {
      $tmp =~ s/^\n//;
      $tmp =~ s|\n|<br />\n|g;
      $session->param($sessme, $tmp);
      delete $params{$sessme};
    }
  }

  # handle user check
  my $newurl = "http://$ENV{HTTP_HOST}$ENV{SCRIPT_NAME}?";
  foreach (sort keys %params) {
## fixme:  something is undefined here on add location
    $newurl .= "&$_=".$q->url_encode($params{$_});
  }

  # Just In Case
  $session->flush();

  print $q->redirect ( -url => $newurl, -cookie => $sesscookie);
  exit;
} # end changepage

# wrap up the usual suspects for result, warning, or error messages to be displayed
sub show_msgs {
  if ($session->param('resultmsg')) {
    $page->param(resultmsg => $session->param('resultmsg'));
    $session->clear('resultmsg');
  }
  if ($session->param('warnmsg')) {
    $page->param(warnmsg => $session->param('warnmsg'));
    $session->clear('warnmsg');
  }
  if ($session->param('errmsg')) {
    $page->param(errmsg => $session->param('errmsg'));
    $session->clear('errmsg');
  }
} # end show_msgs

sub fillsoa {
  my $defrec = shift;
  my $revrec = shift;
  my $id = shift;
  my $preserve = shift || 'd';	# Flag to use webvar fields or retrieve from database

  my $domname = ($defrec eq 'y' ? '' : "DOMAIN");

  $page->param(defrec	=> $defrec);
  $page->param(revrec	=> $revrec);

# i had a good reason to do this when I wrote it...
#  $page->param(domain	=> $domname);
#  $page->param(group	=> $DNSDB::group);
  $page->param(isgrp => 1) if $defrec eq 'y';
  $page->param(parent => ($defrec eq 'y' ? $dnsdb->groupName($id) :
	($revrec eq 'n' ? $dnsdb->domainName($id) : $dnsdb->revName($id)) ) );

# defaults
  $page->param(defcontact	=> $DNSDB::def{contact});
  $page->param(defns		=> $DNSDB::def{prins});
  $page->param(defsoattl	=> $DNSDB::def{soattl});
  $page->param(defrefresh	=> $DNSDB::def{refresh});
  $page->param(defretry		=> $DNSDB::def{retry});
  $page->param(defexpire	=> $DNSDB::def{expire});
  $page->param(defminttl	=> $DNSDB::def{minttl});

  $page->param(id	=> $id);

  if ($preserve eq 'd') {
    # there are probably better ways to do this.  TMTOWTDI.
    my $soa = $dnsdb->getSOA($defrec, $revrec, $id);

    $page->param(prins	=> ($soa->{prins} ? $soa->{prins} : $DNSDB::def{prins}));
    $page->param(contact	=> ($soa->{contact} ? $soa->{contact} : $DNSDB::def{contact}));
    $page->param(refresh	=> ($soa->{refresh} ? $soa->{refresh} : $DNSDB::def{refresh}));
    $page->param(retry	=> ($soa->{retry} ? $soa->{retry} : $DNSDB::def{retry}));
    $page->param(expire	=> ($soa->{expire} ? $soa->{expire} : $DNSDB::def{expire}));
    $page->param(minttl	=> ($soa->{minttl} ? $soa->{minttl} : $DNSDB::def{minttl}));
    $page->param(ttl	=> ($soa->{ttl} ? $soa->{ttl} : $DNSDB::def{soattl}));
  } else {
    $page->param(prins	=> ($webvar{prins} ? $webvar{prins} : $DNSDB::def{prins}));
    $page->param(contact	=> ($webvar{contact} ? $webvar{contact} : $DNSDB::def{contact}));
    $page->param(refresh	=> ($webvar{refresh} ? $webvar{refresh} : $DNSDB::def{refresh}));
    $page->param(retry	=> ($webvar{retry} ? $webvar{retry} : $DNSDB::def{retry}));
    $page->param(expire	=> ($webvar{expire} ? $webvar{expire} : $DNSDB::def{expire}));
    $page->param(minttl	=> ($webvar{minttl} ? $webvar{minttl} : $DNSDB::def{minttl}));
    $page->param(ttl	=> ($webvar{ttl} ? $webvar{ttl} : $DNSDB::def{soattl}));
  }
}

sub showzone {
  my $def = shift;
  my $rev = shift;
  my $id = shift;

  # get the SOA first
  my $soa = $dnsdb->getSOA($def, $rev, $id);

  $page->param(contact	=> $soa->{contact});
  $page->param(prins	=> $soa->{prins});
  $page->param(serial   => $soa->{serial});
  $page->param(refresh	=> $soa->{refresh});
  $page->param(retry	=> $soa->{retry});
  $page->param(expire	=> $soa->{expire});
  $page->param(minttl	=> $soa->{minttl});
  $page->param(ttl	=> $soa->{ttl});

  my $foo2 = $dnsdb->getRecList(defrec => $def, revrec => $rev, id => $id, offset => $webvar{offset},
	sortby => $sortby, sortorder => $sortorder, filter => $filter);

  foreach my $rec (@$foo2) {
    if ($typemap{$rec->{type}}) {
      $rec->{type} = $typemap{$rec->{type}};
    } else {
      $rec->{type} = "TYPE$rec->{type}";
    }
    $rec->{fwdzone} = $rev eq 'n';
    $rec->{ttl} = '(auto)' if $rec->{ttl} == -1;
    $rec->{distance} = 'n/a' unless ($rec->{type} eq 'MX' || $rec->{type} eq 'SRV'); 
    $rec->{weight} = 'n/a' unless ($rec->{type} eq 'SRV'); 
    $rec->{port} = 'n/a' unless ($rec->{type} eq 'SRV');
# ACLs
    $rec->{record_edit} = ($permissions{admin} || $permissions{record_edit});
    $rec->{record_delete} = ($permissions{admin} || $permissions{record_delete});
    $rec->{locname} = '' unless ($permissions{admin} || $permissions{location_view});
# Timestamps
    if ($rec->{expires}) {
      $rec->{stamptype} = $rec->{ispast} ? 'expired at' : 'expires at';
    } else {
      $rec->{stamptype} = 'valid after';
    }
    # strip seconds and timezone?  no, not yet.  could probably offer a config knob on this display at some point.
#    $rec->{stamp} =~ s/:\d\d-\d+$//;
    delete $rec->{expires};
    delete $rec->{ispast};
  }
  $page->param(reclist => $foo2);
}

sub fill_recdata {
  # le sigh.  we may get called with many empty %webvar keys
  no warnings qw( uninitialized );

##todo:  allow BIND-style bare names, ASS-U-ME that the name is within the domain?
# prefill <domain> or DOMAIN in "Host" space for new records
  if ($webvar{revrec} eq 'n') {
    $page->param(typelist => $dnsdb->getTypelist($webvar{revrec}, $webvar{type}));
    my $domroot = ($webvar{defrec} eq 'y' ? 'DOMAIN' : $dnsdb->domainName($webvar{parentid}));
    $page->param(name	=> ($webvar{name} ? $webvar{name} : $domroot));
    $page->param(address	=> $webvar{address});
    $page->param(distance	=> $webvar{distance})
	if ($webvar{type} == $reverse_typemap{MX} or $webvar{type} == $reverse_typemap{SRV});
    $page->param(weight	=> $webvar{weight}) if $webvar{type} == $reverse_typemap{SRV};
    $page->param(port	=> $webvar{port}) if $webvar{type} == $reverse_typemap{SRV};
  } else {
    my $domroot = ($webvar{defrec} eq 'y' ? 'ADMINDOMAIN' : ".$dnsdb->{domain}");
    $page->param(name	=> ($webvar{name} ? $webvar{name} : $domroot));
    my $zname = ($webvar{defrec} eq 'y' ? 'ZONE' : $dnsdb->revName($webvar{parentid}, 'y'));
    $zname =~ s|\d*/\d+$||;
    $page->param(address	=> ($webvar{address} ? $webvar{address} : $zname));
    $page->param(typelist => $dnsdb->getTypelist($webvar{revrec},
	$webvar{type} || ($zname =~ /:/ ? $reverse_typemap{'AAAA+PTR'} : $reverse_typemap{'A+PTR'})));
  }
# retrieve the right ttl instead of falling (way) back to the hardcoded system default
  my $soa = $dnsdb->getSOA($webvar{defrec}, $webvar{revrec}, $webvar{parentid});
  $page->param(ttl	=> ($webvar{ttl} ? $webvar{ttl} : $soa->{minttl}));
  $page->param(stamp_until => ($webvar{expires} eq 'until'));
  $page->param(stamp => $webvar{stamp});
}

sub fill_actypelist {
  my $curtype = shift || 'u';

  my @actypes;

  my %row1 = (actypeval => 'u', actypename => 'user');
  $row1{typesel} = 1 if $curtype eq 'u';
  push @actypes, \%row1;

  my %row2 = (actypeval => 'S', actypename => 'superuser');
  $row2{typesel} = 1 if $curtype eq 'S';
  push @actypes, \%row2;

  $page->param(actypelist => \@actypes);
}

sub fill_clonemelist {
  # shut up some warnings, but don't stomp on caller's state
  local $webvar{clonesrc} = 0 if !defined($webvar{clonesrc});

  my $clones = $dnsdb->getUserDropdown($curgroup, $webvar{clonesrc});
  $page->param(clonesrc => $clones);
}

sub fill_fpnla {
  my $count = shift;
  if ($offset eq 'all') {
    $page->param(perpage => $perpage);
# uhm....
  } else {
    # all these bits only have sensible behaviour if offset is numeric. err, probably.
    if ($count > $perpage) {
      # if there are more results than the default, always show the "all" link
      $page->param(navall => 1);

      if ($offset > 0) {
        $page->param(navfirst => 1);
        $page->param(navprev => 1);
        $page->param(prevoffs => $offset-1);
      }

      # show "next" and "last" links if we're not on the last page of results
      if ( (($offset+1) * $perpage - $count) < 0 ) {
        $page->param(navnext => 1);
        $page->param(nextoffs => $offset+1);
        $page->param(navlast => 1);
        $page->param(lastoffs => int (($count-1)/$perpage));
      }
    } else {
      $page->param(onepage => 1);
    }
  }
} # end fill_fpnla()

sub fill_pgcount {
  my $pgcount = shift;
  my $pgtype = shift;
  my $parent = shift;

  $page->param(ntot => $pgcount);
  $page->param(nfirst => (($offset eq 'all' ? 0 : $offset)*$perpage+1));
  $page->param(npglast => ($offset eq 'all' ? $pgcount :
	( (($offset+1)*$perpage) > $pgcount ? $pgcount : (($offset+1)*$perpage) )
	));
  $page->param(pgtype => $pgtype);
  $page->param(parent => $parent);
  $page->param(filter => $filter);
} # end fill_pgcount()


sub listdomains { listzones(); }	# temp

sub listzones {
# ACLs
  $page->param(domain_create	=> ($permissions{admin} || $permissions{domain_create}) );
  $page->param(domain_edit	=> ($permissions{admin} || $permissions{domain_edit}) );
  $page->param(domain_delete	=> ($permissions{admin} || $permissions{domain_delete}) );

  my @childgroups;
  $dnsdb->getChildren($curgroup, \@childgroups, 'all') if $searchsubs;
  my $childlist = join(',',@childgroups);

  my $count = $dnsdb->getZoneCount(childlist => $childlist, curgroup => $curgroup, revrec => $webvar{revrec},
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef) );

# fill page count and first-previous-next-last-all bits
  fill_pgcount($count,($webvar{revrec} eq 'n' ? 'domains' : 'revzones'),$dnsdb->groupName($curgroup));
  fill_fpnla($count);

  $sortby = ($webvar{revrec} eq 'n' ? 'domain' : 'revnet');
# sort/order
  $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
  $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};

  $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
  $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

# set up the headers
  my @cols = (($webvar{revrec} eq 'n' ? 'domain' : 'revnet'), 'status', 'group');
  my %colheads = (domain => 'Domain', revnet => 'Reverse Zone', status => 'Status', group => 'Group');
  fill_colheads($sortby, $sortorder, \@cols, \%colheads);

  # hack! hack! pthbttt.  have to rethink the status column storage,
  # or inactive comes "before" active.  *sigh*
  $sortorder = ($sortorder eq 'ASC' ? 'DESC' : 'ASC') if $sortby eq 'status';

# waffle, waffle - keep state on these as well as sortby, sortorder?
##fixme:  put this higher so the count doesn't get munched?
  $page->param("start$startwith" => 1) if $startwith && $startwith =~ /^(?:[a-z]|0-9)$/;

  $page->param(filter => $filter) if $filter;
  $page->param(searchsubs => $searchsubs) if $searchsubs;

  $page->param(group => $curgroup);

  my $zonelist = $dnsdb->getZoneList(childlist => $childlist, curgroup => $curgroup, revrec => $webvar{revrec},
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef),
	offset => $offset, sortby => $sortby, sortorder => $sortorder
	);
# probably don't need this, keeping for reference for now
#  foreach my $rec (@$zonelist) {
#  }
  $page->param(domtable => $zonelist);
} # end listzones()


sub listsecondaryzones {
# ACLs
  $page->param(secondary_create	=> ($permissions{admin} || $permissions{domain_create}) );
  $page->param(secondary_edit	=> ($permissions{admin} || $permissions{domain_edit}) );
  $page->param(secondary_delete	=> ($permissions{admin} || $permissions{domain_delete}) );

  my @childgroups;
  $dnsdb->getChildren($curgroup, \@childgroups, 'all') if $searchsubs;
  my $childlist = join(',',@childgroups);

#  my $count = $dnsdb->getSecondaryZoneCount(childlist => $childlist, curgroup => $curgroup,
  my $count = $dnsdb->getZoneCount(childlist => $childlist, curgroup => $curgroup, secondary => 'y',
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef) );

# fill page count and first-previous-next-last-all bits
  fill_pgcount($count, 'secondary zone(s)', $dnsdb->groupName($curgroup));
  fill_fpnla($count);

  $sortby = 'zone';
# sort/order
  $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
  $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};

  $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
  $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

# set up the headers
  my @cols = ('secondary', 'primaryserver', 'status', 'group');
  my %colheads = (secondary => 'Secondary zone', primaryserver => 'Primary server(s)', status => 'Status', group => 'Group');
  fill_colheads($sortby, $sortorder, \@cols, \%colheads);

  # hack! hack! pthbttt.  have to rethink the status column storage,
  # or inactive comes "before" active.  *sigh*
  $sortorder = ($sortorder eq 'ASC' ? 'DESC' : 'ASC') if $sortby eq 'status';

# waffle, waffle - keep state on these as well as sortby, sortorder?
##fixme:  put this higher so the count doesn't get munched?
  $page->param("start$startwith" => 1) if $startwith && $startwith =~ /^(?:[a-z]|0-9)$/;

  $page->param(filter => $filter) if $filter;
  $page->param(searchsubs => $searchsubs) if $searchsubs;

  $page->param(group => $curgroup);

#  my $zonelist = $dnsdb->getSecondaryZoneList(childlist => $childlist, curgroup => $curgroup,
  my $zonelist = $dnsdb->getZoneList(childlist => $childlist, curgroup => $curgroup, secondary => 'y',
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef),
	offset => $offset, sortby => $sortby, sortorder => $sortorder
	);
# probably don't need this, keeping for reference for now
#  foreach my $rec (@$zonelist) {
#  }
  $page->param(zonetable => $zonelist);
} # end listsecondaryzones()


sub listgroups {

# security check - does the user have permission to view this entity?
  if (!(grep /^$curgroup$/, @viewablegroups)) {
    # hmm.  Reset the current group to the login group?  Yes.  Prevents confusing behaviour elsewhere.
    $session->param('curgroup',$logingroup);
    $page->param(errmsg => "You are not permitted to view the requested group");
    $curgroup = $logingroup;
  }

  my @childgroups;
  $dnsdb->getChildren($curgroup, \@childgroups, 'all') if $searchsubs;
  my $childlist = join(',',@childgroups);

  my ($count) = $dnsdb->getGroupCount(childlist => $childlist, curgroup => $curgroup,
        filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef) );

# fill page count and first-previous-next-last-all bits
  fill_pgcount($count,"groups",'');
  fill_fpnla($count);

  $page->param(gid => $curgroup);

  $sortby = 'group';
# sort/order
  $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
  $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};

  $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
  $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

# set up the headers
  my @cols = ('group','parent','nusers','ndomains','nrevzones');
  my %colnames = (group => 'Group', parent => 'Parent Group', nusers => 'Users', ndomains => 'Domains', nrevzones => 'Reverse Zones');
  fill_colheads($sortby, $sortorder, \@cols, \%colnames);

# waffle, waffle - keep state on these as well as sortby, sortorder?
  $page->param("start$startwith" => 1) if $startwith && $startwith =~ /^(?:[a-z]|0-9)$/;

  $page->param(filter => $filter) if $filter;
  $page->param(searchsubs => $searchsubs) if $searchsubs;

# munge sortby for columns in database
  $sortby = 'g.group_name' if $sortby eq 'group';
  $sortby = 'g2.group_name' if $sortby eq 'parent';

  my $glist = $dnsdb->getGroupList(childlist => $childlist, curgroup => $curgroup,
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef),
	offset => $webvar{offset}, sortby => $sortby, sortorder => $sortorder);

  $page->param(grouptable => $glist);
} # end listgroups()


sub fill_grouplist {
  my $template_var = shift;
  my $cur = shift || $curgroup;

  # little recursive utility sub-sub
  sub getgroupdrop {
    my $root = shift;
    my $cur = shift;	# to tag the selected group
    my $grplist = shift;
    my $indent = shift || '&nbsp;&nbsp;&nbsp;&nbsp;';

    my @childlist;
    $dnsdb->getChildren($root, \@childlist, 'immediate');
    return if $#childlist == -1;
    foreach (@childlist) {
      my %row;
      $row{groupval} = $_;
      $row{groupactive} = ($_ == $cur);
      $row{groupname} = $indent.$dnsdb->groupName($_);
      push @{$grplist}, \%row;
      getgroupdrop($_, $cur, $grplist, $indent.'&nbsp;&nbsp;&nbsp;&nbsp;');
    }
  }

  my @grouplist;
  push @grouplist, { groupval => $logingroup, groupactive => $logingroup == $curgroup,
	groupname => $dnsdb->groupName($logingroup) };
  getgroupdrop($logingroup, $curgroup, \@grouplist);

  $page->param("$template_var" => \@grouplist);
} # end fill_grouplist()


sub fill_loclist {
  my $cur = shift || $curgroup;
  my $defloc = shift || '';

  return unless ($permissions{admin} || $permissions{location_view});

  $page->param(location_view => ($permissions{admin} || $permissions{location_view}));

  if ($permissions{admin} || $permissions{record_locchg}) {
    my $loclist = $dnsdb->getLocDropdown($cur, $defloc);
    $page->param(record_locchg => 1);
    $page->param(loclist => $loclist);
  } else {
    my $loc = $dnsdb->getLoc($defloc);
    $page->param(loc_name => $loc->{description});
  }
} # end fill_loclist()


sub list_users {

  my @childgroups;
  $dnsdb->getChildren($curgroup, \@childgroups, 'all') if $searchsubs;
  my $childlist = join(',',@childgroups);

  my $count = $dnsdb->getUserCount(childlist => $childlist, curgroup => $curgroup,
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef) );

# fill page count and first-previous-next-last-all bits
  fill_pgcount($count,"users",'');
  fill_fpnla($count);

  $sortby = 'user';
# sort/order
  $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
  $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};

  $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
  $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

# set up the headers
  my @cols = ('user','fname','type','group','status');
  my %colnames = (user => 'Username', fname => 'Full Name', type => 'Type', group => 'Group', status => 'Status');
  fill_colheads($sortby, $sortorder, \@cols, \%colnames);

# waffle, waffle - keep state on these as well as sortby, sortorder?
  $page->param("start$startwith" => 1) if $startwith && $startwith =~ /^(?:[a-z]|0-9)$/;

  $page->param(filter => $filter) if $filter;
  $page->param(searchsubs => $searchsubs) if $searchsubs;

  my $ulist = $dnsdb->getUserList(childlist => $childlist, curgroup => $curgroup,
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef),
	offset => $webvar{offset}, sortby => $sortby, sortorder => $sortorder);
  # Some UI things need to be done to the list (unlike other lists)
  foreach my $u (@{$ulist}) {
    $u->{eduser} = ($permissions{admin} ||
	($permissions{user_edit} && $u->{type} ne 'S') ||
	($permissions{self_edit} && $u->{user_id} == $session->param('uid')) );
    $u->{deluser} = ($permissions{admin} || ($permissions{user_delete} && $u->{type} ne 'S'));
    $u->{type} = ($u->{type} eq 'S' ? 'superuser' : 'user');
  }
  $page->param(usertable => $ulist);
} # end list_users()


sub list_locations {

  my @childgroups;
  $dnsdb->getChildren($curgroup, \@childgroups, 'all') if $searchsubs;
  my $childlist = join(',',@childgroups);

  my $count = $dnsdb->getLocCount(childlist => $childlist, curgroup => $curgroup,
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef) );

# fill page count and first-previous-next-last-all bits
  fill_pgcount($count,"locations/views",'');
  fill_fpnla($count);

  $sortby = 'user';
# sort/order
  $session->param($webvar{page}.'sortby', $webvar{sortby}) if $webvar{sortby};
  $session->param($webvar{page}.'order', $webvar{order}) if $webvar{order};

  $sortby = $session->param($webvar{page}.'sortby') if $session->param($webvar{page}.'sortby');
  $sortorder = $session->param($webvar{page}.'order') if $session->param($webvar{page}.'order');

# set up the headers
  my @cols = ('description', 'iplist', 'group');
  my %colnames = (description => 'Location/View Name', iplist => 'Permitted IPs/Ranges', group => 'Group');
  fill_colheads($sortby, $sortorder, \@cols, \%colnames);

# waffle, waffle - keep state on these as well as sortby, sortorder?
  $page->param("start$startwith" => 1) if $startwith && $startwith =~ /^(?:[a-z]|0-9)$/;

  $page->param(filter => $filter) if $filter;
  $page->param(searchsubs => $searchsubs) if $searchsubs;

  my $loclist = $dnsdb->getLocList(childlist => $childlist, curgroup => $curgroup,
	filter => ($filter ? $filter : undef), startwith => ($startwith ? $startwith : undef),
	offset => $webvar{offset}, sortby => $sortby, sortorder => $sortorder);
  # Some UI things need to be done to the list
  foreach my $l (@{$loclist}) {
    $l->{iplist} = "(All IPs)" if !$l->{iplist};
    $l->{edloc} = ($permissions{admin} || $permissions{loc_edit});
    $l->{delloc} = ($permissions{admin} || $permissions{loc_delete});
  }
  $page->param(loctable => $loclist);
} # end list_locations()


# Generate all of the glop necessary to add or not the appropriate marker/flag for
# the sort order and column in domain, user, group, and record lists
# Takes an array ref and hash ref
sub fill_colheads {
  my $sortby = shift;
  my $sortorder = shift;
  my $cols = shift;
  my $colnames = shift;
  my $custom = shift;

  my @headings;

  foreach my $col (@$cols) {
    my %coldata;
    $coldata{page} = $webvar{page};
    $coldata{offset} = $webvar{offset} if $webvar{offset};
    $coldata{sortby} = $col;
    $coldata{colname} = $colnames->{$col};
    if ($col eq $sortby) {
      $coldata{order} = ($sortorder eq 'ASC' ? 'DESC' : 'ASC');
      $coldata{sortorder} = $sortorder;
    } else {
      $coldata{order} = 'ASC';
    }
    if ($custom) {
      foreach my $ckey (keys %$custom) {
        $coldata{$ckey} = $custom->{$ckey};
      }
    }
    push @headings, \%coldata;
  }

  $page->param(colheads => \@headings);

} # end fill_colheads()


# we have to do this in a variety of places;  let's make it consistent
sub fill_permissions {
  my $template = shift;	# may need to do several sets on a single page
  my $permset = shift;	# hashref to permissions on object
  my $usercan = shift || \%permissions;	# allow alternate user-is-allowed permission block

  foreach (@permtypes) {
    $template->param("may_$_" => ($usercan->{admin} || $usercan->{$_}));
    $template->param($_ => $permset->{$_});
  }
}

# so simple when defined as a sub instead of inline.  O_o
sub check_scope {
  my %args = @_;
  my $entity = $args{id} || 0;	# prevent the shooting of feet with SQL "... intcolumn = '' ..."
  my $entype = $args{type} || '';

  if ($entype eq 'group') {
    return 1 if grep /^$entity$/, @viewablegroups;
  } else {
    foreach (@viewablegroups) {
      return 1 if $dnsdb->isParent($_, 'group', $entity, $entype);
    }
  }
}
