#!/usr/bin/perl
# Main add-IP-to-list CGI
##
# $Id: dnsbl.cgi 76 2025-09-09 17:32:40Z kdeugau $
# Copyright 2009-2012,2014,2015,2017,2018,2025 Kris Deugau <kdeugau@deepnet.cx>
# 
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version. 
# 
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
# 
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##

use strict;
use warnings;
no warnings qw(uninitialized);

use CGI::Carp qw (fatalsToBrowser);
use CGI::Simple;
use HTML::Template;
use Net::DNS;

# push "the directory the script is in" into @INC
use FindBin;
use lib "$FindBin::RealBin/";

use DNSBL 3.0;
use DNSBLweb;

# Set up the CGI object...
my $q = new CGI::Simple;
# ... and get query-string params as well as POST params if necessary
$q->parse_query_string;

my %webvar;
# This is probably excessive fiddling, but it puts the parameters somewhere my fingers know about...
foreach ($q->param()) {
  $webvar{$_} = $q->param($_);
}

my $dnsbl = new DNSBL;
# here be drag'ns, should theoretically be $DNSBL::maxlvl, but we
# only have up to level 4 in the report HTML/template
my $maxlvl = 4;

# try to be friendly to non-US-ASCII characters.  Still need to find what
# difference from RH<->Debian is still at fault.
print $q->header(-charset=>'utf8');

# Load a config ref containing DB host, name, user, and pass info based on
# from the server name + full script web path.  This allows us to host
# multiple instances without having to duplicate the code.
# This file is a Perl fragment to be processed inline.
my $cfgname = $ENV{SERVER_NAME}.$ENV{REQUEST_URI};
$cfgname =~ s|[./-]|_|g;
$cfgname =~ s|_dnsbl_cgi.*||;
$cfgname =~ s|_$||;

my $dnsbl = new DNSBL (configfile => "/etc/dnsbl/$cfgname.conf");
$dnsbl->connect;

my $page;
my $templatedir = $ENV{SCRIPT_FILENAME};
$templatedir =~ s/\w+\.cgi//;
$templatedir .= "templates";
$ENV{HTML_TEMPLATE_ROOT} = $templatedir;

my %config;
my $sth = $dbh->prepare("SELECT key,value FROM misc");
$sth->execute;
while (my ($key,$value) = $sth->fetchrow_array) {
  $config{$key} = $value;
}

# decide which page to spit out...
if (!$webvar{page}) {
  $page = HTML::Template->new(filename => "index.tmpl");
} else {
  $page = HTML::Template->new(filename => "$webvar{page}.tmpl");
}

$page->param(pgtitle => $config{pgtitle}) if defined($config{pgtitle});
$page->param(pgcomment => $config{pgcomment}) if defined($config{pgcomment});

if ($webvar{page} eq 'report') {
  my $dnsblsiteroot = $ENV{REQUEST_URI};
  $dnsblsiteroot =~ s|/dnsbl\.cgi\?.+|/|;
  $page->param(dnsblsiteroot => $dnsblsiteroot);

  ($webvar{ip}) = ($webvar{ip} =~ /(\d+\.\d+\.\d+\.\d+)/);
#  $webvar{ip} =~ s/^\s*//;
#  $webvar{ip} =~ s/\s*$//;
  $page->param(ip => $webvar{ip});

##fixme
# at some point this may need to be tweaked for Net::DNS's 1.x calling convention when they drop support for the older one
  #my @ptr = rr($webvar{ip});
  #$page->param(revinfo => join(',',@ptr));
  my $res = new Net::DNS::Resolver;
  $res->tcp_timeout(2);    # make me adjustable!
  $res->udp_timeout(2);    # make me adjustable!
  my $query = $res->query($webvar{ip}, "PTR");
  if ($query) {
    my @rdata;
    foreach my $rr ($query->answer) {
      my ($host,$ttl,$class,$type,$data) =
	($rr->string =~ /^([0-9a-zA-Z_.-]+)\s+(\d+)\s+([A-Za-z]+)\s+([A-Za-z]+)\s+(.+)$/s);
      push @rdata, $data;
    }
    $page->param(revinfo => join(', ',@rdata));
  } else {
    $page->param(revinfo => "DNS error: ".$res->errorstring);
  }

  my $ipinfo = $dnsbl->ipexists($webvar{ip});
  $page->param(nreports => $ipinfo->[0]) if $ipinfo;
  $page->param(ipexclude => $ipinfo->[1]) if $ipinfo;
  # extract and list the entire tree the IP is part of
  $page->param(browsebits =>
	DNSBLweb::retlvl($dbh, $dnsbl, 0, ip => $webvar{ip}, block => $dnsbl->getcontainer($webvar{ip},0) ));
  for (my $i=0; $i <= $maxlvl; $i++) {
    my ($block,$comment,$org) = $dnsbl->getcontainer($webvar{ip},$i);
    if ($block) {
      $page->param("comment$i" => $comment);
      my ($bcl,$bal,$bwl) = $dnsbl->islisted($block);
      $page->param("autob$i" => $bcl);
      $page->param("flag$i" => ($bwl ? 'exclude' : ($bal ? 'b1list' : '')) );
      $page->param("excl$i" => $bwl);
      my ($ol) = $dnsbl->islisted($org);
      $page->param("listorg$i" => $ol);
      $page->param("block$i" => $block);
      $page->param("org$i" => $org);
    }
  }
} elsif ($webvar{page} eq 'dbreport') {
  my $dnsblsiteroot = $ENV{REQUEST_URI};
  $dnsblsiteroot =~ s|/dnsbl\.cgi\?.+|/|;
  $page->param(dnsblsiteroot => $dnsblsiteroot);

  my $err = '';

  ($webvar{ip}) = ($webvar{ip} =~ /(\d+\.\d+\.\d+\.\d+)/);
#  $webvar{ip} =~ s/^\s*//;
#  $webvar{ip} =~ s/\s*$//;

  # basic algo:  for each listing level, add the org and block if not already present.
  # escape the loop if we check a level with no block entered.

  # there are still error checks that should probably be done.  changes in block
  # level/parenting should also Just Work(TM), rather than requiring setparents.pl
  # or setparents-full.pl
  for (my $i = 0; $i <= $maxlvl; $i++) {
    my $orgn = "org$i";
    my $blockn = "block$i";
    my $commentn = "comment$i";
    my $excln = "exclude$i";
    $webvar{$orgn} =~ s/^\s*//;
    $webvar{$orgn} =~ s/\s*$//;
    $webvar{$blockn} =~ s/^\s*//;
    $webvar{$blockn} =~ s/\s*$//;
    $webvar{$commentn} =~ s/^\s*//;
    $webvar{$commentn} =~ s/\s*$//;
    $webvar{$excln} =~ s/on/1/;
    my $orgid = $dnsbl->orgexists($webvar{$orgn});
    if (!$orgid) {
      $orgid = $dnsbl->addorg($webvar{$orgn});
      $page->param($orgn => $webvar{$orgn});
    }
    if ($webvar{$blockn} =~ /-/) {
      my $tmp = new NetAddr::IP $webvar{$blockn};
      if (!$tmp) {
        # Don't need to autofind ranges that are already CIDR-matched
        $err .= "Autofinding CIDR block containing $webvar{ip} for range '$webvar{$blockn}': ";
        my ($s,$f) = split /[\s-]+/, $webvar{$blockn};
        my $cidr = $dnsbl->range2cidr($s, $f, $webvar{ip});
        $err .= "$cidr<br>\n";
        $webvar{$blockn} = $cidr;
      }
    }
    if (!$dnsbl->blockexists($webvar{$blockn})) {
      my $ret = $dnsbl->addblock($webvar{$blockn}, $orgid, $i, $webvar{$excln}, $webvar{$commentn});
      $err .= "error adding $webvar{$blockn}: $ret<br>\n" if $ret;
      $page->param($blockn => $webvar{$blockn});
    } else {
      my $ret = $dnsbl->updateblock($webvar{$blockn}, $orgid, $i, $webvar{$excln}, $webvar{$commentn});
      $err .= "error updating $webvar{$blockn}: $ret<br>\n" if $ret;
    }
    last unless $webvar{"block".($i+1)};
  }

  my $count = $dnsbl->report($webvar{ip}, $webvar{excludeip});

  $page->param(ip => $webvar{ip});
  $page->param(err => $err);

  $page->param(browsebits => 
	DNSBLweb::retlvl($dbh, $dnsbl, 0, ip => $webvar{ip}, block => $dnsbl->getcontainer($webvar{ip},0) ));
}

print $page->output;
